/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.dbdiscovery.spi;

import org.apache.shardingsphere.infra.util.spi.type.typed.TypedSPI;

import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Optional;

/**
 * Database discovery provider.
 */
public interface DatabaseDiscoveryProvider extends TypedSPI {
    
    /**
     * Check environment.
     * 
     * @param databaseName database name
     * @param dataSources data sources
     */
    void checkEnvironment(String databaseName, Collection<DataSource> dataSources);
    
    /**
     * Judge whether database instance is primary.
     * 
     * @param dataSource data source to be judged
     * @return is primary database instance or not
     * @throws SQLException SQL exception
     */
    boolean isPrimaryInstance(DataSource dataSource) throws SQLException;
    
    /**
     * Load replica data source status.
     * 
     * @param replicaDataSource to be loaded replica data source
     * @return replica data source status
     * @throws SQLException SQL exception
     */
    ReplicaDataSourceStatus loadReplicaStatus(DataSource replicaDataSource) throws SQLException;
    
    /**
     * Get min enabled replicas.
     * 
     * @return min enabled replicas
     */
    Optional<Integer> getMinEnabledReplicas();
}
