/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLParameters;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.AbstractConfig;
import org.apache.kafka.common.config.SslConfigs;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.errors.InvalidConfigurationException;
import org.apache.kafka.common.memory.MemoryPool;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.network.CertStores;
import org.apache.kafka.common.network.ChannelBuilder;
import org.apache.kafka.common.network.ChannelBuilders;
import org.apache.kafka.common.network.ChannelMetadataRegistry;
import org.apache.kafka.common.network.ChannelState;
import org.apache.kafka.common.network.DefaultChannelMetadataRegistry;
import org.apache.kafka.common.network.KafkaChannel;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.common.network.ListenerReconfigurable;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.network.NetworkReceive;
import org.apache.kafka.common.network.NetworkSend;
import org.apache.kafka.common.network.NetworkTestUtils;
import org.apache.kafka.common.network.NioEchoServer;
import org.apache.kafka.common.network.PlaintextChannelBuilder;
import org.apache.kafka.common.network.Selector;
import org.apache.kafka.common.network.Send;
import org.apache.kafka.common.network.SslChannelBuilder;
import org.apache.kafka.common.network.SslTransportLayer;
import org.apache.kafka.common.security.TestSecurityConfig;
import org.apache.kafka.common.security.auth.SecurityProtocol;
import org.apache.kafka.common.security.ssl.DefaultSslEngineFactory;
import org.apache.kafka.common.security.ssl.SslFactory;
import org.apache.kafka.common.utils.Java;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.test.TestCondition;
import org.apache.kafka.test.TestSslUtils;
import org.apache.kafka.test.TestUtils;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class SslTransportLayerTest {
    private static final int BUFFER_SIZE = 4096;
    private static Time time = Time.SYSTEM;
    private final String tlsProtocol;
    private final boolean useInlinePem;
    private NioEchoServer server;
    private Selector selector;
    private CertStores serverCertStores;
    private CertStores clientCertStores;
    private Map<String, Object> sslClientConfigs;
    private Map<String, Object> sslServerConfigs;
    private Map<String, Object> sslConfigOverrides;

    @Parameterized.Parameters(name="tlsProtocol={0}, useInlinePem={1}")
    public static Collection<Object[]> data() {
        ArrayList<Object[]> values = new ArrayList<Object[]>();
        values.add(new Object[]{"TLSv1.2", false});
        values.add(new Object[]{"TLSv1.2", true});
        if (Java.IS_JAVA11_COMPATIBLE) {
            values.add(new Object[]{"TLSv1.3", false});
        }
        return values;
    }

    public SslTransportLayerTest(String tlsProtocol, boolean useInlinePem) {
        this.tlsProtocol = tlsProtocol;
        this.useInlinePem = useInlinePem;
        this.sslConfigOverrides = new HashMap<String, Object>();
        this.sslConfigOverrides.put("ssl.protocol", tlsProtocol);
        this.sslConfigOverrides.put("ssl.enabled.protocols", Collections.singletonList(tlsProtocol));
    }

    private CertStores.Builder certBuilder(boolean isServer, String cn) {
        return new CertStores.Builder(isServer).cn(cn).usePem(this.useInlinePem);
    }

    @Before
    public void setup() throws Exception {
        this.serverCertStores = this.certBuilder(true, "server").addHostName("localhost").build();
        this.clientCertStores = this.certBuilder(false, "client").addHostName("localhost").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.sslServerConfigs.put("ssl.engine.factory.class", DefaultSslEngineFactory.class);
        this.sslClientConfigs.put("ssl.engine.factory.class", DefaultSslEngineFactory.class);
        LogContext logContext = new LogContext();
        SslChannelBuilder channelBuilder = new SslChannelBuilder(Mode.CLIENT, null, false, logContext);
        channelBuilder.configure(this.sslClientConfigs);
        this.selector = new Selector(5000L, new Metrics(), time, "MetricGroup", (ChannelBuilder)channelBuilder, logContext);
    }

    @After
    public void teardown() throws Exception {
        if (this.selector != null) {
            this.selector.close();
        }
        if (this.server != null) {
            this.server.close();
        }
    }

    @Test
    public void testValidEndpointIdentificationSanDns() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
        this.server.verifyAuthenticationMetrics(1, 0);
    }

    @Test
    public void testValidEndpointIdentificationSanIp() throws Exception {
        String node = "0";
        this.serverCertStores = this.certBuilder(true, "server").hostAddress(InetAddress.getByName("127.0.0.1")).build();
        this.clientCertStores = this.certBuilder(false, "client").hostAddress(InetAddress.getByName("127.0.0.1")).build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("127.0.0.1", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testValidEndpointIdentificationCN() throws Exception {
        this.serverCertStores = this.certBuilder(true, "localhost").build();
        this.clientCertStores = this.certBuilder(false, "localhost").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.verifySslConfigs();
    }

    @Test
    public void testEndpointIdentificationNoReverseLookup() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("127.0.0.1", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node, ChannelState.State.AUTHENTICATION_FAILED);
    }

    @Test
    public void testClientEndpointNotValidated() throws Exception {
        String node = "0";
        this.clientCertStores = this.certBuilder(false, "non-existent.com").build();
        this.serverCertStores = this.certBuilder(true, "localhost").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        TestSslChannelBuilder serverChannelBuilder = new TestSslChannelBuilder(Mode.SERVER){

            @Override
            protected TestSslChannelBuilder.TestSslTransportLayer newTransportLayer(String id, SelectionKey key, SSLEngine sslEngine) throws IOException {
                SSLParameters sslParams = sslEngine.getSSLParameters();
                sslParams.setEndpointIdentificationAlgorithm("HTTPS");
                sslEngine.setSSLParameters(sslParams);
                return super.newTransportLayer(id, key, sslEngine);
            }
        };
        serverChannelBuilder.configure(this.sslServerConfigs);
        this.server = new NioEchoServer(ListenerName.forSecurityProtocol((SecurityProtocol)SecurityProtocol.SSL), SecurityProtocol.SSL, new TestSecurityConfig(this.sslServerConfigs), "localhost", (ChannelBuilder)serverChannelBuilder, null, time);
        this.server.start();
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testInvalidEndpointIdentification() throws Exception {
        this.serverCertStores = this.certBuilder(true, "server").addHostName("notahost").build();
        this.clientCertStores = this.certBuilder(false, "client").addHostName("localhost").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.verifySslConfigsWithHandshakeFailure();
    }

    @Test
    public void testEndpointIdentificationDisabled() throws Exception {
        this.serverCertStores = this.certBuilder(true, "server").addHostName("notahost").build();
        this.clientCertStores = this.certBuilder(false, "client").addHostName("localhost").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        String node = "1";
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "");
        this.createSelector(this.sslClientConfigs);
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
        String node2 = "2";
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", null);
        this.createSelector(this.sslClientConfigs);
        this.selector.connect(node2, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node2, 100, 10);
        String node3 = "3";
        this.sslClientConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.createSelector(this.sslClientConfigs);
        this.selector.connect(node3, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node3, ChannelState.State.AUTHENTICATION_FAILED);
        this.selector.close();
    }

    @Test
    public void testClientAuthenticationRequiredValidProvided() throws Exception {
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.verifySslConfigs();
    }

    @Test
    public void testListenerConfigOverride() throws Exception {
        String node = "0";
        ListenerName clientListenerName = new ListenerName("client");
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.sslServerConfigs.put(clientListenerName.configPrefix() + "ssl.client.auth", "none");
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.createSelector(this.sslClientConfigs);
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
        this.selector.close();
        CertStores.KEYSTORE_PROPS.forEach(this.sslClientConfigs::remove);
        this.createSelector(this.sslClientConfigs);
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node, ChannelState.State.AUTHENTICATION_FAILED);
        this.selector.close();
        this.server.close();
        this.server = this.createEchoServer(clientListenerName, SecurityProtocol.SSL);
        addr = new InetSocketAddress("localhost", this.server.port());
        this.createSelector(this.sslClientConfigs);
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    @Test
    public void testClientAuthenticationRequiredUntrustedProvided() throws Exception {
        this.sslServerConfigs = this.serverCertStores.getUntrustingConfig();
        this.sslServerConfigs.putAll(this.sslConfigOverrides);
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.verifySslConfigsWithHandshakeFailure();
    }

    @Test
    public void testClientAuthenticationRequiredNotProvided() throws Exception {
        this.sslServerConfigs.put("ssl.client.auth", "required");
        CertStores.KEYSTORE_PROPS.forEach(this.sslClientConfigs::remove);
        this.verifySslConfigsWithHandshakeFailure();
    }

    @Test
    public void testClientAuthenticationDisabledUntrustedProvided() throws Exception {
        this.sslServerConfigs = this.serverCertStores.getUntrustingConfig();
        this.sslServerConfigs.putAll(this.sslConfigOverrides);
        this.sslServerConfigs.put("ssl.client.auth", "none");
        this.verifySslConfigs();
    }

    @Test
    public void testClientAuthenticationDisabledNotProvided() throws Exception {
        this.sslServerConfigs.put("ssl.client.auth", "none");
        CertStores.KEYSTORE_PROPS.forEach(this.sslClientConfigs::remove);
        this.verifySslConfigs();
    }

    @Test
    public void testClientAuthenticationRequestedValidProvided() throws Exception {
        this.sslServerConfigs.put("ssl.client.auth", "requested");
        this.verifySslConfigs();
    }

    @Test
    public void testClientAuthenticationRequestedNotProvided() throws Exception {
        this.sslServerConfigs.put("ssl.client.auth", "requested");
        CertStores.KEYSTORE_PROPS.forEach(this.sslClientConfigs::remove);
        this.verifySslConfigs();
    }

    @Test
    public void testDsaKeyPair() throws Exception {
        Assume.assumeTrue((boolean)this.tlsProtocol.equals("TLSv1.2"));
        this.serverCertStores = this.certBuilder(true, "server").keyAlgorithm("DSA").build();
        this.clientCertStores = this.certBuilder(false, "client").keyAlgorithm("DSA").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.verifySslConfigs();
    }

    @Test
    public void testECKeyPair() throws Exception {
        this.serverCertStores = this.certBuilder(true, "server").keyAlgorithm("EC").build();
        this.clientCertStores = this.certBuilder(false, "client").keyAlgorithm("EC").build();
        this.sslServerConfigs = this.getTrustingConfig(this.serverCertStores, this.clientCertStores);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, this.serverCertStores);
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.verifySslConfigs();
    }

    @Test
    public void testPemFiles() throws Exception {
        TestSslUtils.convertToPem(this.sslServerConfigs, true, true);
        TestSslUtils.convertToPem(this.sslClientConfigs, true, true);
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.verifySslConfigs();
    }

    @Test
    public void testPemFilesWithoutClientKeyPassword() throws Exception {
        TestSslUtils.convertToPem(this.sslServerConfigs, !this.useInlinePem, true);
        TestSslUtils.convertToPem(this.sslClientConfigs, !this.useInlinePem, false);
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        if (this.useInlinePem) {
            this.verifySslConfigs();
        } else {
            Assert.assertThrows(KafkaException.class, () -> this.createSelector(this.sslClientConfigs));
        }
    }

    @Test
    public void testPemFilesWithoutServerKeyPassword() throws Exception {
        TestSslUtils.convertToPem(this.sslServerConfigs, !this.useInlinePem, false);
        TestSslUtils.convertToPem(this.sslClientConfigs, !this.useInlinePem, true);
        if (this.useInlinePem) {
            this.verifySslConfigs();
        } else {
            Assert.assertThrows(KafkaException.class, () -> this.createEchoServer(SecurityProtocol.SSL));
        }
    }

    @Test
    public void testInvalidSecureRandomImplementation() {
        try (SslChannelBuilder channelBuilder = this.newClientChannelBuilder();){
            this.sslClientConfigs.put("ssl.secure.random.implementation", "invalid");
            channelBuilder.configure(this.sslClientConfigs);
            Assert.fail((String)"SSL channel configured with invalid SecureRandom implementation");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    @Test
    public void testInvalidTruststorePassword() {
        try (SslChannelBuilder channelBuilder = this.newClientChannelBuilder();){
            this.sslClientConfigs.put("ssl.truststore.password", "invalid");
            channelBuilder.configure(this.sslClientConfigs);
            Assert.fail((String)"SSL channel configured with invalid truststore password");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    @Test
    public void testInvalidKeystorePassword() {
        try (SslChannelBuilder channelBuilder = this.newClientChannelBuilder();){
            this.sslClientConfigs.put("ssl.keystore.password", "invalid");
            channelBuilder.configure(this.sslClientConfigs);
            Assert.fail((String)"SSL channel configured with invalid keystore password");
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    @Test
    public void testNullTruststorePassword() throws Exception {
        this.sslClientConfigs.remove("ssl.truststore.password");
        this.sslServerConfigs.remove("ssl.truststore.password");
        this.verifySslConfigs();
    }

    @Test
    public void testInvalidKeyPassword() throws Exception {
        this.sslServerConfigs.put("ssl.key.password", new Password("invalid"));
        if (this.useInlinePem) {
            Assert.assertThrows(InvalidConfigurationException.class, () -> this.createEchoServer(SecurityProtocol.SSL));
            return;
        }
        this.verifySslConfigsWithHandshakeFailure();
    }

    @Test
    public void testTlsDefaults() throws Exception {
        this.sslServerConfigs = this.serverCertStores.getTrustingConfig(this.clientCertStores);
        this.sslClientConfigs = this.clientCertStores.getTrustingConfig(this.serverCertStores);
        Assert.assertEquals((Object)SslConfigs.DEFAULT_SSL_PROTOCOL, (Object)this.sslServerConfigs.get("ssl.protocol"));
        Assert.assertEquals((Object)SslConfigs.DEFAULT_SSL_PROTOCOL, (Object)this.sslClientConfigs.get("ssl.protocol"));
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect("0", addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, "0", 10, 100);
        this.server.verifyAuthenticationMetrics(1, 0);
        this.selector.close();
    }

    private void checkAuthenticationFailed(String node, String tlsVersion) throws IOException {
        this.sslClientConfigs.put("ssl.enabled.protocols", Arrays.asList(tlsVersion));
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node, ChannelState.State.AUTHENTICATION_FAILED);
        this.selector.close();
    }

    @Test
    public void testUnsupportedCiphers() throws Exception {
        SSLContext context = SSLContext.getInstance(this.tlsProtocol);
        context.init(null, null, null);
        String[] cipherSuites = context.getDefaultSSLParameters().getCipherSuites();
        this.sslServerConfigs.put("ssl.cipher.suites", Arrays.asList(cipherSuites[0]));
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.sslClientConfigs.put("ssl.cipher.suites", Arrays.asList(cipherSuites[1]));
        this.createSelector(this.sslClientConfigs);
        this.checkAuthenticationFailed("1", this.tlsProtocol);
        this.server.verifyAuthenticationMetrics(0, 1);
    }

    @Test
    public void testServerRequestMetrics() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs, 16384, 16384, 16384);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 102400, 102400);
        NetworkTestUtils.waitForChannelReady(this.selector, node);
        int messageSize = 0x100000;
        String message = TestUtils.randomString(messageSize);
        this.selector.send((Send)new NetworkSend(node, ByteBuffer.wrap(message.getBytes())));
        while (this.selector.completedReceives().isEmpty()) {
            this.selector.poll(100L);
        }
        int totalBytes = messageSize + 4;
        this.server.waitForMetric("incoming-byte", totalBytes);
        this.server.waitForMetric("outgoing-byte", totalBytes);
        this.server.waitForMetric("request", 1.0);
        this.server.waitForMetric("response", 1.0);
    }

    @Test
    public void testSelectorPollReadSize() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs, 16384, 16384, 16384);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 102400, 102400);
        NetworkTestUtils.checkClientConnection(this.selector, node, 81920, 1);
        String message = TestUtils.randomString(81920);
        this.selector.send((Send)new NetworkSend(node, ByteBuffer.wrap(message.getBytes())));
        TestUtils.waitForCondition(new TestCondition(){

            @Override
            public boolean conditionMet() {
                try {
                    SslTransportLayerTest.this.selector.poll(100L);
                }
                catch (IOException e) {
                    return false;
                }
                return SslTransportLayerTest.this.selector.completedSends().size() > 0;
            }
        }, "Timed out waiting for message to be sent");
        TestUtils.waitForCondition(new TestCondition(){

            @Override
            public boolean conditionMet() {
                return SslTransportLayerTest.this.server.numSent() >= 2;
            }
        }, "Timed out waiting for echo server to send message");
        this.selector.poll(1000L);
        Collection receiveList = this.selector.completedReceives();
        Assert.assertEquals((long)1L, (long)receiveList.size());
        Assert.assertEquals((Object)message, (Object)new String(Utils.toArray((ByteBuffer)((NetworkReceive)receiveList.iterator().next()).payload())));
    }

    @Test
    public void testNetReadBufferResize() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs, 10, null, null);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 64000, 10);
    }

    @Test
    public void testNetWriteBufferResize() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs, null, 10, null);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 64000, 10);
    }

    @Test
    public void testApplicationBufferResize() throws Exception {
        String node = "0";
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs, null, null, 10);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 64000, 10);
    }

    @Test
    public void testNetworkThreadTimeRecorded() throws Exception {
        LogContext logContext = new LogContext();
        SslChannelBuilder channelBuilder = new SslChannelBuilder(Mode.CLIENT, null, false, logContext);
        channelBuilder.configure(this.sslClientConfigs);
        try (final Selector selector = new Selector(-1, -1L, new Metrics(), Time.SYSTEM, "MetricGroup", new HashMap(), false, true, (ChannelBuilder)channelBuilder, MemoryPool.NONE, logContext);){
            String node = "0";
            this.server = this.createEchoServer(SecurityProtocol.SSL);
            InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
            selector.connect(node, addr, 4096, 4096);
            String message = TestUtils.randomString(0x100000);
            NetworkTestUtils.waitForChannelReady(selector, node);
            KafkaChannel channel = selector.channel(node);
            Assert.assertTrue((String)"SSL handshake time not recorded", (channel.getAndResetNetworkThreadTimeNanos() > 0L ? 1 : 0) != 0);
            Assert.assertEquals((String)"Time not reset", (long)0L, (long)channel.getAndResetNetworkThreadTimeNanos());
            selector.mute(node);
            selector.send((Send)new NetworkSend(node, ByteBuffer.wrap(message.getBytes())));
            while (selector.completedSends().isEmpty()) {
                selector.poll(100L);
            }
            long sendTimeNanos = channel.getAndResetNetworkThreadTimeNanos();
            Assert.assertTrue((String)("Send time not recorded: " + sendTimeNanos), (sendTimeNanos > 0L ? 1 : 0) != 0);
            Assert.assertEquals((String)"Time not reset", (long)0L, (long)channel.getAndResetNetworkThreadTimeNanos());
            Assert.assertFalse((String)"Unexpected bytes buffered", (boolean)channel.hasBytesBuffered());
            Assert.assertEquals((long)0L, (long)selector.completedReceives().size());
            selector.unmute(node);
            TestUtils.waitForCondition(new TestCondition(){

                @Override
                public boolean conditionMet() {
                    try {
                        selector.poll(100L);
                    }
                    catch (IOException e) {
                        return false;
                    }
                    return !selector.completedReceives().isEmpty();
                }
            }, "Timed out waiting for a message to receive from echo server");
            long receiveTimeNanos = channel.getAndResetNetworkThreadTimeNanos();
            Assert.assertTrue((String)("Receive time not recorded: " + receiveTimeNanos), (receiveTimeNanos > 0L ? 1 : 0) != 0);
        }
    }

    @Test
    public void testIOExceptionsDuringHandshakeRead() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.testIOExceptionsDuringHandshake(FailureAction.THROW_IO_EXCEPTION, FailureAction.NO_OP);
    }

    @Test
    public void testIOExceptionsDuringHandshakeWrite() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.testIOExceptionsDuringHandshake(FailureAction.NO_OP, FailureAction.THROW_IO_EXCEPTION);
    }

    @Test
    public void testUngracefulRemoteCloseDuringHandshakeRead() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.testIOExceptionsDuringHandshake(this.server::closeSocketChannels, FailureAction.NO_OP);
    }

    @Test
    public void testUngracefulRemoteCloseDuringHandshakeWrite() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.testIOExceptionsDuringHandshake(FailureAction.NO_OP, this.server::closeSocketChannels);
    }

    @Test
    public void testGracefulRemoteCloseDuringHandshakeRead() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.testIOExceptionsDuringHandshake(FailureAction.NO_OP, this.server::closeKafkaChannels);
    }

    @Test
    public void testGracefulRemoteCloseDuringHandshakeWrite() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.testIOExceptionsDuringHandshake(this.server::closeKafkaChannels, FailureAction.NO_OP);
    }

    private void testIOExceptionsDuringHandshake(FailureAction readFailureAction, FailureAction flushFailureAction) throws Exception {
        TestSslChannelBuilder channelBuilder = new TestSslChannelBuilder(Mode.CLIENT);
        boolean done = false;
        for (int i = 1; i <= 100; ++i) {
            KafkaChannel channel;
            String node = String.valueOf(i);
            channelBuilder.readFailureAction = readFailureAction;
            channelBuilder.flushFailureAction = flushFailureAction;
            channelBuilder.failureIndex = i;
            channelBuilder.configure(this.sslClientConfigs);
            this.selector = new Selector(5000L, new Metrics(), time, "MetricGroup", (ChannelBuilder)channelBuilder, new LogContext());
            InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
            this.selector.connect(node, addr, 4096, 4096);
            for (int j = 0; j < 30; ++j) {
                this.selector.poll(1000L);
                KafkaChannel channel2 = this.selector.channel(node);
                if (channel2 != null && channel2.ready()) {
                    done = true;
                    break;
                }
                if (!this.selector.disconnected().containsKey(node)) continue;
                ChannelState.State state = ((ChannelState)this.selector.disconnected().get(node)).state();
                Assert.assertTrue((String)("Unexpected channel state " + state), (state == ChannelState.State.AUTHENTICATE || state == ChannelState.State.READY ? 1 : 0) != 0);
                break;
            }
            if ((channel = this.selector.channel(node)) != null) {
                Assert.assertTrue((String)("Channel not ready or disconnected:" + channel.state().state()), (boolean)channel.ready());
            }
            this.selector.close();
        }
        Assert.assertTrue((String)"Too many invocations of read/write during SslTransportLayer.handshake()", (boolean)done);
    }

    @Test
    public void testPeerNotifiedOfHandshakeFailure() throws Exception {
        this.sslServerConfigs = this.serverCertStores.getUntrustingConfig();
        this.sslServerConfigs.putAll(this.sslConfigOverrides);
        this.sslServerConfigs.put("ssl.client.auth", "required");
        int i = 0;
        while (i < 3) {
            String node = "0";
            TestSslChannelBuilder serverChannelBuilder = new TestSslChannelBuilder(Mode.SERVER);
            serverChannelBuilder.configure(this.sslServerConfigs);
            serverChannelBuilder.flushDelayCount = i++;
            this.server = new NioEchoServer(ListenerName.forSecurityProtocol((SecurityProtocol)SecurityProtocol.SSL), SecurityProtocol.SSL, new TestSecurityConfig(this.sslServerConfigs), "localhost", (ChannelBuilder)serverChannelBuilder, null, time);
            this.server.start();
            this.createSelector(this.sslClientConfigs);
            InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
            this.selector.connect(node, addr, 4096, 4096);
            NetworkTestUtils.waitForChannelClose(this.selector, node, ChannelState.State.AUTHENTICATION_FAILED);
            this.server.close();
            this.selector.close();
            serverChannelBuilder.close();
        }
    }

    @Test
    public void testCloseSsl() throws Exception {
        this.testClose(SecurityProtocol.SSL, (ChannelBuilder)this.newClientChannelBuilder());
    }

    @Test
    public void testClosePlaintext() throws Exception {
        this.testClose(SecurityProtocol.PLAINTEXT, (ChannelBuilder)new PlaintextChannelBuilder(null));
    }

    private SslChannelBuilder newClientChannelBuilder() {
        return new SslChannelBuilder(Mode.CLIENT, null, false, new LogContext());
    }

    private void testClose(SecurityProtocol securityProtocol, ChannelBuilder clientChannelBuilder) throws Exception {
        String node = "0";
        this.server = this.createEchoServer(securityProtocol);
        clientChannelBuilder.configure(this.sslClientConfigs);
        this.selector = new Selector(5000L, new Metrics(), time, "MetricGroup", clientChannelBuilder, new LogContext());
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelReady(this.selector, node);
        TestUtils.waitForCondition(() -> this.server.selector().channels().stream().allMatch(KafkaChannel::ready), "Channel not ready");
        final ByteArrayOutputStream bytesOut = new ByteArrayOutputStream();
        this.server.outputChannel(Channels.newChannel(bytesOut));
        this.server.selector().muteAll();
        byte[] message = TestUtils.randomString(100).getBytes();
        int count = 20;
        final int totalSendSize = count * (message.length + 4);
        for (int i = 0; i < count; ++i) {
            this.selector.send((Send)new NetworkSend(node, ByteBuffer.wrap(message)));
            do {
                this.selector.poll(0L);
            } while (this.selector.completedSends().isEmpty());
        }
        this.server.selector().unmuteAll();
        this.selector.close(node);
        TestUtils.waitForCondition(new TestCondition(){

            @Override
            public boolean conditionMet() {
                return bytesOut.toByteArray().length == totalSendSize;
            }
        }, 5000L, "All requests sent were not processed");
    }

    @Test
    public void testInterBrokerSslConfigValidation() throws Exception {
        SecurityProtocol securityProtocol = SecurityProtocol.SSL;
        this.sslServerConfigs.put("ssl.client.auth", "required");
        this.sslServerConfigs.put("ssl.endpoint.identification.algorithm", "HTTPS");
        this.sslServerConfigs.putAll(this.serverCertStores.keyStoreProps());
        this.sslServerConfigs.putAll(this.serverCertStores.trustStoreProps());
        this.sslClientConfigs.putAll(this.serverCertStores.keyStoreProps());
        this.sslClientConfigs.putAll(this.serverCertStores.trustStoreProps());
        TestSecurityConfig config = new TestSecurityConfig(this.sslServerConfigs);
        ListenerName listenerName = ListenerName.forSecurityProtocol((SecurityProtocol)securityProtocol);
        ChannelBuilder serverChannelBuilder = ChannelBuilders.serverChannelBuilder((ListenerName)listenerName, (boolean)true, (SecurityProtocol)securityProtocol, (AbstractConfig)config, null, null, (Time)time, (LogContext)new LogContext());
        this.server = new NioEchoServer(listenerName, securityProtocol, config, "localhost", serverChannelBuilder, null, time);
        this.server.start();
        this.selector = this.createSelector(this.sslClientConfigs, null, null, null);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        this.selector.connect("0", addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, "0", 100, 10);
    }

    @Test(expected=KafkaException.class)
    public void testInterBrokerSslConfigValidationFailure() {
        SecurityProtocol securityProtocol = SecurityProtocol.SSL;
        this.sslServerConfigs.put("ssl.client.auth", "required");
        TestSecurityConfig config = new TestSecurityConfig(this.sslServerConfigs);
        ListenerName listenerName = ListenerName.forSecurityProtocol((SecurityProtocol)securityProtocol);
        ChannelBuilders.serverChannelBuilder((ListenerName)listenerName, (boolean)true, (SecurityProtocol)securityProtocol, (AbstractConfig)config, null, null, (Time)time, (LogContext)new LogContext());
    }

    @Test
    public void testServerKeystoreDynamicUpdate() throws Exception {
        SecurityProtocol securityProtocol = SecurityProtocol.SSL;
        TestSecurityConfig config = new TestSecurityConfig(this.sslServerConfigs);
        ListenerName listenerName = ListenerName.forSecurityProtocol((SecurityProtocol)securityProtocol);
        ChannelBuilder serverChannelBuilder = ChannelBuilders.serverChannelBuilder((ListenerName)listenerName, (boolean)false, (SecurityProtocol)securityProtocol, (AbstractConfig)config, null, null, (Time)time, (LogContext)new LogContext());
        this.server = new NioEchoServer(listenerName, securityProtocol, config, "localhost", serverChannelBuilder, null, time);
        this.server.start();
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        String oldNode = "0";
        Selector oldClientSelector = this.createSelector(this.sslClientConfigs);
        oldClientSelector.connect(oldNode, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, oldNode, 100, 10);
        CertStores newServerCertStores = this.certBuilder(true, "server").addHostName("localhost").build();
        Map<String, Object> newKeystoreConfigs = newServerCertStores.keyStoreProps();
        Assert.assertTrue((String)"SslChannelBuilder not reconfigurable", (boolean)(serverChannelBuilder instanceof ListenerReconfigurable));
        ListenerReconfigurable reconfigurableBuilder = (ListenerReconfigurable)serverChannelBuilder;
        Assert.assertEquals((Object)listenerName, (Object)reconfigurableBuilder.listenerName());
        reconfigurableBuilder.validateReconfiguration(newKeystoreConfigs);
        reconfigurableBuilder.reconfigure(newKeystoreConfigs);
        oldClientSelector.connect("1", addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(oldClientSelector, "1", ChannelState.State.AUTHENTICATION_FAILED);
        this.sslClientConfigs = this.getTrustingConfig(this.clientCertStores, newServerCertStores);
        Selector newClientSelector = this.createSelector(this.sslClientConfigs);
        newClientSelector.connect("2", addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(newClientSelector, "2", 100, 10);
        NetworkTestUtils.checkClientConnection(oldClientSelector, oldNode, 100, 10);
        CertStores invalidCertStores = this.certBuilder(true, "server").addHostName("127.0.0.1").build();
        Map<String, Object> invalidConfigs = this.getTrustingConfig(invalidCertStores, this.clientCertStores);
        this.verifyInvalidReconfigure(reconfigurableBuilder, invalidConfigs, "keystore with different SubjectAltName");
        HashMap<String, Object> missingStoreConfigs = new HashMap<String, Object>();
        missingStoreConfigs.put("ssl.keystore.type", "PKCS12");
        missingStoreConfigs.put("ssl.keystore.location", "some.keystore.path");
        missingStoreConfigs.put("ssl.keystore.password", new Password("some.keystore.password"));
        missingStoreConfigs.put("ssl.key.password", new Password("some.key.password"));
        this.verifyInvalidReconfigure(reconfigurableBuilder, missingStoreConfigs, "keystore not found");
        newClientSelector.connect("3", addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(newClientSelector, "3", 100, 10);
    }

    @Test
    public void testServerKeystoreDynamicUpdateWithNewSubjectAltName() throws Exception {
        SecurityProtocol securityProtocol = SecurityProtocol.SSL;
        TestSecurityConfig config = new TestSecurityConfig(this.sslServerConfigs);
        ListenerName listenerName = ListenerName.forSecurityProtocol((SecurityProtocol)securityProtocol);
        ChannelBuilder serverChannelBuilder = ChannelBuilders.serverChannelBuilder((ListenerName)listenerName, (boolean)false, (SecurityProtocol)securityProtocol, (AbstractConfig)config, null, null, (Time)time, (LogContext)new LogContext());
        this.server = new NioEchoServer(listenerName, securityProtocol, config, "localhost", serverChannelBuilder, null, time);
        this.server.start();
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        Selector selector = this.createSelector(this.sslClientConfigs);
        String node1 = "1";
        selector.connect(node1, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(selector, node1, 100, 10);
        selector.close();
        TestSslUtils.CertificateBuilder certBuilder = new TestSslUtils.CertificateBuilder().sanDnsNames("localhost", "*.example.com");
        String truststorePath = (String)this.sslClientConfigs.get("ssl.truststore.location");
        File truststoreFile = truststorePath != null ? new File(truststorePath) : null;
        TestSslUtils.SslConfigsBuilder builder = new TestSslUtils.SslConfigsBuilder(Mode.SERVER).useClientCert(false).certAlias("server").cn("server").certBuilder(certBuilder).createNewTrustStore(truststoreFile).usePem(this.useInlinePem);
        Map<String, Object> newConfigs = builder.build();
        HashMap<String, Object> newKeystoreConfigs = new HashMap<String, Object>();
        for (String string : CertStores.KEYSTORE_PROPS) {
            newKeystoreConfigs.put(string, newConfigs.get(string));
        }
        ListenerReconfigurable reconfigurableBuilder = (ListenerReconfigurable)serverChannelBuilder;
        reconfigurableBuilder.validateReconfiguration(newKeystoreConfigs);
        reconfigurableBuilder.reconfigure(newKeystoreConfigs);
        for (String propName : CertStores.TRUSTSTORE_PROPS) {
            this.sslClientConfigs.put(propName, newConfigs.get(propName));
        }
        selector = this.createSelector(this.sslClientConfigs);
        String string = "2";
        selector.connect(string, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(selector, string, 100, 10);
        TestSslUtils.CertificateBuilder invalidBuilder = new TestSslUtils.CertificateBuilder().sanDnsNames("localhost");
        if (!this.useInlinePem) {
            builder.useExistingTrustStore(truststoreFile);
        }
        Map<String, Object> invalidConfig = builder.certBuilder(invalidBuilder).build();
        HashMap<String, Object> invalidKeystoreConfigs = new HashMap<String, Object>();
        for (String propName : CertStores.KEYSTORE_PROPS) {
            invalidKeystoreConfigs.put(propName, invalidConfig.get(propName));
        }
        this.verifyInvalidReconfigure(reconfigurableBuilder, invalidKeystoreConfigs, "keystore without existing SubjectAltName");
        String node3 = "3";
        selector.connect(node3, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(selector, node3, 100, 10);
    }

    @Test
    public void testServerTruststoreDynamicUpdate() throws Exception {
        SecurityProtocol securityProtocol = SecurityProtocol.SSL;
        this.sslServerConfigs.put("ssl.client.auth", "required");
        TestSecurityConfig config = new TestSecurityConfig(this.sslServerConfigs);
        ListenerName listenerName = ListenerName.forSecurityProtocol((SecurityProtocol)securityProtocol);
        ChannelBuilder serverChannelBuilder = ChannelBuilders.serverChannelBuilder((ListenerName)listenerName, (boolean)false, (SecurityProtocol)securityProtocol, (AbstractConfig)config, null, null, (Time)time, (LogContext)new LogContext());
        this.server = new NioEchoServer(listenerName, securityProtocol, config, "localhost", serverChannelBuilder, null, time);
        this.server.start();
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        String oldNode = "0";
        Selector oldClientSelector = this.createSelector(this.sslClientConfigs);
        oldClientSelector.connect(oldNode, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, oldNode, 100, 10);
        CertStores newClientCertStores = this.certBuilder(true, "client").addHostName("localhost").build();
        this.sslClientConfigs = this.getTrustingConfig(newClientCertStores, this.serverCertStores);
        Map<String, Object> newTruststoreConfigs = newClientCertStores.trustStoreProps();
        Assert.assertTrue((String)"SslChannelBuilder not reconfigurable", (boolean)(serverChannelBuilder instanceof ListenerReconfigurable));
        ListenerReconfigurable reconfigurableBuilder = (ListenerReconfigurable)serverChannelBuilder;
        Assert.assertEquals((Object)listenerName, (Object)reconfigurableBuilder.listenerName());
        reconfigurableBuilder.validateReconfiguration(newTruststoreConfigs);
        reconfigurableBuilder.reconfigure(newTruststoreConfigs);
        oldClientSelector.connect("1", addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(oldClientSelector, "1", ChannelState.State.AUTHENTICATION_FAILED);
        Selector newClientSelector = this.createSelector(this.sslClientConfigs);
        newClientSelector.connect("2", addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(newClientSelector, "2", 100, 10);
        NetworkTestUtils.checkClientConnection(oldClientSelector, oldNode, 100, 10);
        HashMap<String, Object> invalidConfigs = new HashMap<String, Object>(newTruststoreConfigs);
        invalidConfigs.put("ssl.truststore.type", "INVALID_TYPE");
        this.verifyInvalidReconfigure(reconfigurableBuilder, invalidConfigs, "invalid truststore type");
        HashMap<String, Object> missingStoreConfigs = new HashMap<String, Object>();
        missingStoreConfigs.put("ssl.truststore.type", "PKCS12");
        missingStoreConfigs.put("ssl.truststore.location", "some.truststore.path");
        missingStoreConfigs.put("ssl.truststore.password", new Password("some.truststore.password"));
        this.verifyInvalidReconfigure(reconfigurableBuilder, missingStoreConfigs, "truststore not found");
        newClientSelector.connect("3", addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(newClientSelector, "3", 100, 10);
    }

    @Test
    public void testCustomClientSslEngineFactory() throws Exception {
        this.sslClientConfigs.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        this.verifySslConfigs();
    }

    @Test
    public void testCustomServerSslEngineFactory() throws Exception {
        this.sslServerConfigs.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        this.verifySslConfigs();
    }

    @Test
    public void testCustomClientAndServerSslEngineFactory() throws Exception {
        this.sslClientConfigs.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        this.sslServerConfigs.put("ssl.engine.factory.class", TestSslUtils.TestSslEngineFactory.class);
        this.verifySslConfigs();
    }

    @Test(expected=KafkaException.class)
    public void testInvalidSslEngineFactory() {
        this.sslClientConfigs.put("ssl.engine.factory.class", String.class);
        this.createSelector(this.sslClientConfigs);
    }

    private void verifyInvalidReconfigure(ListenerReconfigurable reconfigurable, Map<String, Object> invalidConfigs, String errorMessage) {
        try {
            reconfigurable.validateReconfiguration(invalidConfigs);
            Assert.fail((String)("Should have failed validation with an exception: " + errorMessage));
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
        try {
            reconfigurable.reconfigure(invalidConfigs);
            Assert.fail((String)("Should have failed to reconfigure: " + errorMessage));
        }
        catch (KafkaException kafkaException) {
            // empty catch block
        }
    }

    private Selector createSelector(Map<String, Object> sslClientConfigs) {
        return this.createSelector(sslClientConfigs, null, null, null);
    }

    private Selector createSelector(Map<String, Object> sslClientConfigs, Integer netReadBufSize, Integer netWriteBufSize, Integer appBufSize) {
        TestSslChannelBuilder channelBuilder = new TestSslChannelBuilder(Mode.CLIENT);
        channelBuilder.configureBufferSizes(netReadBufSize, netWriteBufSize, appBufSize);
        channelBuilder.configure(sslClientConfigs);
        this.selector = new Selector(500000L, new Metrics(), time, "MetricGroup", (ChannelBuilder)channelBuilder, new LogContext());
        return this.selector;
    }

    private NioEchoServer createEchoServer(ListenerName listenerName, SecurityProtocol securityProtocol) throws Exception {
        return NetworkTestUtils.createEchoServer(listenerName, securityProtocol, new TestSecurityConfig(this.sslServerConfigs), null, time);
    }

    private NioEchoServer createEchoServer(SecurityProtocol securityProtocol) throws Exception {
        return this.createEchoServer(ListenerName.forSecurityProtocol((SecurityProtocol)securityProtocol), securityProtocol);
    }

    private Map<String, Object> getTrustingConfig(CertStores certStores, CertStores peerCertStores) {
        Map<String, Object> configs = certStores.getTrustingConfig(peerCertStores);
        configs.putAll(this.sslConfigOverrides);
        return configs;
    }

    private void verifySslConfigs() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        String node = "0";
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.checkClientConnection(this.selector, node, 100, 10);
    }

    public void verifySslConfigsWithHandshakeFailure() throws Exception {
        this.server = this.createEchoServer(SecurityProtocol.SSL);
        this.createSelector(this.sslClientConfigs);
        InetSocketAddress addr = new InetSocketAddress("localhost", this.server.port());
        String node = "0";
        this.selector.connect(node, addr, 4096, 4096);
        NetworkTestUtils.waitForChannelClose(this.selector, node, ChannelState.State.AUTHENTICATION_FAILED);
        this.server.verifyAuthenticationMetrics(0, 1);
    }

    static class TestSslChannelBuilder
    extends SslChannelBuilder {
        private Integer netReadBufSizeOverride;
        private Integer netWriteBufSizeOverride;
        private Integer appBufSizeOverride;
        private long failureIndex = Long.MAX_VALUE;
        FailureAction readFailureAction = FailureAction.NO_OP;
        FailureAction flushFailureAction = FailureAction.NO_OP;
        int flushDelayCount = 0;

        public TestSslChannelBuilder(Mode mode) {
            super(mode, null, false, new LogContext());
        }

        public void configureBufferSizes(Integer netReadBufSize, Integer netWriteBufSize, Integer appBufSize) {
            this.netReadBufSizeOverride = netReadBufSize;
            this.netWriteBufSizeOverride = netWriteBufSize;
            this.appBufSizeOverride = appBufSize;
        }

        protected SslTransportLayer buildTransportLayer(SslFactory sslFactory, String id, SelectionKey key, ChannelMetadataRegistry metadataRegistry) throws IOException {
            SocketChannel socketChannel = (SocketChannel)key.channel();
            SSLEngine sslEngine = sslFactory.createSslEngine(socketChannel.socket());
            return this.newTransportLayer(id, key, sslEngine);
        }

        protected TestSslTransportLayer newTransportLayer(String id, SelectionKey key, SSLEngine sslEngine) throws IOException {
            return new TestSslTransportLayer(id, key, sslEngine);
        }

        static class ResizeableBufferSize {
            private Integer bufSizeOverride;

            ResizeableBufferSize(Integer bufSizeOverride) {
                this.bufSizeOverride = bufSizeOverride;
            }

            int updateAndGet(int actualSize, boolean update) {
                int size = actualSize;
                if (this.bufSizeOverride != null) {
                    if (update) {
                        this.bufSizeOverride = Math.min(this.bufSizeOverride * 2, size);
                    }
                    size = this.bufSizeOverride;
                }
                return size;
            }
        }

        class TestSslTransportLayer
        extends SslTransportLayer {
            private final ResizeableBufferSize netReadBufSize;
            private final ResizeableBufferSize netWriteBufSize;
            private final ResizeableBufferSize appBufSize;
            private final AtomicLong numReadsRemaining;
            private final AtomicLong numFlushesRemaining;
            private final AtomicInteger numDelayedFlushesRemaining;

            public TestSslTransportLayer(String channelId, SelectionKey key, SSLEngine sslEngine) {
                super(channelId, key, sslEngine, (ChannelMetadataRegistry)new DefaultChannelMetadataRegistry());
                this.netReadBufSize = new ResizeableBufferSize(TestSslChannelBuilder.this.netReadBufSizeOverride);
                this.netWriteBufSize = new ResizeableBufferSize(TestSslChannelBuilder.this.netWriteBufSizeOverride);
                this.appBufSize = new ResizeableBufferSize(TestSslChannelBuilder.this.appBufSizeOverride);
                this.numReadsRemaining = new AtomicLong(TestSslChannelBuilder.this.failureIndex);
                this.numFlushesRemaining = new AtomicLong(TestSslChannelBuilder.this.failureIndex);
                this.numDelayedFlushesRemaining = new AtomicInteger(TestSslChannelBuilder.this.flushDelayCount);
            }

            protected int netReadBufferSize() {
                ByteBuffer netReadBuffer = this.netReadBuffer();
                boolean updateBufSize = netReadBuffer != null && !this.netReadBuffer().hasRemaining();
                return this.netReadBufSize.updateAndGet(super.netReadBufferSize(), updateBufSize);
            }

            protected int netWriteBufferSize() {
                return this.netWriteBufSize.updateAndGet(super.netWriteBufferSize(), true);
            }

            protected int applicationBufferSize() {
                return this.appBufSize.updateAndGet(super.applicationBufferSize(), true);
            }

            protected int readFromSocketChannel() throws IOException {
                if (this.numReadsRemaining.decrementAndGet() == 0L && !this.ready()) {
                    TestSslChannelBuilder.this.readFailureAction.run();
                }
                return super.readFromSocketChannel();
            }

            protected boolean flush(ByteBuffer buf) throws IOException {
                if (this.numFlushesRemaining.decrementAndGet() == 0L && !this.ready()) {
                    TestSslChannelBuilder.this.flushFailureAction.run();
                } else if (this.numDelayedFlushesRemaining.getAndDecrement() != 0) {
                    return false;
                }
                this.resetDelayedFlush();
                return super.flush(buf);
            }

            protected void startHandshake() throws IOException {
                Assert.assertTrue((String)"SSL handshake initialized too early", (boolean)this.socketChannel().isConnected());
                super.startHandshake();
            }

            private void resetDelayedFlush() {
                this.numDelayedFlushesRemaining.set(TestSslChannelBuilder.this.flushDelayCount);
            }
        }
    }

    @FunctionalInterface
    private static interface FailureAction {
        public static final FailureAction NO_OP = () -> {};
        public static final FailureAction THROW_IO_EXCEPTION = () -> {
            throw new IOException("Test IO exception");
        };

        public void run() throws IOException;
    }
}

