/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.ssl;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.UnrecoverableEntryException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.TrustManagerFactory;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.SslClientAuth;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.security.ssl.NettySslEngineBuilder;
import org.apache.kafka.common.utils.SecurityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslEngineBuilder {
    private static final Logger log = LoggerFactory.getLogger(SslEngineBuilder.class);
    private final Map<String, ?> configs;
    private final String protocol;
    private final String provider;
    private final String kmfAlgorithm;
    private final String tmfAlgorithm;
    private final SecurityStore keystore;
    private final SecurityStore truststore;
    private final String[] cipherSuites;
    private final String[] enabledProtocols;
    private final SecureRandom secureRandomImplementation;
    private final SSLContext sslContext;
    private final SslClientAuth sslClientAuth;
    private final NettySslEngineBuilder nettySslEngineBuilder;

    SslEngineBuilder(Map<String, ?> configs, boolean nettyAllowed) {
        this.configs = Collections.unmodifiableMap(configs);
        this.protocol = (String)configs.get("ssl.protocol");
        this.provider = (String)configs.get("ssl.provider");
        SecurityUtils.addConfiguredSecurityProviders(this.configs);
        List cipherSuitesList = (List)configs.get("ssl.cipher.suites");
        this.cipherSuites = cipherSuitesList != null && !cipherSuitesList.isEmpty() ? cipherSuitesList.toArray(new String[cipherSuitesList.size()]) : null;
        List enabledProtocolsList = (List)configs.get("ssl.enabled.protocols");
        this.enabledProtocols = enabledProtocolsList != null && !enabledProtocolsList.isEmpty() ? enabledProtocolsList.toArray(new String[enabledProtocolsList.size()]) : null;
        this.secureRandomImplementation = SslEngineBuilder.createSecureRandom((String)configs.get("ssl.secure.random.implementation"));
        this.sslClientAuth = SslEngineBuilder.createSslClientAuth((String)configs.get("ssl.client.auth"));
        this.kmfAlgorithm = (String)configs.get("ssl.keymanager.algorithm");
        this.tmfAlgorithm = (String)configs.get("ssl.trustmanager.algorithm");
        this.keystore = SslEngineBuilder.createKeystore((String)configs.get("ssl.keystore.type"), (String)configs.get("ssl.keystore.location"), (Password)configs.get("ssl.keystore.password"), (Password)configs.get("ssl.key.password"));
        this.truststore = SslEngineBuilder.createTruststore((String)configs.get("ssl.truststore.type"), (String)configs.get("ssl.truststore.location"), (Password)configs.get("ssl.truststore.password"));
        this.sslContext = this.createSSLContext();
        if (!nettyAllowed) {
            this.nettySslEngineBuilder = null;
        } else if (!SslEngineBuilder.sslEngineBuilderClassIsNetty((String)configs.get("ssl.engine.builder.class"))) {
            this.nettySslEngineBuilder = null;
        } else if (this.keystore == null) {
            log.warn("Disabling netty because no keystore is configured.");
            this.nettySslEngineBuilder = null;
        } else {
            this.nettySslEngineBuilder = NettySslEngineBuilder.maybeCreate(this);
        }
    }

    private static boolean sslEngineBuilderClassIsNetty(String engineBuilderClass) {
        if (engineBuilderClass == null || engineBuilderClass.equals("org.apache.kafka.common.security.ssl.KafkaSslEngineBuilder")) {
            return false;
        }
        if (engineBuilderClass.equals("io.confluent.kafka.security.ssl.NettySslEngineBuilder")) {
            return true;
        }
        throw new RuntimeException("Invalid configuration value for ssl.engine.builder.class");
    }

    private static SslClientAuth createSslClientAuth(String key) {
        SslClientAuth auth = SslClientAuth.forConfig(key);
        if (auth != null) {
            return auth;
        }
        log.warn("Unrecognized client authentication configuration {}.  Falling back to NONE.  Recognized client authentication configurations are {}.", (Object)key, (Object)String.join((CharSequence)", ", SslClientAuth.VALUES.stream().map(a -> a.name()).collect(Collectors.toList())));
        return SslClientAuth.NONE;
    }

    private static SecureRandom createSecureRandom(String key) {
        if (key == null) {
            return null;
        }
        try {
            return SecureRandom.getInstance(key);
        }
        catch (GeneralSecurityException e) {
            throw new KafkaException(e);
        }
    }

    private SSLContext createSSLContext() {
        try {
            SSLContext sslContext = this.provider != null ? SSLContext.getInstance(this.protocol, this.provider) : SSLContext.getInstance(this.protocol);
            KeyManager[] keyManagers = null;
            if (this.keystore != null || this.kmfAlgorithm != null) {
                String kmfAlgorithm = this.kmfAlgorithm != null ? this.kmfAlgorithm : KeyManagerFactory.getDefaultAlgorithm();
                KeyManagerFactory kmf = KeyManagerFactory.getInstance(kmfAlgorithm);
                if (this.keystore != null) {
                    KeyStore ks = this.keystore.load();
                    Password keyPassword = this.keystore.keyPassword != null ? this.keystore.keyPassword : this.keystore.password;
                    kmf.init(ks, keyPassword.value().toCharArray());
                } else {
                    kmf.init(null, null);
                }
                keyManagers = kmf.getKeyManagers();
            }
            String tmfAlgorithm = this.tmfAlgorithm != null ? this.tmfAlgorithm : TrustManagerFactory.getDefaultAlgorithm();
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
            KeyStore ts = this.truststore == null ? null : this.truststore.load();
            tmf.init(ts);
            sslContext.init(keyManagers, tmf.getTrustManagers(), this.secureRandomImplementation);
            log.debug("Created SSL context with keystore {}, truststore {}, provider {}.", new Object[]{this.keystore, this.truststore, sslContext.getProvider().getName()});
            return sslContext;
        }
        catch (Exception e) {
            throw new KafkaException(e);
        }
    }

    private static SecurityStore createKeystore(String type, String path, Password password, Password keyPassword) {
        if (path == null && password != null) {
            throw new KafkaException("SSL key store is not specified, but key store password is specified.");
        }
        if (path != null && password == null) {
            throw new KafkaException("SSL key store is specified, but key store password is not specified.");
        }
        if (path != null && password != null) {
            return new SecurityStore(type, path, password, keyPassword);
        }
        return null;
    }

    private static SecurityStore createTruststore(String type, String path, Password password) {
        if (path == null && password != null) {
            throw new KafkaException("SSL trust store is not specified, but trust store password is specified.");
        }
        if (path != null) {
            return new SecurityStore(type, path, password, null);
        }
        return null;
    }

    Map<String, Object> configs() {
        return this.configs;
    }

    public SecurityStore keystore() {
        return this.keystore;
    }

    public SecurityStore truststore() {
        return this.truststore;
    }

    String[] cipherSuites() {
        return this.cipherSuites;
    }

    String[] enabledProtocols() {
        return this.enabledProtocols;
    }

    SslClientAuth sslClientAuth() {
        return this.sslClientAuth;
    }

    public SSLEngine createSslEngine(Mode mode, String peerHost, int peerPort, String endpointIdentification) {
        if (mode == Mode.SERVER && this.nettySslEngineBuilder != null) {
            return this.nettySslEngineBuilder.newEngine(peerHost, peerPort);
        }
        SSLEngine sslEngine = this.sslContext.createSSLEngine(peerHost, peerPort);
        if (this.cipherSuites != null) {
            sslEngine.setEnabledCipherSuites(this.cipherSuites);
        }
        if (this.enabledProtocols != null) {
            sslEngine.setEnabledProtocols(this.enabledProtocols);
        }
        if (mode == Mode.SERVER) {
            sslEngine.setUseClientMode(false);
            switch (this.sslClientAuth) {
                case REQUIRED: {
                    sslEngine.setNeedClientAuth(true);
                    break;
                }
                case REQUESTED: {
                    sslEngine.setWantClientAuth(true);
                    break;
                }
            }
            sslEngine.setUseClientMode(false);
        } else {
            sslEngine.setUseClientMode(true);
            SSLParameters sslParams = sslEngine.getSSLParameters();
            sslParams.setEndpointIdentificationAlgorithm(endpointIdentification);
            sslEngine.setSSLParameters(sslParams);
        }
        return sslEngine;
    }

    public SSLContext sslContext() {
        return this.sslContext;
    }

    public boolean shouldBeRebuilt(Map<String, Object> nextConfigs) {
        if (!nextConfigs.equals(this.configs)) {
            return true;
        }
        if (this.truststore != null && this.truststore.modified()) {
            return true;
        }
        return this.keystore != null && this.keystore.modified();
    }

    public Closeable sslEngineCloser(SSLEngine engine) {
        return new CloseableSslEngine(engine);
    }

    static class SecurityStore {
        private final String type;
        private final String path;
        private final Password password;
        private final Password keyPassword;
        private final Long fileLastModifiedMs;

        SecurityStore(String type, String path, Password password, Password keyPassword) {
            Objects.requireNonNull(type, "type must not be null");
            this.type = type;
            this.path = path;
            this.password = password;
            this.keyPassword = keyPassword;
            this.fileLastModifiedMs = this.lastModifiedMs(path);
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        KeyStore load() {
            try (InputStream in = Files.newInputStream(Paths.get(this.path, new String[0]), new OpenOption[0]);){
                KeyStore ks = KeyStore.getInstance(this.type);
                char[] passwordChars = this.password != null ? this.password.value().toCharArray() : null;
                ks.load(in, passwordChars);
                KeyStore keyStore = ks;
                return keyStore;
            }
            catch (IOException | GeneralSecurityException e) {
                throw new KafkaException("Failed to load SSL keystore " + this.path + " of type " + this.type, e);
            }
        }

        private Long lastModifiedMs(String path) {
            try {
                return Files.getLastModifiedTime(Paths.get(path, new String[0]), new LinkOption[0]).toMillis();
            }
            catch (IOException e) {
                log.error("Modification time of key store could not be obtained: " + path, (Throwable)e);
                return null;
            }
        }

        boolean modified() {
            Long modifiedMs = this.lastModifiedMs(this.path);
            return modifiedMs != null && !Objects.equals(modifiedMs, this.fileLastModifiedMs);
        }

        PrivateKeyData loadPrivateKeyData() {
            KeyStore store = this.load();
            KeyStore.PasswordProtection keyProtection = this.keyPassword == null ? null : new KeyStore.PasswordProtection(this.keyPassword.value().toCharArray());
            try {
                Enumeration<String> aliases = store.aliases();
                while (aliases.hasMoreElements()) {
                    String alias = aliases.nextElement();
                    if (!store.isKeyEntry(alias)) continue;
                    try {
                        KeyStore.Entry entry = store.getEntry(alias, keyProtection);
                        if (!(entry instanceof KeyStore.PrivateKeyEntry)) continue;
                        KeyStore.PrivateKeyEntry privateKeyEntry = (KeyStore.PrivateKeyEntry)entry;
                        PrivateKey privateKey = privateKeyEntry.getPrivateKey();
                        Certificate[] certs = privateKeyEntry.getCertificateChain();
                        if (!(certs instanceof X509Certificate[])) {
                            throw new RuntimeException("Expected a certificate chain of type X509Certificate for alias " + alias);
                        }
                        return new PrivateKeyData(privateKey, (X509Certificate[])certs);
                    }
                    catch (NoSuchAlgorithmException e) {
                        log.info("can't find the algorithm for recovering the {} entry.", (Object)alias);
                    }
                    catch (UnrecoverableEntryException e) {
                        log.trace("ignoring alias {}, since the password doesn't match.", (Object)alias);
                    }
                }
            }
            catch (KeyStoreException e) {
                throw new KafkaException(e);
            }
            throw new RuntimeException("No private key found protected with the given password in " + this.path);
        }

        X509Certificate[] loadAllCertificates() {
            KeyStore store = this.load();
            ArrayList<X509Certificate> all = new ArrayList<X509Certificate>();
            try {
                Enumeration<String> aliases = store.aliases();
                while (aliases.hasMoreElements()) {
                    String alias = aliases.nextElement();
                    if (!store.isCertificateEntry(alias)) continue;
                    Certificate cert = store.getCertificate(alias);
                    if (!(cert instanceof X509Certificate)) {
                        throw new RuntimeException("Expected a certificate of type X509Certificate for alias " + alias);
                    }
                    all.add((X509Certificate)cert);
                }
            }
            catch (KeyStoreException e) {
                throw new KafkaException(e);
            }
            return all.toArray(new X509Certificate[0]);
        }

        public String toString() {
            return "SecurityStore(path=" + this.path + ", modificationTime=" + (this.fileLastModifiedMs == null ? null : new Date(this.fileLastModifiedMs)) + ")";
        }
    }

    static class PrivateKeyData {
        private final PrivateKey key;
        private final X509Certificate[] certificateChain;

        PrivateKeyData(PrivateKey key, X509Certificate[] certificateChain) {
            this.key = key;
            this.certificateChain = certificateChain;
        }

        PrivateKey key() {
            return this.key;
        }

        X509Certificate[] certificateChain() {
            return this.certificateChain;
        }
    }

    class CloseableSslEngine
    implements Closeable {
        private final SSLEngine engine;

        CloseableSslEngine(SSLEngine engine) {
            this.engine = engine;
        }

        @Override
        public void close() throws IOException {
            if (SslEngineBuilder.this.nettySslEngineBuilder != null) {
                SslEngineBuilder.this.nettySslEngineBuilder.closeEngine(this.engine);
            }
        }
    }
}

