/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mapreduce;

import com.google.common.base.Charsets;
import com.google.common.base.Throwables;
import com.google.common.collect.Ordering;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.mapreduce.TableSplit;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.MapFile;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableComparable;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.MapFileOutputFormat;
import org.apache.hadoop.mapreduce.lib.partition.TotalOrderPartitioner;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

public class HashTable
extends Configured
implements Tool {
    private static final Log LOG = LogFactory.getLog(HashTable.class);
    private static final int DEFAULT_BATCH_SIZE = 8000;
    private static final String HASH_BATCH_SIZE_CONF_KEY = "hash.batch.size";
    static final String PARTITIONS_FILE_NAME = "partitions";
    static final String MANIFEST_FILE_NAME = "manifest";
    static final String HASH_DATA_DIR = "hashes";
    static final String OUTPUT_DATA_FILE_PREFIX = "part-r-";
    static final String IGNORE_TIMESTAMPS = "ignoreTimestamps";
    private static final String TMP_MANIFEST_FILE_NAME = "manifest.tmp";
    TableHash tableHash = new TableHash();
    Path destPath;
    private static final int NUM_ARGS = 2;

    public HashTable(Configuration conf) {
        super(conf);
    }

    static boolean isTableStartRow(byte[] row) {
        return Bytes.equals((byte[])HConstants.EMPTY_START_ROW, (byte[])row);
    }

    static boolean isTableEndRow(byte[] row) {
        return Bytes.equals((byte[])HConstants.EMPTY_END_ROW, (byte[])row);
    }

    public Job createSubmittableJob(String[] args) throws IOException {
        Path partitionsPath = new Path(this.destPath, PARTITIONS_FILE_NAME);
        this.generatePartitions(partitionsPath);
        Job job = Job.getInstance((Configuration)this.getConf(), (String)this.getConf().get("mapreduce.job.name", "hashTable_" + this.tableHash.tableName));
        Configuration jobConf = job.getConfiguration();
        jobConf.setLong(HASH_BATCH_SIZE_CONF_KEY, this.tableHash.batchSize);
        jobConf.setBoolean(IGNORE_TIMESTAMPS, this.tableHash.ignoreTimestamps);
        job.setJarByClass(HashTable.class);
        TableMapReduceUtil.initTableMapperJob(this.tableHash.tableName, this.tableHash.initScan(), HashMapper.class, ImmutableBytesWritable.class, ImmutableBytesWritable.class, job);
        job.setPartitionerClass(TotalOrderPartitioner.class);
        TotalOrderPartitioner.setPartitionFile((Configuration)jobConf, (Path)partitionsPath);
        job.setReducerClass(Reducer.class);
        job.setNumReduceTasks(this.tableHash.numHashFiles);
        job.setOutputKeyClass(ImmutableBytesWritable.class);
        job.setOutputValueClass(ImmutableBytesWritable.class);
        job.setOutputFormatClass(MapFileOutputFormat.class);
        FileOutputFormat.setOutputPath((Job)job, (Path)new Path(this.destPath, HASH_DATA_DIR));
        return job;
    }

    private void generatePartitions(Path partitionsPath) throws IOException {
        Connection connection = ConnectionFactory.createConnection((Configuration)this.getConf());
        Pair regionKeys = connection.getRegionLocator(TableName.valueOf((String)this.tableHash.tableName)).getStartEndKeys();
        connection.close();
        this.tableHash.selectPartitions((Pair<byte[][], byte[][]>)regionKeys);
        LOG.info((Object)("Writing " + this.tableHash.partitions.size() + " partition keys to " + partitionsPath));
        this.tableHash.writePartitionFile(this.getConf(), partitionsPath);
    }

    private void writeTempManifestFile() throws IOException {
        Path tempManifestPath = new Path(this.destPath, TMP_MANIFEST_FILE_NAME);
        FileSystem fs = tempManifestPath.getFileSystem(this.getConf());
        this.tableHash.writePropertiesFile(fs, tempManifestPath);
    }

    private void completeManifest() throws IOException {
        Path tempManifestPath = new Path(this.destPath, TMP_MANIFEST_FILE_NAME);
        Path manifestPath = new Path(this.destPath, MANIFEST_FILE_NAME);
        FileSystem fs = tempManifestPath.getFileSystem(this.getConf());
        fs.rename(tempManifestPath, manifestPath);
    }

    private static void printUsage(String errorMsg) {
        if (errorMsg != null && errorMsg.length() > 0) {
            System.err.println("ERROR: " + errorMsg);
            System.err.println();
        }
        System.err.println("Usage: HashTable [options] <tablename> <outputpath>");
        System.err.println();
        System.err.println("Options:");
        System.err.println(" batchsize         the target amount of bytes to hash in each batch");
        System.err.println("                   rows are added to the batch until this size is reached");
        System.err.println("                   (defaults to 8000 bytes)");
        System.err.println(" numhashfiles      the number of hash files to create");
        System.err.println("                   if set to fewer than number of regions then");
        System.err.println("                   the job will create this number of reducers");
        System.err.println("                   (defaults to 1/100 of regions -- at least 1)");
        System.err.println(" startrow          the start row");
        System.err.println(" stoprow           the stop row");
        System.err.println(" starttime         beginning of the time range (unixtime in millis)");
        System.err.println("                   without endtime means from starttime to forever");
        System.err.println(" endtime           end of the time range.");
        System.err.println("                   Ignored if no starttime specified.");
        System.err.println(" scanbatch         scanner batch size to support intra row scans");
        System.err.println(" versions          number of cell versions to include");
        System.err.println(" families          comma-separated list of families to include");
        System.err.println(" ignoreTimestamps  if true, ignores cell timestamps");
        System.err.println("                   when calculating hashes");
        System.err.println();
        System.err.println("Args:");
        System.err.println(" tablename     Name of the table to hash");
        System.err.println(" outputpath    Filesystem path to put the output data");
        System.err.println();
        System.err.println("Examples:");
        System.err.println(" To hash 'TestTable' in 32kB batches for a 1 hour window into 50 files:");
        System.err.println(" $ bin/hbase org.apache.hadoop.hbase.mapreduce.HashTable --batchsize=32000 --numhashfiles=50 --starttime=1265875194289 --endtime=1265878794289 --families=cf2,cf3 TestTable /hashes/testTable");
    }

    private boolean doCommandLine(String[] args) {
        if (args.length < 2) {
            HashTable.printUsage(null);
            return false;
        }
        try {
            this.tableHash.tableName = args[args.length - 2];
            this.destPath = new Path(args[args.length - 1]);
            for (int i = 0; i < args.length - 2; ++i) {
                String cmd = args[i];
                if (cmd.equals("-h") || cmd.startsWith("--h")) {
                    HashTable.printUsage(null);
                    return false;
                }
                String batchSizeArgKey = "--batchsize=";
                if (cmd.startsWith("--batchsize=")) {
                    this.tableHash.batchSize = Long.parseLong(cmd.substring("--batchsize=".length()));
                    continue;
                }
                String numHashFilesArgKey = "--numhashfiles=";
                if (cmd.startsWith("--numhashfiles=")) {
                    this.tableHash.numHashFiles = Integer.parseInt(cmd.substring("--numhashfiles=".length()));
                    continue;
                }
                String startRowArgKey = "--startrow=";
                if (cmd.startsWith("--startrow=")) {
                    this.tableHash.startRow = Bytes.fromHex((String)cmd.substring("--startrow=".length()));
                    continue;
                }
                String stopRowArgKey = "--stoprow=";
                if (cmd.startsWith("--stoprow=")) {
                    this.tableHash.stopRow = Bytes.fromHex((String)cmd.substring("--stoprow=".length()));
                    continue;
                }
                String startTimeArgKey = "--starttime=";
                if (cmd.startsWith("--starttime=")) {
                    this.tableHash.startTime = Long.parseLong(cmd.substring("--starttime=".length()));
                    continue;
                }
                String endTimeArgKey = "--endtime=";
                if (cmd.startsWith("--endtime=")) {
                    this.tableHash.endTime = Long.parseLong(cmd.substring("--endtime=".length()));
                    continue;
                }
                String scanBatchArgKey = "--scanbatch=";
                if (cmd.startsWith("--scanbatch=")) {
                    this.tableHash.scanBatch = Integer.parseInt(cmd.substring("--scanbatch=".length()));
                    continue;
                }
                String versionsArgKey = "--versions=";
                if (cmd.startsWith("--versions=")) {
                    this.tableHash.versions = Integer.parseInt(cmd.substring("--versions=".length()));
                    continue;
                }
                String familiesArgKey = "--families=";
                if (cmd.startsWith("--families=")) {
                    this.tableHash.families = cmd.substring("--families=".length());
                    continue;
                }
                String ignoreTimestampsKey = "--ignoreTimestamps=";
                if (cmd.startsWith("--ignoreTimestamps=")) {
                    this.tableHash.ignoreTimestamps = Boolean.parseBoolean(cmd.substring("--ignoreTimestamps=".length()));
                    continue;
                }
                HashTable.printUsage("Invalid argument '" + cmd + "'");
                return false;
            }
            if ((this.tableHash.startTime != 0L || this.tableHash.endTime != 0L) && this.tableHash.startTime >= this.tableHash.endTime) {
                HashTable.printUsage("Invalid time range filter: starttime=" + this.tableHash.startTime + " >=  endtime=" + this.tableHash.endTime);
                return false;
            }
        }
        catch (Exception e) {
            e.printStackTrace();
            HashTable.printUsage("Can't start because " + e.getMessage());
            return false;
        }
        return true;
    }

    public static void main(String[] args) throws Exception {
        int ret = ToolRunner.run((Tool)new HashTable(HBaseConfiguration.create()), (String[])args);
        System.exit(ret);
    }

    public int run(String[] args) throws Exception {
        String[] otherArgs = new GenericOptionsParser(this.getConf(), args).getRemainingArgs();
        if (!this.doCommandLine(otherArgs)) {
            return 1;
        }
        Job job = this.createSubmittableJob(otherArgs);
        this.writeTempManifestFile();
        if (!job.waitForCompletion(true)) {
            LOG.info((Object)"Map-reduce job failed!");
            return 1;
        }
        this.completeManifest();
        return 0;
    }

    public static class HashMapper
    extends TableMapper<ImmutableBytesWritable, ImmutableBytesWritable> {
        private ResultHasher hasher;
        private long targetBatchSize;
        private ImmutableBytesWritable currentRow;

        protected void setup(Mapper.Context context) throws IOException, InterruptedException {
            this.targetBatchSize = context.getConfiguration().getLong(HashTable.HASH_BATCH_SIZE_CONF_KEY, 8000L);
            this.hasher = new ResultHasher();
            this.hasher.ignoreTimestamps = context.getConfiguration().getBoolean(HashTable.IGNORE_TIMESTAMPS, false);
            TableSplit split = (TableSplit)context.getInputSplit();
            this.hasher.startBatch(new ImmutableBytesWritable(split.getStartRow()));
        }

        protected void map(ImmutableBytesWritable key, Result value, Mapper.Context context) throws IOException, InterruptedException {
            if (this.currentRow == null || !this.currentRow.equals((Object)key)) {
                this.currentRow = new ImmutableBytesWritable(key);
                if (this.hasher.getBatchSize() >= this.targetBatchSize) {
                    this.hasher.finishBatch();
                    context.write((Object)this.hasher.getBatchStartKey(), (Object)this.hasher.getBatchHash());
                    this.hasher.startBatch(this.currentRow);
                }
            }
            this.hasher.hashResult(value);
        }

        protected void cleanup(Mapper.Context context) throws IOException, InterruptedException {
            this.hasher.finishBatch();
            context.write((Object)this.hasher.getBatchStartKey(), (Object)this.hasher.getBatchHash());
        }
    }

    static class ResultHasher {
        private MessageDigest digest;
        private boolean batchStarted = false;
        private ImmutableBytesWritable batchStartKey;
        private ImmutableBytesWritable batchHash;
        private long batchSize = 0L;
        boolean ignoreTimestamps;

        public ResultHasher() {
            try {
                this.digest = MessageDigest.getInstance("MD5");
            }
            catch (NoSuchAlgorithmException e) {
                Throwables.propagate((Throwable)e);
            }
        }

        public void startBatch(ImmutableBytesWritable row) {
            if (this.batchStarted) {
                throw new RuntimeException("Cannot start new batch without finishing existing one.");
            }
            this.batchStarted = true;
            this.batchSize = 0L;
            this.batchStartKey = row;
            this.batchHash = null;
        }

        public void hashResult(Result result) {
            if (!this.batchStarted) {
                throw new RuntimeException("Cannot add to batch that has not been started.");
            }
            for (Cell cell : result.rawCells()) {
                short rowLength = cell.getRowLength();
                byte familyLength = cell.getFamilyLength();
                int qualifierLength = cell.getQualifierLength();
                int valueLength = cell.getValueLength();
                this.digest.update(cell.getRowArray(), cell.getRowOffset(), rowLength);
                this.digest.update(cell.getFamilyArray(), cell.getFamilyOffset(), familyLength);
                this.digest.update(cell.getQualifierArray(), cell.getQualifierOffset(), qualifierLength);
                long ts = cell.getTimestamp();
                if (!this.ignoreTimestamps) {
                    for (int i = 8; i > 0; --i) {
                        this.digest.update((byte)ts);
                        ts >>>= 8;
                    }
                }
                this.digest.update(cell.getValueArray(), cell.getValueOffset(), valueLength);
                this.batchSize += (long)(rowLength + familyLength + qualifierLength + 8 + valueLength);
            }
        }

        public void finishBatch() {
            if (!this.batchStarted) {
                throw new RuntimeException("Cannot finish batch that has not started.");
            }
            this.batchStarted = false;
            this.batchHash = new ImmutableBytesWritable(this.digest.digest());
        }

        public boolean isBatchStarted() {
            return this.batchStarted;
        }

        public ImmutableBytesWritable getBatchStartKey() {
            return this.batchStartKey;
        }

        public ImmutableBytesWritable getBatchHash() {
            return this.batchHash;
        }

        public long getBatchSize() {
            return this.batchSize;
        }
    }

    public static class TableHash {
        Path hashDir;
        String tableName;
        String families = null;
        long batchSize = 8000L;
        int numHashFiles = 0;
        byte[] startRow = HConstants.EMPTY_START_ROW;
        byte[] stopRow = HConstants.EMPTY_END_ROW;
        int scanBatch = 0;
        int versions = -1;
        long startTime = 0L;
        long endTime = 0L;
        boolean ignoreTimestamps;
        List<ImmutableBytesWritable> partitions;

        public static TableHash read(Configuration conf, Path hashDir) throws IOException {
            TableHash tableHash = new TableHash();
            FileSystem fs = hashDir.getFileSystem(conf);
            tableHash.hashDir = hashDir;
            tableHash.readPropertiesFile(fs, new Path(hashDir, HashTable.MANIFEST_FILE_NAME));
            tableHash.readPartitionFile(fs, conf, new Path(hashDir, HashTable.PARTITIONS_FILE_NAME));
            return tableHash;
        }

        void writePropertiesFile(FileSystem fs, Path path) throws IOException {
            Properties p = new Properties();
            p.setProperty("table", this.tableName);
            if (this.families != null) {
                p.setProperty("columnFamilies", this.families);
            }
            p.setProperty("targetBatchSize", Long.toString(this.batchSize));
            p.setProperty("numHashFiles", Integer.toString(this.numHashFiles));
            if (!HashTable.isTableStartRow(this.startRow)) {
                p.setProperty("startRowHex", Bytes.toHex((byte[])this.startRow));
            }
            if (!HashTable.isTableEndRow(this.stopRow)) {
                p.setProperty("stopRowHex", Bytes.toHex((byte[])this.stopRow));
            }
            if (this.scanBatch > 0) {
                p.setProperty("scanBatch", Integer.toString(this.scanBatch));
            }
            if (this.versions >= 0) {
                p.setProperty("versions", Integer.toString(this.versions));
            }
            if (this.startTime != 0L) {
                p.setProperty("startTimestamp", Long.toString(this.startTime));
            }
            if (this.endTime != 0L) {
                p.setProperty("endTimestamp", Long.toString(this.endTime));
            }
            try (OutputStreamWriter osw = new OutputStreamWriter((OutputStream)fs.create(path), Charsets.UTF_8);){
                p.store(osw, null);
            }
        }

        void readPropertiesFile(FileSystem fs, Path path) throws IOException {
            String endTimeString;
            String startTimeString;
            String versionString;
            String scanBatchString;
            String stopRowHex;
            Properties p = new Properties();
            try (FSDataInputStream in = fs.open(path);
                 InputStreamReader isr = new InputStreamReader((InputStream)in, Charsets.UTF_8);){
                p.load(isr);
            }
            this.tableName = p.getProperty("table");
            this.families = p.getProperty("columnFamilies");
            this.batchSize = Long.parseLong(p.getProperty("targetBatchSize"));
            this.numHashFiles = Integer.parseInt(p.getProperty("numHashFiles"));
            String startRowHex = p.getProperty("startRowHex");
            if (startRowHex != null) {
                this.startRow = Bytes.fromHex((String)startRowHex);
            }
            if ((stopRowHex = p.getProperty("stopRowHex")) != null) {
                this.stopRow = Bytes.fromHex((String)stopRowHex);
            }
            if ((scanBatchString = p.getProperty("scanBatch")) != null) {
                this.scanBatch = Integer.parseInt(scanBatchString);
            }
            if ((versionString = p.getProperty("versions")) != null) {
                this.versions = Integer.parseInt(versionString);
            }
            if ((startTimeString = p.getProperty("startTimestamp")) != null) {
                this.startTime = Long.parseLong(startTimeString);
            }
            if ((endTimeString = p.getProperty("endTimestamp")) != null) {
                this.endTime = Long.parseLong(endTimeString);
            }
        }

        Scan initScan() throws IOException {
            Scan scan = new Scan();
            scan.setCacheBlocks(false);
            if (this.startTime != 0L || this.endTime != 0L) {
                scan.setTimeRange(this.startTime, this.endTime == 0L ? Long.MAX_VALUE : this.endTime);
            }
            if (this.scanBatch > 0) {
                scan.setBatch(this.scanBatch);
            }
            if (this.versions >= 0) {
                scan.setMaxVersions(this.versions);
            }
            if (!HashTable.isTableStartRow(this.startRow)) {
                scan.setStartRow(this.startRow);
            }
            if (!HashTable.isTableEndRow(this.stopRow)) {
                scan.setStopRow(this.stopRow);
            }
            if (this.families != null) {
                for (String fam : this.families.split(",")) {
                    scan.addFamily(Bytes.toBytes((String)fam));
                }
            }
            return scan;
        }

        void selectPartitions(Pair<byte[][], byte[][]> regionStartEndKeys) {
            ArrayList<byte[]> startKeys = new ArrayList<byte[]>();
            for (int i = 0; i < ((byte[][])regionStartEndKeys.getFirst()).length; ++i) {
                byte[] regionStartKey = ((byte[][])regionStartEndKeys.getFirst())[i];
                byte[] regionEndKey = ((byte[][])regionStartEndKeys.getSecond())[i];
                if (!HashTable.isTableStartRow(this.startRow) && !HashTable.isTableEndRow(regionEndKey) && Bytes.compareTo((byte[])this.startRow, (byte[])regionEndKey) >= 0 || !HashTable.isTableEndRow(this.stopRow) && !HashTable.isTableStartRow(regionStartKey) && Bytes.compareTo((byte[])this.stopRow, (byte[])regionStartKey) <= 0) continue;
                startKeys.add(regionStartKey);
            }
            int numRegions = startKeys.size();
            if (this.numHashFiles == 0) {
                this.numHashFiles = numRegions / 100;
            }
            if (this.numHashFiles == 0) {
                this.numHashFiles = 1;
            }
            if (this.numHashFiles > numRegions) {
                this.numHashFiles = numRegions;
            }
            this.partitions = new ArrayList<ImmutableBytesWritable>(this.numHashFiles - 1);
            for (long i = 1L; i < (long)this.numHashFiles; ++i) {
                int splitIndex = (int)((long)numRegions * i / (long)this.numHashFiles);
                this.partitions.add(new ImmutableBytesWritable((byte[])startKeys.get(splitIndex)));
            }
        }

        void writePartitionFile(Configuration conf, Path path) throws IOException {
            FileSystem fs = path.getFileSystem(conf);
            SequenceFile.Writer writer = SequenceFile.createWriter((FileSystem)fs, (Configuration)conf, (Path)path, ImmutableBytesWritable.class, NullWritable.class);
            for (int i = 0; i < this.partitions.size(); ++i) {
                writer.append((Writable)this.partitions.get(i), (Writable)NullWritable.get());
            }
            writer.close();
        }

        private void readPartitionFile(FileSystem fs, Configuration conf, Path path) throws IOException {
            SequenceFile.Reader reader = new SequenceFile.Reader(fs, path, conf);
            ImmutableBytesWritable key = new ImmutableBytesWritable();
            this.partitions = new ArrayList<ImmutableBytesWritable>();
            while (reader.next((Writable)key)) {
                this.partitions.add(new ImmutableBytesWritable(key.copyBytes()));
            }
            reader.close();
            if (!Ordering.natural().isOrdered(this.partitions)) {
                throw new IOException("Partitions are not ordered!");
            }
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("tableName=").append(this.tableName);
            if (this.families != null) {
                sb.append(", families=").append(this.families);
            }
            sb.append(", batchSize=").append(this.batchSize);
            sb.append(", numHashFiles=").append(this.numHashFiles);
            if (!HashTable.isTableStartRow(this.startRow)) {
                sb.append(", startRowHex=").append(Bytes.toHex((byte[])this.startRow));
            }
            if (!HashTable.isTableEndRow(this.stopRow)) {
                sb.append(", stopRowHex=").append(Bytes.toHex((byte[])this.stopRow));
            }
            if (this.scanBatch >= 0) {
                sb.append(", scanBatch=").append(this.scanBatch);
            }
            if (this.versions >= 0) {
                sb.append(", versions=").append(this.versions);
            }
            if (this.startTime != 0L) {
                sb.append("startTime=").append(this.startTime);
            }
            if (this.endTime != 0L) {
                sb.append("endTime=").append(this.endTime);
            }
            return sb.toString();
        }

        static String getDataFileName(int hashFileIndex) {
            return String.format("part-r-%05d", hashFileIndex);
        }

        public Reader newReader(Configuration conf, ImmutableBytesWritable startKey) throws IOException {
            return new Reader(conf, startKey);
        }

        public class Reader
        implements Closeable {
            private final Configuration conf;
            private int hashFileIndex;
            private MapFile.Reader mapFileReader;
            private boolean cachedNext;
            private ImmutableBytesWritable key;
            private ImmutableBytesWritable hash;

            Reader(Configuration conf, ImmutableBytesWritable startKey) throws IOException {
                this.conf = conf;
                int partitionIndex = Collections.binarySearch(TableHash.this.partitions, startKey);
                this.hashFileIndex = partitionIndex >= 0 ? partitionIndex + 1 : -1 - partitionIndex;
                this.openHashFile();
                this.hash = new ImmutableBytesWritable();
                this.key = (ImmutableBytesWritable)this.mapFileReader.getClosest((WritableComparable)startKey, (Writable)this.hash);
                if (this.key == null) {
                    this.cachedNext = false;
                    this.hash = null;
                } else {
                    this.cachedNext = true;
                }
            }

            public boolean next() throws IOException {
                if (this.cachedNext) {
                    this.cachedNext = false;
                    return true;
                }
                this.key = new ImmutableBytesWritable();
                this.hash = new ImmutableBytesWritable();
                while (true) {
                    boolean hasNext;
                    if (hasNext = this.mapFileReader.next((WritableComparable)this.key, (Writable)this.hash)) {
                        return true;
                    }
                    ++this.hashFileIndex;
                    if (this.hashFileIndex >= TableHash.this.numHashFiles) break;
                    this.mapFileReader.close();
                    this.openHashFile();
                }
                this.key = null;
                this.hash = null;
                return false;
            }

            public ImmutableBytesWritable getCurrentKey() {
                return this.key;
            }

            public ImmutableBytesWritable getCurrentHash() {
                return this.hash;
            }

            private void openHashFile() throws IOException {
                if (this.mapFileReader != null) {
                    this.mapFileReader.close();
                }
                Path dataDir = new Path(TableHash.this.hashDir, HashTable.HASH_DATA_DIR);
                Path dataFile = new Path(dataDir, TableHash.getDataFileName(this.hashFileIndex));
                this.mapFileReader = new MapFile.Reader(dataFile, this.conf, new SequenceFile.Reader.Option[0]);
            }

            @Override
            public void close() throws IOException {
                this.mapFileReader.close();
            }
        }
    }
}

