/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.CatalogNotExistException;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.catalog.Catalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.CatalogView;
import org.apache.flink.table.catalog.ObjectIdentifier;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.UnresolvedIdentifier;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class CatalogManager {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogManager.class);
    private Map<String, Catalog> catalogs;
    private Map<ObjectIdentifier, CatalogBaseTable> temporaryTables;
    private String currentCatalogName;
    private String currentDatabaseName;
    private final String builtInCatalogName;

    public CatalogManager(String defaultCatalogName, Catalog defaultCatalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)defaultCatalogName) ? 1 : 0) != 0, (Object)"Default catalog name cannot be null or empty");
        Preconditions.checkNotNull((Object)defaultCatalog, (String)"Default catalog cannot be null");
        this.catalogs = new LinkedHashMap<String, Catalog>();
        this.catalogs.put(defaultCatalogName, defaultCatalog);
        this.currentCatalogName = defaultCatalogName;
        this.currentDatabaseName = defaultCatalog.getDefaultDatabase();
        this.temporaryTables = new HashMap<ObjectIdentifier, CatalogBaseTable>();
        this.builtInCatalogName = defaultCatalogName;
    }

    public void registerCatalog(String catalogName, Catalog catalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalog, (String)"Catalog cannot be null");
        if (this.catalogs.containsKey(catalogName)) {
            throw new CatalogException(String.format("Catalog %s already exists.", catalogName));
        }
        this.catalogs.put(catalogName, catalog);
        catalog.open();
    }

    public Optional<Catalog> getCatalog(String catalogName) {
        return Optional.ofNullable(this.catalogs.get(catalogName));
    }

    public String getCurrentCatalog() {
        return this.currentCatalogName;
    }

    public void setCurrentCatalog(String catalogName) throws CatalogNotExistException {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Catalog potentialCurrentCatalog = this.catalogs.get(catalogName);
        if (potentialCurrentCatalog == null) {
            throw new CatalogException(String.format("A catalog with name [%s] does not exist.", catalogName));
        }
        if (!this.currentCatalogName.equals(catalogName)) {
            this.currentCatalogName = catalogName;
            this.currentDatabaseName = potentialCurrentCatalog.getDefaultDatabase();
            LOG.info("Set the current default catalog as [{}] and the current default database as [{}].", (Object)this.currentCatalogName, (Object)this.currentDatabaseName);
        }
    }

    public String getCurrentDatabase() {
        return this.currentDatabaseName;
    }

    public void setCurrentDatabase(String databaseName) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)databaseName) ? 1 : 0) != 0, (Object)"The database name cannot be null or empty.");
        if (!this.catalogs.get(this.currentCatalogName).databaseExists(databaseName)) {
            throw new CatalogException(String.format("A database with name [%s] does not exist in the catalog: [%s].", databaseName, this.currentCatalogName));
        }
        if (!this.currentDatabaseName.equals(databaseName)) {
            this.currentDatabaseName = databaseName;
            LOG.info("Set the current default database as [{}] in the current default catalog [{}].", (Object)this.currentDatabaseName, (Object)this.currentCatalogName);
        }
    }

    public String getBuiltInCatalogName() {
        return this.builtInCatalogName;
    }

    public String getBuiltInDatabaseName() {
        return this.catalogs.get(this.getBuiltInCatalogName()).getDefaultDatabase();
    }

    public Optional<TableLookupResult> getTable(ObjectIdentifier objectIdentifier) {
        try {
            CatalogBaseTable temporaryTable = this.temporaryTables.get(objectIdentifier);
            if (temporaryTable != null) {
                return Optional.of(TableLookupResult.temporary(temporaryTable));
            }
            return this.getPermanentTable(objectIdentifier);
        }
        catch (TableNotExistException tableNotExistException) {
            return Optional.empty();
        }
    }

    private Optional<TableLookupResult> getPermanentTable(ObjectIdentifier objectIdentifier) throws TableNotExistException {
        Catalog currentCatalog = this.catalogs.get(objectIdentifier.getCatalogName());
        ObjectPath objectPath = objectIdentifier.toObjectPath();
        if (currentCatalog != null && currentCatalog.tableExists(objectPath)) {
            return Optional.of(TableLookupResult.permanent(currentCatalog.getTable(objectPath)));
        }
        return Optional.empty();
    }

    public Set<String> listCatalogs() {
        return Collections.unmodifiableSet(this.catalogs.keySet());
    }

    public Set<String> listTables() {
        return this.listTables(this.getCurrentCatalog(), this.getCurrentDatabase());
    }

    public Set<String> listTables(String catalogName, String databaseName) {
        Catalog currentCatalog = this.catalogs.get(this.getCurrentCatalog());
        try {
            return Stream.concat(currentCatalog.listTables(this.getCurrentDatabase()).stream(), this.listTemporaryTablesInternal(catalogName, databaseName).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName())).collect(Collectors.toSet());
        }
        catch (DatabaseNotExistException e2) {
            throw new ValidationException("Current database does not exist", e2);
        }
    }

    public Set<String> listTemporaryTables() {
        return this.listTemporaryTablesInternal(this.getCurrentCatalog(), this.getCurrentDatabase()).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName()).collect(Collectors.toSet());
    }

    public Set<String> listTemporaryViews() {
        return this.listTemporaryTablesInternal(this.getCurrentCatalog(), this.getCurrentDatabase()).filter(e -> e.getValue() instanceof CatalogView).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName()).collect(Collectors.toSet());
    }

    private Stream<Map.Entry<ObjectIdentifier, CatalogBaseTable>> listTemporaryTablesInternal(String catalogName, String databaseName) {
        return this.temporaryTables.entrySet().stream().filter(e -> {
            ObjectIdentifier identifier = (ObjectIdentifier)e.getKey();
            return identifier.getCatalogName().equals(catalogName) && identifier.getDatabaseName().equals(databaseName);
        });
    }

    public Set<String> listSchemas() {
        return Stream.concat(this.catalogs.keySet().stream(), this.temporaryTables.keySet().stream().map(ObjectIdentifier::getCatalogName)).collect(Collectors.toSet());
    }

    public Set<String> listSchemas(String catalogName) {
        return Stream.concat(Optional.ofNullable(this.catalogs.get(catalogName)).map(Catalog::listDatabases).orElse(Collections.emptyList()).stream(), this.temporaryTables.keySet().stream().filter(i -> i.getCatalogName().equals(catalogName)).map(ObjectIdentifier::getDatabaseName)).collect(Collectors.toSet());
    }

    public boolean schemaExists(String catalogName) {
        return this.getCatalog(catalogName).isPresent() || this.temporaryTables.keySet().stream().anyMatch(i -> i.getCatalogName().equals(catalogName));
    }

    public boolean schemaExists(String catalogName, String databaseName) {
        return this.temporaryDatabaseExists(catalogName, databaseName) || this.permanentDatabaseExists(catalogName, databaseName);
    }

    private boolean temporaryDatabaseExists(String catalogName, String databaseName) {
        return this.temporaryTables.keySet().stream().anyMatch(i -> i.getCatalogName().equals(catalogName) && i.getDatabaseName().equals(databaseName));
    }

    private boolean permanentDatabaseExists(String catalogName, String databaseName) {
        return this.getCatalog(catalogName).map(c -> c.databaseExists(databaseName)).orElse(false);
    }

    public ObjectIdentifier qualifyIdentifier(UnresolvedIdentifier identifier) {
        return ObjectIdentifier.of(identifier.getCatalogName().orElseGet(this::getCurrentCatalog), identifier.getDatabaseName().orElseGet(this::getCurrentDatabase), identifier.getObjectName());
    }

    public void createTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        this.execute((catalog, path) -> catalog.createTable(path, table, ignoreIfExists), objectIdentifier, false, "CreateTable");
    }

    public void createTemporaryTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean replace) {
        this.temporaryTables.compute(objectIdentifier, (k, v) -> {
            if (v != null && !replace) {
                throw new ValidationException(String.format("Temporary table %s already exists", objectIdentifier));
            }
            return table;
        });
    }

    public boolean dropTemporaryTable(UnresolvedIdentifier identifier) {
        return this.dropTemporaryTableInternal(identifier, table -> table instanceof CatalogTable);
    }

    public boolean dropTemporaryView(UnresolvedIdentifier identifier) {
        return this.dropTemporaryTableInternal(identifier, table -> table instanceof CatalogView);
    }

    private boolean dropTemporaryTableInternal(UnresolvedIdentifier unresolvedIdentifier, Predicate<CatalogBaseTable> filter) {
        ObjectIdentifier objectIdentifier = this.qualifyIdentifier(unresolvedIdentifier);
        CatalogBaseTable catalogBaseTable = this.temporaryTables.get(objectIdentifier);
        if (filter.test(catalogBaseTable)) {
            this.temporaryTables.remove(objectIdentifier);
            return true;
        }
        return false;
    }

    public void alterTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.execute((catalog, path) -> catalog.alterTable(path, table, ignoreIfNotExists), objectIdentifier, ignoreIfNotExists, "AlterTable");
    }

    public void dropTable(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        if (this.temporaryTables.containsKey(objectIdentifier)) {
            throw new ValidationException(String.format("Temporary table with identifier '%s' exists. Drop it first before removing the permanent table.", objectIdentifier));
        }
        this.execute((catalog, path) -> catalog.dropTable(path, ignoreIfNotExists), objectIdentifier, ignoreIfNotExists, "DropTable");
    }

    private void execute(ModifyCatalog command, ObjectIdentifier objectIdentifier, boolean ignoreNoCatalog, String commandName) {
        Optional<Catalog> catalog = this.getCatalog(objectIdentifier.getCatalogName());
        if (catalog.isPresent()) {
            try {
                command.execute(catalog.get(), objectIdentifier.toObjectPath());
            }
            catch (DatabaseNotExistException | TableAlreadyExistException | TableNotExistException e) {
                throw new ValidationException(this.getErrorMessage(objectIdentifier, commandName), e);
            }
            catch (Exception e) {
                throw new TableException(this.getErrorMessage(objectIdentifier, commandName), e);
            }
        } else if (!ignoreNoCatalog) {
            throw new ValidationException(String.format("Catalog %s does not exist.", objectIdentifier.getCatalogName()));
        }
    }

    private String getErrorMessage(ObjectIdentifier objectIdentifier, String commandName) {
        return String.format("Could not execute %s in path %s", commandName, objectIdentifier);
    }

    private static interface ModifyCatalog {
        public void execute(Catalog var1, ObjectPath var2) throws Exception;
    }

    public static class TableLookupResult {
        private final boolean isTemporary;
        private final CatalogBaseTable table;

        private static TableLookupResult temporary(CatalogBaseTable table) {
            return new TableLookupResult(true, table);
        }

        private static TableLookupResult permanent(CatalogBaseTable table) {
            return new TableLookupResult(false, table);
        }

        private TableLookupResult(boolean isTemporary, CatalogBaseTable table) {
            this.isTemporary = isTemporary;
            this.table = table;
        }

        public boolean isTemporary() {
            return this.isTemporary;
        }

        public CatalogBaseTable getTable() {
            return this.table;
        }
    }
}

