/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.concurrent.Executor;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.state.StateHandle;
import org.apache.flink.runtime.zookeeper.StateStorageHelper;
import org.apache.flink.runtime.zookeeper.ZooKeeperStateHandleStore;
import org.apache.flink.shaded.org.apache.curator.framework.CuratorFramework;
import org.apache.flink.shaded.org.apache.curator.framework.api.BackgroundCallback;
import org.apache.flink.shaded.org.apache.curator.framework.api.CuratorEvent;
import org.apache.flink.shaded.org.apache.curator.framework.api.CuratorEventType;
import org.apache.flink.shaded.org.apache.curator.utils.ZKPaths;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooKeeperCompletedCheckpointStore
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(ZooKeeperCompletedCheckpointStore.class);
    private final CuratorFramework client;
    private final ZooKeeperStateHandleStore<CompletedCheckpoint> checkpointsInZooKeeper;
    private final int maxNumberOfCheckpointsToRetain;
    private final ClassLoader userClassLoader;
    private final ArrayDeque<Tuple2<StateHandle<CompletedCheckpoint>, String>> checkpointStateHandles;

    public ZooKeeperCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, ClassLoader userClassLoader, CuratorFramework client, String checkpointsPath, StateStorageHelper<CompletedCheckpoint> stateStorage, Executor executor) throws Exception {
        Preconditions.checkArgument((maxNumberOfCheckpointsToRetain >= 1 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint.");
        Preconditions.checkNotNull(stateStorage, (String)"State storage");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        this.userClassLoader = (ClassLoader)Preconditions.checkNotNull((Object)userClassLoader, (String)"User class loader");
        Preconditions.checkNotNull((Object)client, (String)"Curator client");
        Preconditions.checkNotNull((Object)checkpointsPath, (String)"Checkpoints path");
        client.newNamespaceAwareEnsurePath(checkpointsPath).ensure(client.getZookeeperClient());
        this.client = client.usingNamespace(client.getNamespace() + checkpointsPath);
        this.checkpointsInZooKeeper = new ZooKeeperStateHandleStore(this.client, stateStorage, executor);
        this.checkpointStateHandles = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        LOG.info("Initialized in '{}'.", (Object)checkpointsPath);
    }

    @Override
    public void recover() throws Exception {
        List<Tuple2<StateHandle<CompletedCheckpoint>, String>> initialCheckpoints;
        LOG.info("Recovering checkpoints from ZooKeeper.");
        this.checkpointStateHandles.clear();
        while (true) {
            try {
                initialCheckpoints = this.checkpointsInZooKeeper.getAllSortedByName();
            }
            catch (ConcurrentModificationException e) {
                LOG.warn("Concurrent modification while reading from ZooKeeper. Retrying.");
                continue;
            }
            break;
        }
        int numberOfInitialCheckpoints = initialCheckpoints.size();
        LOG.info("Found {} checkpoints in ZooKeeper.", (Object)numberOfInitialCheckpoints);
        if (numberOfInitialCheckpoints > 0) {
            CompletedCheckpoint latestCheckpoint;
            Tuple2<StateHandle<CompletedCheckpoint>, String> latest = initialCheckpoints.get(numberOfInitialCheckpoints - 1);
            long checkpointId = ZooKeeperCompletedCheckpointStore.pathToCheckpointId((String)latest.f1);
            LOG.info("Trying to retrieve checkpoint {}.", (Object)checkpointId);
            try {
                latestCheckpoint = (CompletedCheckpoint)((StateHandle)latest.f0).getState(this.userClassLoader);
            }
            catch (Exception e) {
                throw new Exception("Could not retrieve the completed checkpoint " + checkpointId + " from the state storage.", e);
            }
            this.checkpointStateHandles.add(latest);
            LOG.info("Initialized with {}. Removing all older checkpoints.", (Object)latestCheckpoint);
            for (int i = 0; i < numberOfInitialCheckpoints - 1; ++i) {
                try {
                    this.removeFromZooKeeperAndDiscardCheckpoint(initialCheckpoints.get(i));
                    continue;
                }
                catch (Exception e) {
                    LOG.error("Failed to discard checkpoint", (Throwable)e);
                }
            }
        }
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint) throws Exception {
        Preconditions.checkNotNull((Object)checkpoint, (String)"Checkpoint");
        String path = ZooKeeperCompletedCheckpointStore.checkpointIdToPath(checkpoint.getCheckpointID());
        StateHandle<CompletedCheckpoint> stateHandle = this.checkpointsInZooKeeper.add(path, checkpoint);
        this.checkpointStateHandles.addLast((Tuple2<StateHandle<CompletedCheckpoint>, String>)new Tuple2(stateHandle, (Object)path));
        if (this.checkpointStateHandles.size() > this.maxNumberOfCheckpointsToRetain) {
            this.removeFromZooKeeperAndDiscardCheckpoint(this.checkpointStateHandles.removeFirst());
        }
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    @Override
    public CompletedCheckpoint getLatestCheckpoint() throws Exception {
        if (this.checkpointStateHandles.isEmpty()) {
            return null;
        }
        return (CompletedCheckpoint)((StateHandle)this.checkpointStateHandles.getLast().f0).getState(this.userClassLoader);
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() throws Exception {
        ArrayList<CompletedCheckpoint> checkpoints = new ArrayList<CompletedCheckpoint>(this.checkpointStateHandles.size());
        for (Tuple2<StateHandle<CompletedCheckpoint>, String> stateHandle : this.checkpointStateHandles) {
            checkpoints.add((CompletedCheckpoint)((StateHandle)stateHandle.f0).getState(this.userClassLoader));
        }
        return checkpoints;
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.checkpointStateHandles.size();
    }

    @Override
    public void shutdown() throws Exception {
        LOG.info("Shutting down");
        for (Tuple2<StateHandle<CompletedCheckpoint>, String> checkpoint : this.checkpointStateHandles) {
            try {
                this.removeFromZooKeeperAndDiscardCheckpoint(checkpoint);
            }
            catch (Exception e) {
                LOG.error("Failed to discard checkpoint.", (Throwable)e);
            }
        }
        this.checkpointStateHandles.clear();
        String path = "/" + this.client.getNamespace();
        LOG.info("Removing {} from ZooKeeper", (Object)path);
        ZKPaths.deleteChildren(this.client.getZookeeperClient().getZooKeeper(), path, true);
    }

    @Override
    public void suspend() throws Exception {
        LOG.info("Suspending");
        this.checkpointStateHandles.clear();
    }

    private void removeFromZooKeeperAndDiscardCheckpoint(final Tuple2<StateHandle<CompletedCheckpoint>, String> stateHandleAndPath) throws Exception {
        BackgroundCallback callback = new BackgroundCallback(){

            /*
             * Enabled force condition propagation
             * Lifted jumps to return sites
             */
            @Override
            public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
                long checkpointId = ZooKeeperCompletedCheckpointStore.pathToCheckpointId((String)stateHandleAndPath.f1);
                try {
                    Exception newException;
                    if (event.getType() != CuratorEventType.DELETE) throw new IllegalStateException("Unexpected event type " + (Object)((Object)event.getType()) + " in '" + event + "' callback.");
                    if (event.getResultCode() != 0) throw new IllegalStateException("Unexpected result code " + event.getResultCode() + " in '" + event + "' callback.");
                    Exception exception = null;
                    CompletedCheckpoint checkpoint = null;
                    try {
                        checkpoint = (CompletedCheckpoint)((StateHandle)stateHandleAndPath.f0).getState(ZooKeeperCompletedCheckpointStore.this.userClassLoader);
                    }
                    catch (Exception e) {
                        newException = new Exception("Could not retrieve the completed checkpoint " + checkpointId + " from the state storage.", e);
                        exception = (Exception)ExceptionUtils.firstOrSuppressed((Throwable)newException, exception);
                    }
                    if (checkpoint != null) {
                        try {
                            checkpoint.discard(ZooKeeperCompletedCheckpointStore.this.userClassLoader);
                        }
                        catch (Exception e) {
                            newException = new Exception("Could not discard the completed checkpoint " + checkpoint + '.', e);
                            exception = (Exception)ExceptionUtils.firstOrSuppressed((Throwable)newException, (Throwable)exception);
                        }
                    }
                    try {
                        ((StateHandle)stateHandleAndPath.f0).discardState();
                    }
                    catch (Exception e) {
                        newException = new Exception("Could not discard meta data of completed checkpoint " + checkpointId + '.', e);
                        exception = (Exception)ExceptionUtils.firstOrSuppressed((Throwable)newException, (Throwable)exception);
                    }
                    if (exception != null) {
                        throw exception;
                    }
                    LOG.debug("Discarded {}.", (Object)checkpoint);
                    return;
                }
                catch (Exception e) {
                    LOG.warn("Failed to discard checkpoint {}.", (Object)checkpointId, (Object)e);
                }
            }
        };
        this.checkpointsInZooKeeper.remove((String)stateHandleAndPath.f1, callback);
    }

    protected static String checkpointIdToPath(long checkpointId) {
        return String.format("/%s", checkpointId);
    }

    protected static long pathToCheckpointId(String path) {
        try {
            String numberString = '/' == path.charAt(0) ? path.substring(1) : path;
            return Long.parseLong(numberString);
        }
        catch (NumberFormatException e) {
            LOG.warn("Could not parse checkpoint id from {}. This indicates that the checkpoint id to path conversion has changed.", (Object)path);
            return -1L;
        }
    }
}

