/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import org.apache.flink.core.fs.AutoCloseableRegistry;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.testutils.CheckedThread;
import org.apache.flink.runtime.execution.CancelTaskException;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferBuilder;
import org.apache.flink.runtime.io.network.buffer.BufferListener;
import org.apache.flink.runtime.io.network.buffer.BufferPool;
import org.apache.flink.runtime.io.network.buffer.BufferProvider;
import org.apache.flink.runtime.io.network.buffer.LocalBufferPool;
import org.apache.flink.runtime.io.network.buffer.NetworkBufferPool;
import org.apache.flink.runtime.io.network.buffer.TestingBufferListener;
import org.apache.flink.testutils.executor.TestExecutorExtension;
import org.apache.flink.util.Preconditions;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;
import org.junit.jupiter.api.extension.RegisterExtension;

class LocalBufferPoolTest {
    private static final int numBuffers = 1024;
    private static final int memorySegmentSize = 128;
    private NetworkBufferPool networkBufferPool;
    private BufferPool localBufferPool;
    @RegisterExtension
    private static final TestExecutorExtension<ExecutorService> EXECUTOR_EXTENSION = new TestExecutorExtension(Executors::newCachedThreadPool);

    LocalBufferPoolTest() {
    }

    @BeforeEach
    void setupLocalBufferPool() {
        this.networkBufferPool = new NetworkBufferPool(1024, 128);
        this.localBufferPool = new LocalBufferPool(this.networkBufferPool, 1);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
    }

    @AfterEach
    void destroyAndVerifyAllBuffersReturned() {
        if (!this.localBufferPool.isDestroyed()) {
            this.localBufferPool.lazyDestroy();
        }
        ((AbstractIntegerAssert)Assertions.assertThat((int)this.networkBufferPool.getNumberOfAvailableMemorySegments()).withFailMessage("Did not return all buffers to memory segment pool after test.", new Object[0])).isEqualTo(1024);
        this.networkBufferPool.destroyAllBufferPools();
        this.networkBufferPool.destroy();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testReserveSegments() throws Exception {
        NetworkBufferPool networkBufferPool = new NetworkBufferPool(2, 128, Duration.ofSeconds(2L));
        try {
            BufferPool bufferPool1 = networkBufferPool.createBufferPool(1, 2);
            Assertions.assertThatThrownBy(() -> bufferPool1.reserveSegments(2)).isInstanceOf(IllegalArgumentException.class);
            ArrayList<Buffer> buffers = new ArrayList<Buffer>(2);
            buffers.add(bufferPool1.requestBuffer());
            buffers.add(bufferPool1.requestBuffer());
            Assertions.assertThat(buffers).hasSize(2);
            BufferPool bufferPool2 = networkBufferPool.createBufferPool(1, 10);
            Assertions.assertThatThrownBy(() -> bufferPool2.reserveSegments(1)).isInstanceOf(IOException.class);
            Assertions.assertThat((boolean)bufferPool2.isAvailable()).isFalse();
            buffers.forEach(Buffer::recycleBuffer);
            bufferPool1.lazyDestroy();
            bufferPool2.lazyDestroy();
            BufferPool bufferPool3 = networkBufferPool.createBufferPool(2, 10);
            Assertions.assertThat((int)bufferPool3.getNumberOfAvailableMemorySegments()).isOne();
            bufferPool3.reserveSegments(2);
            Assertions.assertThat((int)bufferPool3.getNumberOfAvailableMemorySegments()).isEqualTo(2);
            bufferPool3.lazyDestroy();
            Assertions.assertThatThrownBy(() -> bufferPool3.reserveSegments(1)).isInstanceOf(CancelTaskException.class);
        }
        finally {
            networkBufferPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    @Timeout(value=10L)
    void testReserveSegmentsAndCancel() throws Exception {
        int totalSegments = 4;
        int segmentsToReserve = 2;
        NetworkBufferPool globalPool = new NetworkBufferPool(totalSegments, 128);
        BufferPool localPool1 = globalPool.createBufferPool(segmentsToReserve, totalSegments);
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>();
        try {
            for (int i = 0; i < totalSegments; ++i) {
                segments.add(localPool1.requestMemorySegmentBlocking());
            }
            BufferPool localPool2 = globalPool.createBufferPool(segmentsToReserve, totalSegments);
            Thread reserveThread = new Thread(() -> {
                try {
                    localPool2.reserveSegments(segmentsToReserve);
                }
                catch (Throwable throwable) {
                    // empty catch block
                }
            });
            reserveThread.start();
            Thread.sleep(100L);
            Thread cancelThread = new Thread(() -> {
                localPool1.lazyDestroy();
                localPool2.lazyDestroy();
            });
            cancelThread.start();
            Thread interruptThread = new Thread(() -> {
                try {
                    do {
                        reserveThread.interrupt();
                        Thread.sleep(100L);
                    } while (reserveThread.isAlive() || cancelThread.isAlive());
                }
                catch (Throwable throwable) {
                    // empty catch block
                }
            });
            interruptThread.start();
            interruptThread.join();
        }
        finally {
            segments.forEach(arg_0 -> ((BufferPool)localPool1).recycle(arg_0));
            localPool1.lazyDestroy();
            Assertions.assertThat((int)globalPool.getNumberOfUsedMemorySegments()).isZero();
            globalPool.destroy();
        }
    }

    @Test
    void testRequestMoreThanAvailable() {
        this.localBufferPool.setNumBuffers(1024);
        ArrayList<Buffer> requests = new ArrayList<Buffer>(1024);
        for (int i = 1; i <= 1024; ++i) {
            Buffer buffer2 = this.localBufferPool.requestBuffer();
            Assertions.assertThat((int)this.getNumRequestedFromMemorySegmentPool()).isEqualTo(Math.min(i + 1, 1024));
            Assertions.assertThat((Object)buffer2).isNotNull();
            requests.add(buffer2);
        }
        Buffer buffer = this.localBufferPool.requestBuffer();
        Assertions.assertThat((int)this.getNumRequestedFromMemorySegmentPool()).isEqualTo(1024);
        Assertions.assertThat((Object)buffer).isNull();
        for (Buffer buffer2 : requests) {
            buffer2.recycleBuffer();
        }
    }

    @Test
    void testSetNumAfterDestroyDoesNotProactivelyFetchSegments() {
        this.localBufferPool.setNumBuffers(2);
        Assertions.assertThat((int)this.localBufferPool.getNumBuffers()).isEqualTo(2);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        this.localBufferPool.lazyDestroy();
        this.localBufferPool.setNumBuffers(3);
        Assertions.assertThat((int)this.localBufferPool.getNumBuffers()).isEqualTo(3);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
    }

    @Test
    void testRecycleAfterDestroy() {
        this.localBufferPool.setNumBuffers(1024);
        ArrayList<Buffer> requests = new ArrayList<Buffer>(1024);
        for (int i = 0; i < 1024; ++i) {
            requests.add(this.localBufferPool.requestBuffer());
        }
        this.localBufferPool.lazyDestroy();
        Assertions.assertThat((int)this.getNumRequestedFromMemorySegmentPool()).isEqualTo(1024);
        for (Buffer buffer : requests) {
            buffer.recycleBuffer();
        }
    }

    @Test
    void testDecreasePoolSize() throws Exception {
        int maxMemorySegments = 10;
        int requiredMemorySegments = 4;
        this.testDecreasePoolSizeInternal(10, 4, 7, 5, 2, 5, 0, 5, 0);
        this.testDecreasePoolSizeInternal(10, 4, 6, 4, 2, 2, 0, 3, 1);
        this.testDecreasePoolSizeInternal(10, 4, 7, 5, 2, 7, 2, 5, 0);
        this.testDecreasePoolSizeInternal(10, 4, 9, 5, 3, 9, 4, 5, 0);
        this.testDecreasePoolSizeInternal(10, 4, 7, 5, 4, 7, 2, 5, 0);
        this.testDecreasePoolSizeInternal(10, 4, 7, 5, 6, 9, 4, 5, 0);
    }

    void testDecreasePoolSizeInternal(int maxMemorySegments, int requiredMemorySegments, int largePoolSize, int smallPoolSize, int maxOverdraftBuffers, int numBuffersToRequest, int numRequestedOverdraftBuffersAfterDecreasing, int numRequestedOrdinaryBuffersAfterDecreasing, int numAvailableBuffersAfterDecreasing) throws Exception {
        LocalBufferPool bufferPool = new LocalBufferPool(this.networkBufferPool, requiredMemorySegments, maxMemorySegments, 0, Integer.MAX_VALUE, maxOverdraftBuffers);
        ArrayList<MemorySegment> buffers = new ArrayList<MemorySegment>();
        bufferPool.setNumBuffers(largePoolSize);
        Assertions.assertThat((int)bufferPool.getNumBuffers()).isEqualTo(largePoolSize);
        for (int i = 0; i < numBuffersToRequest; ++i) {
            buffers.add(bufferPool.requestMemorySegmentBlocking());
        }
        bufferPool.setNumBuffers(smallPoolSize);
        Assertions.assertThat((int)bufferPool.getNumBuffers()).isEqualTo(smallPoolSize);
        Assertions.assertThat((int)LocalBufferPoolTest.getNumberRequestedOverdraftBuffers(bufferPool)).isEqualTo(numRequestedOverdraftBuffersAfterDecreasing);
        Assertions.assertThat((int)LocalBufferPoolTest.getNumberRequestedOrdinaryBuffers(bufferPool)).isEqualTo(numRequestedOrdinaryBuffersAfterDecreasing);
        Assertions.assertThat((int)bufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(numAvailableBuffersAfterDecreasing);
        Assertions.assertThat((boolean)bufferPool.isAvailable()).isEqualTo(numAvailableBuffersAfterDecreasing > 0);
        for (MemorySegment buffer : buffers) {
            bufferPool.recycle(buffer);
        }
        bufferPool.lazyDestroy();
    }

    @Test
    void testIncreasePoolSize() throws Exception {
        int maxMemorySegments = 100;
        int requiredMemorySegments = 5;
        int smallPoolSize = 5;
        int maxOverdraftBuffers = 2;
        this.testIncreasePoolSizeInternal(100, 5, 5, 6, 2, 1, 0, false);
        this.testIncreasePoolSizeInternal(100, 5, 5, 7, 2, 0, 0, false);
        this.testIncreasePoolSizeInternal(100, 5, 5, 8, 2, 0, 1, true);
        this.testIncreasePoolSizeInternal(10, 8, 8, 10, 2, 0, 0, false);
    }

    void testIncreasePoolSizeInternal(int maxMemorySegments, int requiredMemorySegments, int smallPoolSize, int largePoolSize, int maxOverdraftBuffers, int numOverdraftBuffersAfterIncreasePoolSize, int numAvailableBuffersAfterIncreasePoolSize, boolean isAvailableAfterIncreasePoolSize) throws Exception {
        int i;
        LocalBufferPool bufferPool = new LocalBufferPool(this.networkBufferPool, requiredMemorySegments, maxMemorySegments, 0, Integer.MAX_VALUE, maxOverdraftBuffers);
        ArrayList<MemorySegment> buffers = new ArrayList<MemorySegment>();
        bufferPool.setNumBuffers(smallPoolSize);
        Assertions.assertThat((int)bufferPool.getNumBuffers()).isEqualTo(smallPoolSize);
        for (i = 0; i < smallPoolSize; ++i) {
            buffers.add(bufferPool.requestMemorySegmentBlocking());
        }
        Assertions.assertThat((boolean)bufferPool.isAvailable()).isFalse();
        for (i = 0; i < maxOverdraftBuffers; ++i) {
            buffers.add(bufferPool.requestMemorySegmentBlocking());
        }
        Assertions.assertThat((Object)bufferPool.requestMemorySegment()).isNull();
        Assertions.assertThat((int)LocalBufferPoolTest.getNumberRequestedOverdraftBuffers(bufferPool)).isEqualTo(maxOverdraftBuffers);
        Assertions.assertThat((boolean)bufferPool.isAvailable()).isFalse();
        bufferPool.setNumBuffers(largePoolSize);
        Assertions.assertThat((int)bufferPool.getNumBuffers()).isEqualTo(largePoolSize);
        Assertions.assertThat((int)bufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(numAvailableBuffersAfterIncreasePoolSize);
        Assertions.assertThat((int)LocalBufferPoolTest.getNumberRequestedOverdraftBuffers(bufferPool)).isEqualTo(numOverdraftBuffersAfterIncreasePoolSize);
        Assertions.assertThat((boolean)bufferPool.isAvailable()).isEqualTo(isAvailableAfterIncreasePoolSize);
        for (MemorySegment buffer : buffers) {
            bufferPool.recycle(buffer);
        }
        bufferPool.lazyDestroy();
    }

    @Test
    @Timeout(value=30L)
    void testRequestBufferOnRecycleWithOverdraft() throws Exception {
        this.testRequestBuffersOnRecycle(true);
    }

    @Test
    @Timeout(value=30L)
    void testRequestBufferOnRecycleWithoutOverdraft() throws Exception {
        this.testRequestBuffersOnRecycle(false);
    }

    private void testRequestBuffersOnRecycle(boolean supportOverdraftBuffer) throws Exception {
        BufferPool bufferPool1 = this.networkBufferPool.createBufferPool(512, 2048, 0, Integer.MAX_VALUE, supportOverdraftBuffer ? 5 : 0);
        ArrayList<MemorySegment> segments = new ArrayList<MemorySegment>();
        for (int i = 0; i < 1023; ++i) {
            segments.add(bufferPool1.requestMemorySegmentBlocking());
        }
        final BufferPool bufferPool2 = this.networkBufferPool.createBufferPool(512, 512, 0, Integer.MAX_VALUE, supportOverdraftBuffer ? 5 : 0);
        final ArrayList segments2 = new ArrayList();
        CheckedThread checkedThread = new CheckedThread(){

            public void go() throws Exception {
                for (int i = 0; i < 512; ++i) {
                    segments2.add(bufferPool2.requestMemorySegmentBlocking());
                }
            }
        };
        checkedThread.start();
        for (MemorySegment segment : segments) {
            bufferPool1.recycle(segment);
        }
        bufferPool1.lazyDestroy();
        checkedThread.sync();
        for (MemorySegment segment : segments2) {
            bufferPool2.recycle(segment);
        }
        bufferPool2.lazyDestroy();
    }

    @Test
    void testRecycleExcessBuffersAfterRecycling() {
        int i;
        this.localBufferPool.setNumBuffers(1024);
        ArrayList<Buffer> requests = new ArrayList<Buffer>(1024);
        for (i = 1; i <= 1024; ++i) {
            requests.add(this.localBufferPool.requestBuffer());
        }
        Assertions.assertThat((int)this.getNumRequestedFromMemorySegmentPool()).isEqualTo(1024);
        this.localBufferPool.setNumBuffers(512);
        Assertions.assertThat((int)this.getNumRequestedFromMemorySegmentPool()).isEqualTo(1024);
        for (i = 1; i < 512; ++i) {
            ((Buffer)requests.remove(0)).recycleBuffer();
            Assertions.assertThat((int)this.getNumRequestedFromMemorySegmentPool()).isEqualTo(1024 - i);
        }
        for (Buffer buffer : requests) {
            buffer.recycleBuffer();
        }
    }

    @Test
    void testRecycleExcessBuffersAfterChangingNumBuffers() {
        this.localBufferPool.setNumBuffers(1024);
        ArrayList<Buffer> requests = new ArrayList<Buffer>(1024);
        for (int i = 1; i <= 1024; ++i) {
            requests.add(this.localBufferPool.requestBuffer());
        }
        for (Buffer buffer : requests) {
            buffer.recycleBuffer();
        }
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(1024);
        this.localBufferPool.setNumBuffers(512);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(512);
    }

    @Test
    void testSetLessThanRequiredNumBuffers() {
        this.localBufferPool.setNumBuffers(1);
        Assertions.assertThatThrownBy(() -> this.localBufferPool.setNumBuffers(0)).isInstanceOf(IllegalArgumentException.class);
    }

    @Test
    void testPendingRequestWithListenersAfterRecycle() {
        CountBufferListener listener1 = new CountBufferListener();
        CountBufferListener listener2 = new CountBufferListener();
        Buffer available = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)this.localBufferPool.requestBuffer()).isNull();
        Assertions.assertThat((boolean)this.localBufferPool.addBufferListener((BufferListener)listener1)).isTrue();
        Assertions.assertThat((boolean)this.localBufferPool.addBufferListener((BufferListener)listener2)).isTrue();
        ((Buffer)Preconditions.checkNotNull((Object)available)).recycleBuffer();
        Assertions.assertThat((int)listener1.getCount()).isOne();
        Assertions.assertThat((int)listener1.getCount()).isOne();
        Assertions.assertThat((boolean)this.localBufferPool.addBufferListener((BufferListener)listener1)).isFalse();
        Assertions.assertThat((boolean)this.localBufferPool.addBufferListener((BufferListener)listener2)).isFalse();
    }

    @Test
    void testCancelPendingRequestsAfterDestroy() {
        AtomicInteger invokeNotifyBufferDestroyedCounter = new AtomicInteger(0);
        TestingBufferListener listener = TestingBufferListener.builder().setNotifyBufferDestroyedRunnable(invokeNotifyBufferDestroyedCounter::incrementAndGet).build();
        this.localBufferPool.setNumBuffers(1);
        Buffer available = this.localBufferPool.requestBuffer();
        Buffer unavailable = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)available).isNotNull();
        Assertions.assertThat((Object)unavailable).isNull();
        this.localBufferPool.addBufferListener((BufferListener)listener);
        this.localBufferPool.lazyDestroy();
        available.recycleBuffer();
        Assertions.assertThat((AtomicInteger)invokeNotifyBufferDestroyedCounter).hasValue(1);
    }

    @Test
    void testConcurrentRequestRecycle() throws ExecutionException, InterruptedException {
        int i;
        int numConcurrentTasks = 128;
        int numBuffersToRequestPerTask = 1024;
        this.localBufferPool.setNumBuffers(numConcurrentTasks);
        Future[] taskResults = new Future[numConcurrentTasks];
        for (i = 0; i < numConcurrentTasks; ++i) {
            taskResults[i] = EXECUTOR_EXTENSION.getExecutor().submit(new BufferRequesterTask((BufferProvider)this.localBufferPool, numBuffersToRequestPerTask));
        }
        for (i = 0; i < numConcurrentTasks; ++i) {
            Assertions.assertThat((Boolean)((Boolean)taskResults[i].get())).isTrue();
        }
    }

    @Test
    void testBoundedBuffer() {
        this.localBufferPool.lazyDestroy();
        this.localBufferPool = new LocalBufferPool(this.networkBufferPool, 1, 2);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        Assertions.assertThat((int)this.localBufferPool.getMaxNumberOfMemorySegments()).isEqualTo(2);
        this.localBufferPool.setNumBuffers(1);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        Buffer buffer1 = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)buffer1).isNotNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        Assertions.assertThat((Object)this.localBufferPool.requestBuffer()).isNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        buffer1.recycleBuffer();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        this.localBufferPool.setNumBuffers(2);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        buffer1 = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)buffer1).isNotNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        Buffer buffer2 = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)buffer2).isNotNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        Assertions.assertThat((Object)this.localBufferPool.requestBuffer()).isNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        buffer1.recycleBuffer();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        buffer2.recycleBuffer();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(2);
        this.localBufferPool.setNumBuffers(3);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(2);
        buffer1 = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)buffer1).isNotNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        buffer2 = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)buffer2).isNotNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        Assertions.assertThat((Object)this.localBufferPool.requestBuffer()).isNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        buffer1.recycleBuffer();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        buffer2.recycleBuffer();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isEqualTo(2);
        this.localBufferPool.setNumBuffers(1);
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
        buffer1 = this.localBufferPool.requestBuffer();
        Assertions.assertThat((Object)buffer1).isNotNull();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isZero();
        Assertions.assertThat((Object)this.localBufferPool.requestBuffer()).isNull();
        buffer1.recycleBuffer();
        Assertions.assertThat((int)this.localBufferPool.getNumberOfAvailableMemorySegments()).isOne();
    }

    @Test
    void testMaxBuffersPerChannelAndAvailability() throws Exception {
        this.localBufferPool.lazyDestroy();
        this.localBufferPool = new LocalBufferPool(this.networkBufferPool, 1, Integer.MAX_VALUE, 3, 2, 0);
        this.localBufferPool.setNumBuffers(10);
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isDone();
        BufferBuilder bufferBuilder01 = this.localBufferPool.requestBufferBuilderBlocking(0);
        BufferBuilder bufferBuilder11 = this.localBufferPool.requestBufferBuilderBlocking(1);
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isDone();
        BufferBuilder bufferBuilder02 = this.localBufferPool.requestBufferBuilderBlocking(0);
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isNotDone();
        BufferBuilder bufferBuilder03 = this.localBufferPool.requestBufferBuilderBlocking(0);
        BufferBuilder bufferBuilder21 = this.localBufferPool.requestBufferBuilderBlocking(2);
        BufferBuilder bufferBuilder22 = this.localBufferPool.requestBufferBuilderBlocking(2);
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isNotDone();
        bufferBuilder11.close();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isNotDone();
        bufferBuilder21.close();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isNotDone();
        bufferBuilder02.close();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isNotDone();
        bufferBuilder01.close();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isDone();
        bufferBuilder03.close();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isDone();
        bufferBuilder22.close();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isDone();
    }

    @Test
    void testIsAvailableOrNot() throws InterruptedException {
        Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
        try (BufferBuilder bufferBuilder = (BufferBuilder)Preconditions.checkNotNull((Object)this.localBufferPool.requestBufferBuilderBlocking());){
            CompletableFuture availableFuture = this.localBufferPool.getAvailableFuture();
            Assertions.assertThat((CompletableFuture)availableFuture).isNotDone();
            int numLocalBuffers = 5;
            this.localBufferPool.setNumBuffers(5);
            Assertions.assertThat((CompletableFuture)availableFuture).isDone();
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
            ArrayDeque<Object> buffers = new ArrayDeque<Object>(1024);
            for (int i = 0; i < 4; ++i) {
                Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
                buffers.add(Preconditions.checkNotNull((Object)this.localBufferPool.requestBuffer()));
            }
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isFalse();
            ((Buffer)buffers.pop()).recycleBuffer();
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
            for (Buffer buffer : buffers) {
                buffer.recycleBuffer();
            }
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
            this.localBufferPool.setNumBuffers(2);
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
            Buffer buffer2 = (Buffer)Preconditions.checkNotNull((Object)this.localBufferPool.requestBuffer());
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isFalse();
            buffer2.recycleBuffer();
            Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
            this.localBufferPool.setNumBuffers(1);
            Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isNotDone();
        }
        Assertions.assertThat((boolean)this.localBufferPool.isAvailable()).isTrue();
        Assertions.assertThat((CompletableFuture)this.localBufferPool.getAvailableFuture()).isDone();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testConsistentAvailability() throws Exception {
        TestNetworkBufferPool globalPool = new TestNetworkBufferPool(1024, 128);
        try {
            LocalBufferPool localPool = new LocalBufferPool((NetworkBufferPool)globalPool, 1);
            MemorySegment segment = localPool.requestMemorySegmentBlocking();
            localPool.setNumBuffers(2);
            localPool.recycle(segment);
            localPool.lazyDestroy();
        }
        finally {
            globalPool.destroy();
        }
    }

    @Test
    void testOverdraftBufferAndAvailability() throws Exception {
        for (int maxOverdraftBuffers = 0; maxOverdraftBuffers < 3; ++maxOverdraftBuffers) {
            this.useAllOverdraftBuffersAndCheckIsLegal(4, 3, maxOverdraftBuffers, 2, 1);
            this.useAllOverdraftBuffersAndCheckIsLegal(4, 3, maxOverdraftBuffers, 2, 2);
            this.useAllOverdraftBuffersAndCheckIsLegal(4, 3, maxOverdraftBuffers, 3, 2);
            this.useAllOverdraftBuffersAndCheckIsLegal(8, 5, maxOverdraftBuffers, 2, 1);
            this.useAllOverdraftBuffersAndCheckIsLegal(8, 5, maxOverdraftBuffers, 2, 2);
            this.useAllOverdraftBuffersAndCheckIsLegal(8, 5, maxOverdraftBuffers, 3, 2);
            this.useAllOverdraftBuffersAndCheckIsLegal(12, 10, maxOverdraftBuffers, 2, 1);
            this.useAllOverdraftBuffersAndCheckIsLegal(12, 10, maxOverdraftBuffers, 2, 2);
            this.useAllOverdraftBuffersAndCheckIsLegal(12, 10, maxOverdraftBuffers, 3, 2);
        }
    }

    private void useAllOverdraftBuffersAndCheckIsLegal(int poolSize, int maxBuffersPerChannel, int maxOverdraftBuffers, int numberOfChannels, int availableChannels) throws Exception {
        int i;
        Preconditions.checkArgument((maxBuffersPerChannel > poolSize / numberOfChannels ? 1 : 0) != 0);
        Preconditions.checkArgument((numberOfChannels >= availableChannels ? 1 : 0) != 0);
        LocalBufferPool bufferPool = new LocalBufferPool(this.networkBufferPool, 1, Integer.MAX_VALUE, numberOfChannels, maxBuffersPerChannel, maxOverdraftBuffers);
        bufferPool.setNumBuffers(poolSize);
        HashMap<Integer, AutoCloseableRegistry> closeableRegistryMap = new HashMap<Integer, AutoCloseableRegistry>();
        for (int i2 = 0; i2 < poolSize; ++i2) {
            int targetChannel = i2 % availableChannels;
            BufferBuilder bufferBuilder = bufferPool.requestBufferBuilder(targetChannel);
            Assertions.assertThat((Object)bufferBuilder).isNotNull();
            closeableRegistryMap.computeIfAbsent(targetChannel, channel -> new AutoCloseableRegistry()).registerCloseable((AutoCloseable)bufferBuilder);
            boolean isAvailable = i2 + 1 < poolSize && i2 < availableChannels * (maxBuffersPerChannel - 1);
            LocalBufferPoolTest.assertRequestedBufferAndIsAvailable(bufferPool, 0, i2 + 1, isAvailable);
        }
        AutoCloseableRegistry overdraftCloseableRegistry = new AutoCloseableRegistry();
        for (i = 0; i < maxOverdraftBuffers; ++i) {
            int targetChannel = i % availableChannels;
            BufferBuilder bufferBuilder = bufferPool.requestBufferBuilder(targetChannel);
            Assertions.assertThat((Object)bufferBuilder).isNotNull();
            overdraftCloseableRegistry.registerCloseable((AutoCloseable)bufferBuilder);
            int numberOfRequestedOverdraftBuffer = i + 1;
            LocalBufferPoolTest.assertRequestedBufferAndIsAvailable(bufferPool, numberOfRequestedOverdraftBuffer, poolSize + numberOfRequestedOverdraftBuffer, false);
        }
        for (i = 0; i < numberOfChannels; ++i) {
            Assertions.assertThat((Object)bufferPool.requestBufferBuilder(i)).isNull();
            LocalBufferPoolTest.assertRequestedBufferAndIsAvailable(bufferPool, maxOverdraftBuffers, poolSize + maxOverdraftBuffers, false);
        }
        overdraftCloseableRegistry.close();
        LocalBufferPoolTest.assertRequestedBufferAndIsAvailable(bufferPool, 0, poolSize, false);
        int numberOfRequestedBuffer = poolSize;
        for (AutoCloseableRegistry closeableRegistry : closeableRegistryMap.values()) {
            closeableRegistry.close();
            LocalBufferPoolTest.assertRequestedBufferAndIsAvailable(bufferPool, 0, numberOfRequestedBuffer -= closeableRegistry.getNumberOfRegisteredCloseables(), true);
        }
        bufferPool.lazyDestroy();
    }

    private static void assertRequestedBufferAndIsAvailable(LocalBufferPool bufferPool, int numberOfRequestedOverdraftBuffer, int numberOfRequestedBuffer, boolean isAvailable) {
        if (numberOfRequestedOverdraftBuffer > 0) {
            Preconditions.checkArgument((!isAvailable ? 1 : 0) != 0);
        }
        Assertions.assertThat((int)LocalBufferPoolTest.getNumberRequestedOverdraftBuffers(bufferPool)).isEqualTo(numberOfRequestedOverdraftBuffer);
        Assertions.assertThat((int)bufferPool.bestEffortGetNumOfUsedBuffers()).isEqualTo(numberOfRequestedBuffer);
        Assertions.assertThat((boolean)bufferPool.getAvailableFuture().isDone()).isEqualTo(isAvailable);
    }

    private static int getNumberRequestedOverdraftBuffers(LocalBufferPool bufferPool) {
        return Math.max(bufferPool.getNumberOfRequestedMemorySegments() - bufferPool.getNumBuffers(), 0);
    }

    private static int getNumberRequestedOrdinaryBuffers(LocalBufferPool bufferPool) {
        return Math.min(bufferPool.getNumBuffers(), bufferPool.getNumberOfRequestedMemorySegments());
    }

    private int getNumRequestedFromMemorySegmentPool() {
        return this.networkBufferPool.getTotalNumberOfMemorySegments() - this.networkBufferPool.getNumberOfAvailableMemorySegments();
    }

    private static class TestNetworkBufferPool
    extends NetworkBufferPool {
        private int requestCounter;

        public TestNetworkBufferPool(int numberOfSegmentsToAllocate, int segmentSize) {
            super(numberOfSegmentsToAllocate, segmentSize);
        }

        @Nullable
        public MemorySegment requestPooledMemorySegment() {
            if (this.requestCounter++ == 1) {
                return null;
            }
            return super.requestPooledMemorySegment();
        }
    }

    private static class BufferRequesterTask
    implements Callable<Boolean> {
        private final BufferProvider bufferProvider;
        private final int numBuffersToRequest;

        private BufferRequesterTask(BufferProvider bufferProvider, int numBuffersToRequest) {
            this.bufferProvider = bufferProvider;
            this.numBuffersToRequest = numBuffersToRequest;
        }

        @Override
        public Boolean call() {
            try {
                for (int i = 0; i < this.numBuffersToRequest; ++i) {
                    Buffer buffer = (Buffer)Preconditions.checkNotNull((Object)this.bufferProvider.requestBuffer());
                    buffer.recycleBuffer();
                }
            }
            catch (Throwable t) {
                return false;
            }
            return true;
        }
    }

    private static class CountBufferListener
    implements BufferListener {
        private final AtomicInteger times = new AtomicInteger(0);

        private CountBufferListener() {
        }

        public boolean notifyBufferAvailable(Buffer buffer) {
            this.times.incrementAndGet();
            buffer.recycleBuffer();
            return true;
        }

        public void notifyBufferDestroyed() {
        }

        int getCount() {
            return this.times.get();
        }
    }
}

