/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core.crl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import net.snowflake.client.core.crl.CRLCacheEntry;
import net.snowflake.client.core.crl.CRLCacheManager;
import net.snowflake.client.core.crl.CRLValidationConfig;
import net.snowflake.client.core.crl.CRLValidationResult;
import net.snowflake.client.core.crl.CRLValidationUtils;
import net.snowflake.client.core.crl.CertificateValidationResult;
import net.snowflake.client.jdbc.internal.apache.http.client.methods.CloseableHttpResponse;
import net.snowflake.client.jdbc.internal.apache.http.client.methods.HttpGet;
import net.snowflake.client.jdbc.internal.apache.http.impl.client.CloseableHttpClient;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

class CRLValidator {
    private static final SFLogger logger = SFLoggerFactory.getLogger(CRLValidator.class);
    private final Map<String, Lock> urlLocks = new ConcurrentHashMap<String, Lock>();
    private final CRLValidationConfig config;
    private final CloseableHttpClient httpClient;
    private final CRLCacheManager cacheManager;

    CRLValidator(CRLValidationConfig config, CloseableHttpClient httpClient, CRLCacheManager cacheManager) {
        this.httpClient = httpClient;
        this.config = config;
        this.cacheManager = cacheManager;
    }

    boolean validateCertificateChains(List<X509Certificate[]> certificateChains) {
        if (this.config.getCertRevocationCheckMode() == CRLValidationConfig.CertRevocationCheckMode.DISABLED) {
            logger.debug("CRL validation is disabled", new Object[0]);
            return true;
        }
        if (certificateChains == null || certificateChains.isEmpty()) {
            throw new IllegalArgumentException("Certificate chains cannot be null or empty");
        }
        logger.debug("Validating {} certificate chains with subjects: {}", certificateChains.size(), CRLValidationUtils.getCertChainSubjects(certificateChains));
        List<CRLValidationResult> crlValidationResults = this.validateChains(certificateChains);
        if (crlValidationResults.get(crlValidationResults.size() - 1) == CRLValidationResult.CHAIN_UNREVOKED) {
            logger.debug("Found certificate chain with all certificates unrevoked", new Object[0]);
            return true;
        }
        if (this.containsOnlyRevokedChains(crlValidationResults)) {
            logger.debug("Every verified certificate chain contained revoked certificates", new Object[0]);
            return false;
        }
        logger.debug("Some certificate chains didn't pass or driver wasn't able to perform the checks", new Object[0]);
        if (this.config.getCertRevocationCheckMode() == CRLValidationConfig.CertRevocationCheckMode.ADVISORY) {
            logger.debug("Advisory mode: allowing connection despite validation issues", new Object[0]);
            return true;
        }
        return false;
    }

    private List<CRLValidationResult> validateChains(List<X509Certificate[]> certChains) {
        ArrayList<CRLValidationResult> chainsValidationResults = new ArrayList<CRLValidationResult>();
        for (X509Certificate[] certChain : certChains) {
            CRLValidationResult chainResult = CRLValidationResult.CHAIN_UNREVOKED;
            for (int i = 0; i < certChain.length; ++i) {
                boolean isRoot;
                X509Certificate cert = certChain[i];
                boolean bl = isRoot = i == certChain.length - 1;
                if (isRoot) break;
                X509Certificate parentCert = certChain[i + 1];
                if (CRLValidationUtils.isShortLived(cert)) {
                    logger.debug("Skipping short-lived certificate: {}", cert.getSubjectX500Principal());
                    continue;
                }
                List<String> crlUrls = CRLValidationUtils.extractCRLDistributionPoints(cert);
                if (crlUrls.isEmpty()) {
                    if (this.config.isAllowCertificatesWithoutCrlUrl()) {
                        logger.debug("Certificate has missing CRL Distribution Point URLs: {}", cert.getSubjectX500Principal());
                        continue;
                    }
                    chainResult = CRLValidationResult.CHAIN_ERROR;
                    continue;
                }
                CertificateValidationResult certStatus = this.validateCert(cert, parentCert);
                if (certStatus == CertificateValidationResult.CERT_REVOKED) {
                    chainResult = CRLValidationResult.CHAIN_REVOKED;
                    break;
                }
                if (certStatus != CertificateValidationResult.CERT_ERROR) continue;
                chainResult = CRLValidationResult.CHAIN_ERROR;
            }
            chainsValidationResults.add(chainResult);
            if (chainResult != CRLValidationResult.CHAIN_UNREVOKED) continue;
            logger.debug("Found valid certificate chain, stopping validation of remaining chains", new Object[0]);
            break;
        }
        return chainsValidationResults;
    }

    private CertificateValidationResult validateCert(X509Certificate cert, X509Certificate parentCert) {
        List<String> crlUrls = CRLValidationUtils.extractCRLDistributionPoints(cert);
        HashSet<CertificateValidationResult> results = new HashSet<CertificateValidationResult>();
        for (String url : crlUrls) {
            CertificateValidationResult result = this.validateCert(cert, url, parentCert);
            if (result == CertificateValidationResult.CERT_REVOKED) {
                return result;
            }
            results.add(result);
        }
        if (results.contains((Object)CertificateValidationResult.CERT_ERROR)) {
            return CertificateValidationResult.CERT_ERROR;
        }
        return CertificateValidationResult.CERT_UNREVOKED;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private CertificateValidationResult validateCert(X509Certificate cert, String crlUrl, X509Certificate parentCert) {
        Lock lock = this.urlLocks.computeIfAbsent(crlUrl, k -> new ReentrantLock());
        lock.lock();
        try {
            CertificateValidationResult certificateValidationResult;
            boolean shouldUpdateCache;
            Instant downloadTime;
            X509CRL crl;
            block16: {
                Instant now;
                block17: {
                    now = Instant.now();
                    CRLCacheEntry cacheEntry = this.cacheManager.get(crlUrl);
                    crl = cacheEntry != null ? cacheEntry.getCrl() : null;
                    downloadTime = cacheEntry != null ? cacheEntry.getDownloadTime() : null;
                    boolean needsFreshCrl = crl == null || crl.getNextUpdate() != null && crl.getNextUpdate().toInstant().isBefore(now) || downloadTime != null && downloadTime.plus(this.config.getCacheValidityTime()).isBefore(now);
                    shouldUpdateCache = false;
                    if (!needsFreshCrl) break block16;
                    X509CRL newCrl = this.fetchCrl(crlUrl);
                    if (newCrl == null) break block17;
                    boolean bl = shouldUpdateCache = crl == null || newCrl.getThisUpdate().after(crl.getThisUpdate());
                    if (shouldUpdateCache) {
                        logger.debug("Found updated CRL for {}", crlUrl);
                        crl = newCrl;
                        downloadTime = now;
                        break block16;
                    } else if (crl.getNextUpdate() != null && crl.getNextUpdate().toInstant().isAfter(now)) {
                        logger.debug("CRL for {} is up-to-date, using cached version", crlUrl);
                        break block16;
                    } else {
                        logger.warn("CRL for {} is not available or outdated", crlUrl);
                        CertificateValidationResult certificateValidationResult2 = CertificateValidationResult.CERT_ERROR;
                        return certificateValidationResult2;
                    }
                }
                if (crl != null && crl.getNextUpdate() != null && crl.getNextUpdate().toInstant().isAfter(now)) {
                    logger.debug("Using cached CRL for {} (fetch failed but cached version still valid)", crlUrl);
                } else {
                    logger.error("Unable to fetch fresh CRL from {} and no valid cached version available", crlUrl);
                    CertificateValidationResult certificateValidationResult3 = CertificateValidationResult.CERT_ERROR;
                    return certificateValidationResult3;
                }
            }
            logger.debug("CRL has {} revoked entries, next update at {}", crl.getRevokedCertificates() != null ? crl.getRevokedCertificates().size() : 0, crl.getNextUpdate());
            if (!this.isCrlSignatureAndIssuerValid(crl, cert, parentCert, crlUrl)) {
                logger.debug("Unable to verify CRL for {}", crlUrl);
                certificateValidationResult = CertificateValidationResult.CERT_ERROR;
                return certificateValidationResult;
            }
            if (shouldUpdateCache) {
                logger.debug("CRL for {} is valid, updating cache", crlUrl);
                this.cacheManager.put(crlUrl, crl, downloadTime);
            }
            if (this.isCertificateRevoked(crl, cert)) {
                logger.debug("Certificate {} is revoked according to CRL {}", cert.getSerialNumber(), crlUrl);
                certificateValidationResult = CertificateValidationResult.CERT_REVOKED;
                return certificateValidationResult;
            }
            certificateValidationResult = CertificateValidationResult.CERT_UNREVOKED;
            return certificateValidationResult;
        }
        finally {
            lock.unlock();
        }
    }

    private boolean isCrlSignatureAndIssuerValid(X509CRL crl, X509Certificate cert, X509Certificate parentCert, String crlUrl) {
        try {
            if (!crl.getIssuerX500Principal().equals(parentCert.getSubjectX500Principal())) {
                logger.debug("CRL issuer {} does not match parent certificate subject {} for {}", crl.getIssuerX500Principal(), parentCert.getSubjectX500Principal(), "validation");
                return false;
            }
            Date now = new Date();
            if (crl.getNextUpdate() != null && now.after(crl.getNextUpdate())) {
                logger.debug("CRL has expired: nextUpdate={}, now={}", crl.getNextUpdate(), now);
                return false;
            }
            PublicKey parentPublicKey = parentCert.getPublicKey();
            try {
                crl.verify(parentPublicKey);
                logger.debug("CRL signature verified successfully using parent certificate", new Object[0]);
            }
            catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
                logger.debug("CRL signature verification failed: {}", e.getMessage());
                return false;
            }
            if (!CRLValidationUtils.verifyIssuingDistributionPoint(crl, cert, crlUrl)) {
                logger.debug("IDP extension verification failed", new Object[0]);
                return false;
            }
            return true;
        }
        catch (Exception e) {
            logger.debug("CRL validation failed: {}", e.getMessage());
            return false;
        }
    }

    private boolean isCertificateRevoked(X509CRL crl, X509Certificate cert) {
        X509CRLEntry entry = crl.getRevokedCertificate(cert.getSerialNumber());
        return entry != null;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private X509CRL fetchCrl(String crlUrl) {
        try {
            logger.debug("Fetching CRL from {}", crlUrl);
            URL url = new URL(crlUrl);
            HttpGet get = new HttpGet(url.toString());
            try (CloseableHttpResponse response = this.httpClient.execute(get);){
                X509CRL x509CRL;
                block14: {
                    InputStream inputStream = response.getEntity().getContent();
                    try {
                        CertificateFactory cf = CertificateFactory.getInstance("X.509");
                        x509CRL = (X509CRL)cf.generateCRL(inputStream);
                        if (inputStream == null) break block14;
                    }
                    catch (Throwable throwable) {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    inputStream.close();
                }
                return x509CRL;
            }
        }
        catch (IOException | CRLException | CertificateException e) {
            logger.debug("Failed to fetch CRL from {}: {}", crlUrl, e.getMessage());
            return null;
        }
    }

    private boolean containsOnlyRevokedChains(List<CRLValidationResult> results) {
        return !results.isEmpty() && results.stream().allMatch(result -> result == CRLValidationResult.CHAIN_REVOKED);
    }
}

