/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.SizeOf;
import io.trino.operator.GroupByHash;
import io.trino.operator.UpdateMemory;
import io.trino.operator.Work;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.PageBuilder;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.DictionaryBlock;
import io.trino.spi.block.RunLengthEncodedBlock;
import io.trino.spi.type.AbstractLongType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.Type;
import it.unimi.dsi.fastutil.HashCommon;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

public class BigintGroupByHash
implements GroupByHash {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(BigintGroupByHash.class);
    private static final int BATCH_SIZE = 1024;
    private static final float FILL_RATIO = 0.75f;
    private static final List<Type> TYPES = ImmutableList.of((Object)BigintType.BIGINT);
    private static final List<Type> TYPES_WITH_RAW_HASH = ImmutableList.of((Object)BigintType.BIGINT, (Object)BigintType.BIGINT);
    private final int hashChannel;
    private final boolean outputRawHash;
    private int hashCapacity;
    private int maxFill;
    private int mask;
    private long[] values;
    private int[] groupIds;
    private int nullGroupId = -1;
    private long[] valuesByGroupId;
    private int nextGroupId;
    private DictionaryLookBack dictionaryLookBack;
    private final UpdateMemory updateMemory;
    private long preallocatedMemoryInBytes;
    private long currentPageSizeInBytes;

    public BigintGroupByHash(int hashChannel, boolean outputRawHash, int expectedSize, UpdateMemory updateMemory) {
        Preconditions.checkArgument((hashChannel >= 0 ? 1 : 0) != 0, (Object)"hashChannel must be at least zero");
        Preconditions.checkArgument((expectedSize > 0 ? 1 : 0) != 0, (Object)"expectedSize must be greater than zero");
        this.hashChannel = hashChannel;
        this.outputRawHash = outputRawHash;
        this.hashCapacity = HashCommon.arraySize((int)expectedSize, (float)0.75f);
        this.maxFill = BigintGroupByHash.calculateMaxFill(this.hashCapacity);
        this.mask = this.hashCapacity - 1;
        this.values = new long[this.hashCapacity];
        this.groupIds = new int[this.hashCapacity];
        Arrays.fill(this.groupIds, -1);
        this.valuesByGroupId = new long[this.maxFill];
        this.updateMemory = Objects.requireNonNull(updateMemory, "updateMemory is null");
    }

    @Override
    public long getEstimatedSize() {
        return (long)INSTANCE_SIZE + SizeOf.sizeOf((int[])this.groupIds) + SizeOf.sizeOf((long[])this.values) + SizeOf.sizeOf((long[])this.valuesByGroupId) + this.preallocatedMemoryInBytes;
    }

    @Override
    public List<Type> getTypes() {
        return this.outputRawHash ? TYPES_WITH_RAW_HASH : TYPES;
    }

    @Override
    public int getGroupCount() {
        return this.nextGroupId;
    }

    @Override
    public void appendValuesTo(int groupId, PageBuilder pageBuilder) {
        Preconditions.checkArgument((groupId >= 0 ? 1 : 0) != 0, (Object)"groupId is negative");
        BlockBuilder blockBuilder = pageBuilder.getBlockBuilder(0);
        if (groupId == this.nullGroupId) {
            blockBuilder.appendNull();
        } else {
            BigintType.BIGINT.writeLong(blockBuilder, this.valuesByGroupId[groupId]);
        }
        if (this.outputRawHash) {
            BlockBuilder hashBlockBuilder = pageBuilder.getBlockBuilder(1);
            if (groupId == this.nullGroupId) {
                BigintType.BIGINT.writeLong(hashBlockBuilder, 0L);
            } else {
                BigintType.BIGINT.writeLong(hashBlockBuilder, AbstractLongType.hash((long)this.valuesByGroupId[groupId]));
            }
        }
    }

    @Override
    public Work<?> addPage(Page page) {
        this.currentPageSizeInBytes = page.getRetainedSizeInBytes();
        Block block = page.getBlock(this.hashChannel);
        if (block instanceof RunLengthEncodedBlock) {
            RunLengthEncodedBlock rleBlock = (RunLengthEncodedBlock)block;
            return new AddRunLengthEncodedPageWork(rleBlock);
        }
        if (block instanceof DictionaryBlock) {
            DictionaryBlock dictionaryBlock = (DictionaryBlock)block;
            return new AddDictionaryPageWork(dictionaryBlock);
        }
        return new AddPageWork(block);
    }

    @Override
    public Work<int[]> getGroupIds(Page page) {
        this.currentPageSizeInBytes = page.getRetainedSizeInBytes();
        Block block = page.getBlock(this.hashChannel);
        if (block instanceof RunLengthEncodedBlock) {
            RunLengthEncodedBlock rleBlock = (RunLengthEncodedBlock)block;
            return new GetRunLengthEncodedGroupIdsWork(rleBlock);
        }
        if (block instanceof DictionaryBlock) {
            DictionaryBlock dictionaryBlock = (DictionaryBlock)block;
            return new GetDictionaryGroupIdsWork(dictionaryBlock);
        }
        return new GetGroupIdsWork(block);
    }

    @Override
    public boolean contains(int position, Page page, int[] hashChannels) {
        Block block = page.getBlock(this.hashChannel);
        if (block.isNull(position)) {
            return this.nullGroupId >= 0;
        }
        long value = BigintType.BIGINT.getLong(block, position);
        int hashPosition = BigintGroupByHash.getHashPosition(value, this.mask);
        int groupId;
        while ((groupId = this.groupIds[hashPosition]) != -1) {
            if (value == this.values[hashPosition]) {
                return true;
            }
            hashPosition = hashPosition + 1 & this.mask;
        }
        return false;
    }

    @Override
    public long getRawHash(int groupId) {
        return BigintType.hash((long)this.valuesByGroupId[groupId]);
    }

    @Override
    @VisibleForTesting
    public int getCapacity() {
        return this.hashCapacity;
    }

    private int putIfAbsent(int position, Block block) {
        int groupId;
        if (block.isNull(position)) {
            if (this.nullGroupId < 0) {
                this.nullGroupId = this.nextGroupId++;
            }
            return this.nullGroupId;
        }
        long value = BigintType.BIGINT.getLong(block, position);
        int hashPosition = BigintGroupByHash.getHashPosition(value, this.mask);
        while ((groupId = this.groupIds[hashPosition]) != -1) {
            if (value == this.values[hashPosition]) {
                return groupId;
            }
            hashPosition = hashPosition + 1 & this.mask;
        }
        return this.addNewGroup(hashPosition, value);
    }

    private int addNewGroup(int hashPosition, long value) {
        int groupId = this.nextGroupId++;
        this.values[hashPosition] = value;
        this.valuesByGroupId[groupId] = value;
        this.groupIds[hashPosition] = groupId;
        if (this.needRehash()) {
            this.tryRehash();
        }
        return groupId;
    }

    private boolean tryRehash() {
        long newCapacityLong = (long)this.hashCapacity * 2L;
        if (newCapacityLong > Integer.MAX_VALUE) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INSUFFICIENT_RESOURCES, "Size of hash table cannot exceed 1 billion entries");
        }
        int newCapacity = Math.toIntExact(newCapacityLong);
        this.preallocatedMemoryInBytes = (long)newCapacity * 12L + (long)BigintGroupByHash.calculateMaxFill(newCapacity) * 8L + this.currentPageSizeInBytes;
        if (!this.updateMemory.update()) {
            return false;
        }
        int newMask = newCapacity - 1;
        long[] newValues = new long[newCapacity];
        int[] newGroupIds = new int[newCapacity];
        Arrays.fill(newGroupIds, -1);
        for (int i = 0; i < this.values.length; ++i) {
            int groupId = this.groupIds[i];
            if (groupId == -1) continue;
            long value = this.values[i];
            int hashPosition = BigintGroupByHash.getHashPosition(value, newMask);
            while (newGroupIds[hashPosition] != -1) {
                hashPosition = hashPosition + 1 & newMask;
            }
            newValues[hashPosition] = value;
            newGroupIds[hashPosition] = groupId;
        }
        this.mask = newMask;
        this.hashCapacity = newCapacity;
        this.maxFill = BigintGroupByHash.calculateMaxFill(this.hashCapacity);
        this.values = newValues;
        this.groupIds = newGroupIds;
        this.valuesByGroupId = Arrays.copyOf(this.valuesByGroupId, this.maxFill);
        this.preallocatedMemoryInBytes = 0L;
        this.updateMemory.update();
        return true;
    }

    private boolean needRehash() {
        return this.nextGroupId >= this.maxFill;
    }

    private static int getHashPosition(long rawHash, int mask) {
        return (int)(HashCommon.murmurHash3((long)rawHash) & (long)mask);
    }

    private static int calculateMaxFill(int hashSize) {
        Preconditions.checkArgument((hashSize > 0 ? 1 : 0) != 0, (Object)"hashSize must be greater than 0");
        int maxFill = (int)Math.ceil((float)hashSize * 0.75f);
        if (maxFill == hashSize) {
            --maxFill;
        }
        Preconditions.checkArgument((hashSize > maxFill ? 1 : 0) != 0, (Object)"hashSize must be larger than maxFill");
        return maxFill;
    }

    private void updateDictionaryLookBack(Block dictionary) {
        if (this.dictionaryLookBack == null || this.dictionaryLookBack.getDictionary() != dictionary) {
            this.dictionaryLookBack = new DictionaryLookBack(dictionary);
        }
    }

    private int registerGroupId(Block dictionary, int positionInDictionary) {
        if (this.dictionaryLookBack.isProcessed(positionInDictionary)) {
            return this.dictionaryLookBack.getGroupId(positionInDictionary);
        }
        int groupId = this.putIfAbsent(positionInDictionary, dictionary);
        this.dictionaryLookBack.setProcessed(positionInDictionary, groupId);
        return groupId;
    }

    private boolean ensureHashTableSize(int batchSize) {
        int positionCountUntilRehash = this.maxFill - this.nextGroupId;
        while (positionCountUntilRehash < batchSize) {
            if (!this.tryRehash()) {
                return false;
            }
            positionCountUntilRehash = this.maxFill - this.nextGroupId;
        }
        return true;
    }

    @VisibleForTesting
    class AddRunLengthEncodedPageWork
    implements Work<Void> {
        private final RunLengthEncodedBlock block;
        private boolean finished;

        public AddRunLengthEncodedPageWork(RunLengthEncodedBlock block) {
            this.block = Objects.requireNonNull(block, "block is null");
        }

        @Override
        public boolean process() {
            Preconditions.checkState((!this.finished ? 1 : 0) != 0);
            if (this.block.getPositionCount() == 0) {
                this.finished = true;
                return true;
            }
            if (BigintGroupByHash.this.needRehash() && !BigintGroupByHash.this.tryRehash()) {
                return false;
            }
            BigintGroupByHash.this.putIfAbsent(0, this.block.getValue());
            this.finished = true;
            return true;
        }

        @Override
        public Void getResult() {
            throw new UnsupportedOperationException();
        }
    }

    @VisibleForTesting
    class AddDictionaryPageWork
    implements Work<Void> {
        private final Block dictionary;
        private final DictionaryBlock block;
        private int lastPosition;

        public AddDictionaryPageWork(DictionaryBlock block) {
            this.block = Objects.requireNonNull(block, "block is null");
            this.dictionary = block.getDictionary();
            BigintGroupByHash.this.updateDictionaryLookBack(this.dictionary);
        }

        @Override
        public boolean process() {
            int positionCount = this.block.getPositionCount();
            Preconditions.checkState((this.lastPosition <= positionCount ? 1 : 0) != 0, (Object)"position count out of bound");
            if (BigintGroupByHash.this.needRehash() && !BigintGroupByHash.this.tryRehash()) {
                return false;
            }
            while (this.lastPosition < positionCount && !BigintGroupByHash.this.needRehash()) {
                int positionInDictionary = this.block.getId(this.lastPosition);
                BigintGroupByHash.this.registerGroupId(this.dictionary, positionInDictionary);
                ++this.lastPosition;
            }
            return this.lastPosition == positionCount;
        }

        @Override
        public Void getResult() {
            throw new UnsupportedOperationException();
        }
    }

    @VisibleForTesting
    class AddPageWork
    implements Work<Void> {
        private final Block block;
        private int lastPosition;

        public AddPageWork(Block block) {
            this.block = Objects.requireNonNull(block, "block is null");
        }

        @Override
        public boolean process() {
            int batchSize;
            int positionCount = this.block.getPositionCount();
            Preconditions.checkState((this.lastPosition <= positionCount ? 1 : 0) != 0, (Object)"position count out of bound");
            for (int remainingPositions = positionCount - this.lastPosition; remainingPositions != 0; remainingPositions -= batchSize) {
                batchSize = Math.min(remainingPositions, 1024);
                if (!BigintGroupByHash.this.ensureHashTableSize(batchSize)) {
                    return false;
                }
                for (int i = this.lastPosition; i < this.lastPosition + batchSize; ++i) {
                    BigintGroupByHash.this.putIfAbsent(i, this.block);
                }
                this.lastPosition += batchSize;
            }
            Verify.verify((this.lastPosition == positionCount ? 1 : 0) != 0);
            return true;
        }

        @Override
        public Void getResult() {
            throw new UnsupportedOperationException();
        }
    }

    @VisibleForTesting
    class GetRunLengthEncodedGroupIdsWork
    implements Work<int[]> {
        private final RunLengthEncodedBlock block;
        int groupId = -1;
        private boolean processFinished;
        private boolean resultProduced;

        public GetRunLengthEncodedGroupIdsWork(RunLengthEncodedBlock block) {
            this.block = Objects.requireNonNull(block, "block is null");
        }

        @Override
        public boolean process() {
            Preconditions.checkState((!this.processFinished ? 1 : 0) != 0);
            if (this.block.getPositionCount() == 0) {
                this.processFinished = true;
                return true;
            }
            if (BigintGroupByHash.this.needRehash() && !BigintGroupByHash.this.tryRehash()) {
                return false;
            }
            this.groupId = BigintGroupByHash.this.putIfAbsent(0, this.block.getValue());
            this.processFinished = true;
            return true;
        }

        @Override
        public int[] getResult() {
            Preconditions.checkState((boolean)this.processFinished);
            Preconditions.checkState((!this.resultProduced ? 1 : 0) != 0);
            this.resultProduced = true;
            int[] result = new int[this.block.getPositionCount()];
            Arrays.fill(result, this.groupId);
            return result;
        }
    }

    @VisibleForTesting
    class GetDictionaryGroupIdsWork
    implements Work<int[]> {
        private final int[] groupIds;
        private final Block dictionary;
        private final DictionaryBlock block;
        private boolean finished;
        private int lastPosition;

        public GetDictionaryGroupIdsWork(DictionaryBlock block) {
            this.block = Objects.requireNonNull(block, "block is null");
            this.dictionary = block.getDictionary();
            BigintGroupByHash.this.updateDictionaryLookBack(this.dictionary);
            this.groupIds = new int[block.getPositionCount()];
        }

        @Override
        public boolean process() {
            int positionCount = this.block.getPositionCount();
            Preconditions.checkState((this.lastPosition <= positionCount ? 1 : 0) != 0, (Object)"position count out of bound");
            Preconditions.checkState((!this.finished ? 1 : 0) != 0);
            if (BigintGroupByHash.this.needRehash() && !BigintGroupByHash.this.tryRehash()) {
                return false;
            }
            while (this.lastPosition < positionCount && !BigintGroupByHash.this.needRehash()) {
                int groupId;
                int positionInDictionary = this.block.getId(this.lastPosition);
                this.groupIds[this.lastPosition] = groupId = BigintGroupByHash.this.registerGroupId(this.dictionary, positionInDictionary);
                ++this.lastPosition;
            }
            return this.lastPosition == positionCount;
        }

        @Override
        public int[] getResult() {
            Preconditions.checkState((this.lastPosition == this.block.getPositionCount() ? 1 : 0) != 0, (Object)"process has not yet finished");
            Preconditions.checkState((!this.finished ? 1 : 0) != 0, (Object)"result has produced");
            this.finished = true;
            return this.groupIds;
        }
    }

    @VisibleForTesting
    class GetGroupIdsWork
    implements Work<int[]> {
        private final int[] groupIds;
        private final Block block;
        private boolean finished;
        private int lastPosition;

        public GetGroupIdsWork(Block block) {
            this.block = Objects.requireNonNull(block, "block is null");
            this.groupIds = new int[block.getPositionCount()];
        }

        @Override
        public boolean process() {
            int batchSize;
            int positionCount = this.block.getPositionCount();
            Preconditions.checkState((this.lastPosition <= positionCount ? 1 : 0) != 0, (Object)"position count out of bound");
            Preconditions.checkState((!this.finished ? 1 : 0) != 0);
            for (int remainingPositions = positionCount - this.lastPosition; remainingPositions != 0; remainingPositions -= batchSize) {
                batchSize = Math.min(remainingPositions, 1024);
                if (!BigintGroupByHash.this.ensureHashTableSize(batchSize)) {
                    return false;
                }
                for (int i = this.lastPosition; i < this.lastPosition + batchSize; ++i) {
                    this.groupIds[i] = BigintGroupByHash.this.putIfAbsent(i, this.block);
                }
                this.lastPosition += batchSize;
            }
            Verify.verify((this.lastPosition == positionCount ? 1 : 0) != 0);
            return true;
        }

        @Override
        public int[] getResult() {
            Preconditions.checkState((this.lastPosition == this.block.getPositionCount() ? 1 : 0) != 0, (Object)"process has not yet finished");
            Preconditions.checkState((!this.finished ? 1 : 0) != 0, (Object)"result has produced");
            this.finished = true;
            return this.groupIds;
        }
    }

    private static final class DictionaryLookBack {
        private final Block dictionary;
        private final int[] processed;

        public DictionaryLookBack(Block dictionary) {
            this.dictionary = dictionary;
            this.processed = new int[dictionary.getPositionCount()];
            Arrays.fill(this.processed, -1);
        }

        public Block getDictionary() {
            return this.dictionary;
        }

        public int getGroupId(int position) {
            return this.processed[position];
        }

        public boolean isProcessed(int position) {
            return this.processed[position] != -1;
        }

        public void setProcessed(int position, int groupId) {
            this.processed[position] = groupId;
        }
    }
}

