/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.errorprone.annotations.ThreadSafe;
import io.airlift.log.Logger;
import io.airlift.stats.Distribution;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.Tracer;
import io.opentelemetry.context.Context;
import io.opentelemetry.context.ImplicitContextKeyed;
import io.trino.execution.BasicStageStats;
import io.trino.execution.ExecutionFailureInfo;
import io.trino.execution.StageId;
import io.trino.execution.StageInfo;
import io.trino.execution.StageState;
import io.trino.execution.StageStats;
import io.trino.execution.StateMachine;
import io.trino.execution.TableInfo;
import io.trino.execution.TaskInfo;
import io.trino.execution.TaskState;
import io.trino.execution.scheduler.SplitSchedulerStats;
import io.trino.operator.BlockedReason;
import io.trino.operator.OperatorStats;
import io.trino.operator.PipelineStats;
import io.trino.operator.TaskStats;
import io.trino.plugin.base.metrics.TDigestHistogram;
import io.trino.spi.eventlistener.StageGcStatistics;
import io.trino.sql.planner.PlanFragment;
import io.trino.sql.planner.plan.PlanNodeId;
import io.trino.sql.planner.plan.TableScanNode;
import io.trino.tracing.TrinoAttributes;
import io.trino.util.Failures;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.LongFunction;
import java.util.function.Supplier;
import org.joda.time.DateTime;

@ThreadSafe
public class StageStateMachine {
    private static final Logger log = Logger.get(StageStateMachine.class);
    private final StageId stageId;
    private final PlanFragment fragment;
    private final Map<PlanNodeId, TableInfo> tables;
    private final SplitSchedulerStats scheduledStats;
    private final StateMachine<StageState> stageState;
    private final StateMachine<Optional<StageInfo>> finalStageInfo;
    private final Span stageSpan;
    private final AtomicReference<ExecutionFailureInfo> failureCause = new AtomicReference();
    private final AtomicReference<DateTime> schedulingComplete = new AtomicReference();
    private final Distribution getSplitDistribution = new Distribution();
    private final AtomicLong peakUserMemory = new AtomicLong();
    private final AtomicLong peakRevocableMemory = new AtomicLong();
    private final AtomicLong currentUserMemory = new AtomicLong();
    private final AtomicLong currentRevocableMemory = new AtomicLong();
    private final AtomicLong currentTotalMemory = new AtomicLong();

    public StageStateMachine(StageId stageId, PlanFragment fragment, Map<PlanNodeId, TableInfo> tables, Executor executor, Tracer tracer, Span querySpan, SplitSchedulerStats schedulerStats) {
        this.stageId = Objects.requireNonNull(stageId, "stageId is null");
        this.fragment = Objects.requireNonNull(fragment, "fragment is null");
        this.tables = ImmutableMap.copyOf(Objects.requireNonNull(tables, "tables is null"));
        this.scheduledStats = Objects.requireNonNull(schedulerStats, "schedulerStats is null");
        this.stageState = new StateMachine<StageState>("stage " + stageId, executor, StageState.PLANNED, StageState.TERMINAL_STAGE_STATES);
        this.stageState.addStateChangeListener((T state) -> log.debug("Stage %s is %s", new Object[]{stageId, state}));
        this.finalStageInfo = new StateMachine("final stage " + stageId, executor, Optional.empty());
        this.stageSpan = tracer.spanBuilder("stage").setParent(Context.current().with((ImplicitContextKeyed)querySpan)).setAttribute(TrinoAttributes.QUERY_ID, (Object)stageId.getQueryId().toString()).setAttribute(TrinoAttributes.STAGE_ID, (Object)stageId.toString()).startSpan();
        this.stageState.addStateChangeListener((T state) -> {
            this.stageSpan.addEvent("stage_state", Attributes.of(TrinoAttributes.EVENT_STATE, (Object)state.toString()));
            if (state.isDone()) {
                this.stageSpan.end();
            }
        });
    }

    public StageId getStageId() {
        return this.stageId;
    }

    public StageState getState() {
        return this.stageState.get();
    }

    public PlanFragment getFragment() {
        return this.fragment;
    }

    public Span getStageSpan() {
        return this.stageSpan;
    }

    public void addStateChangeListener(StateMachine.StateChangeListener<StageState> stateChangeListener) {
        this.stageState.addStateChangeListener(stateChangeListener);
    }

    public boolean transitionToScheduling() {
        return this.stageState.compareAndSet(StageState.PLANNED, StageState.SCHEDULING);
    }

    public boolean transitionToRunning() {
        this.schedulingComplete.compareAndSet(null, DateTime.now());
        return this.stageState.setIf(StageState.RUNNING, currentState -> currentState != StageState.RUNNING && !currentState.isDone());
    }

    public boolean transitionToPending() {
        return this.stageState.setIf(StageState.PENDING, currentState -> currentState != StageState.PENDING && !currentState.isDone());
    }

    public boolean transitionToFinished() {
        return this.stageState.setIf(StageState.FINISHED, currentState -> !currentState.isDone());
    }

    public boolean transitionToAborted() {
        return this.stageState.setIf(StageState.ABORTED, currentState -> !currentState.isDone());
    }

    public boolean transitionToFailed(Throwable throwable) {
        Objects.requireNonNull(throwable, "throwable is null");
        this.failureCause.compareAndSet(null, Failures.toFailure(throwable));
        boolean failed = this.stageState.setIf(StageState.FAILED, currentState -> !currentState.isDone());
        if (failed) {
            log.error(throwable, "Stage %s failed", new Object[]{this.stageId});
        } else {
            log.debug(throwable, "Failure after stage %s finished", new Object[]{this.stageId});
        }
        return failed;
    }

    public void addFinalStageInfoListener(StateMachine.StateChangeListener<StageInfo> finalStatusListener) {
        AtomicBoolean done = new AtomicBoolean();
        StateMachine.StateChangeListener<Optional> fireOnceStateChangeListener = finalStageInfo -> {
            if (finalStageInfo.isPresent() && done.compareAndSet(false, true)) {
                finalStatusListener.stateChanged((StageInfo)finalStageInfo.get());
            }
        };
        this.finalStageInfo.addStateChangeListener(fireOnceStateChangeListener);
    }

    public void setAllTasksFinal(Iterable<TaskInfo> finalTaskInfos) {
        Objects.requireNonNull(finalTaskInfos, "finalTaskInfos is null");
        Preconditions.checkState((boolean)this.stageState.get().isDone());
        StageInfo stageInfo = this.getStageInfo(() -> finalTaskInfos);
        Preconditions.checkArgument((boolean)stageInfo.isFinalStageInfo(), (Object)"finalTaskInfos are not all done");
        this.finalStageInfo.compareAndSet(Optional.empty(), Optional.of(stageInfo));
    }

    public long getUserMemoryReservation() {
        return this.currentUserMemory.get();
    }

    public long getTotalMemoryReservation() {
        return this.currentTotalMemory.get();
    }

    public void updateMemoryUsage(long deltaUserMemoryInBytes, long deltaRevocableMemoryInBytes, long deltaTotalMemoryInBytes) {
        this.currentUserMemory.addAndGet(deltaUserMemoryInBytes);
        this.currentRevocableMemory.addAndGet(deltaRevocableMemoryInBytes);
        this.currentTotalMemory.addAndGet(deltaTotalMemoryInBytes);
        this.peakUserMemory.updateAndGet(currentPeakValue -> Math.max(this.currentUserMemory.get(), currentPeakValue));
        this.peakRevocableMemory.updateAndGet(currentPeakValue -> Math.max(this.currentRevocableMemory.get(), currentPeakValue));
    }

    public BasicStageStats getBasicStageStats(Supplier<Iterable<TaskInfo>> taskInfosSupplier) {
        Optional<StageInfo> finalStageInfo = this.finalStageInfo.get();
        if (finalStageInfo.isPresent()) {
            return finalStageInfo.get().getStageStats().toBasicStageStats(finalStageInfo.get().getState());
        }
        StageState state = this.stageState.get();
        boolean isScheduled = state == StageState.RUNNING || state == StageState.PENDING || state.isDone();
        ImmutableList taskInfos = ImmutableList.copyOf(taskInfosSupplier.get());
        int failedTasks = 0;
        int totalDrivers = 0;
        int queuedDrivers = 0;
        int runningDrivers = 0;
        int completedDrivers = 0;
        double cumulativeUserMemory = 0.0;
        double failedCumulativeUserMemory = 0.0;
        long userMemoryReservation = 0L;
        long totalMemoryReservation = 0L;
        long totalScheduledTime = 0L;
        long failedScheduledTime = 0L;
        long totalCpuTime = 0L;
        long failedCpuTime = 0L;
        long physicalInputDataSize = 0L;
        long physicalInputPositions = 0L;
        long physicalInputReadTime = 0L;
        long internalNetworkInputDataSize = 0L;
        long internalNetworkInputPositions = 0L;
        long rawInputDataSize = 0L;
        long rawInputPositions = 0L;
        boolean fullyBlocked = true;
        HashSet<BlockedReason> blockedReasons = new HashSet<BlockedReason>();
        for (TaskInfo taskInfo : taskInfos) {
            boolean taskFailedOrFailing;
            TaskState taskState = taskInfo.getTaskStatus().getState();
            TaskStats taskStats = taskInfo.getStats();
            boolean bl = taskFailedOrFailing = taskState == TaskState.FAILED || taskState == TaskState.FAILING;
            if (taskFailedOrFailing) {
                ++failedTasks;
            }
            totalDrivers += taskStats.getTotalDrivers();
            queuedDrivers += taskStats.getQueuedDrivers();
            runningDrivers += taskStats.getRunningDrivers();
            completedDrivers += taskStats.getCompletedDrivers();
            cumulativeUserMemory += taskStats.getCumulativeUserMemory();
            if (taskFailedOrFailing) {
                failedCumulativeUserMemory += taskStats.getCumulativeUserMemory();
            }
            long taskUserMemory = taskStats.getUserMemoryReservation().toBytes();
            long taskRevocableMemory = taskStats.getRevocableMemoryReservation().toBytes();
            userMemoryReservation += taskUserMemory;
            totalMemoryReservation += taskUserMemory + taskRevocableMemory;
            totalScheduledTime += taskStats.getTotalScheduledTime().roundTo(TimeUnit.NANOSECONDS);
            totalCpuTime += taskStats.getTotalCpuTime().roundTo(TimeUnit.NANOSECONDS);
            if (taskFailedOrFailing) {
                failedScheduledTime += taskStats.getTotalScheduledTime().roundTo(TimeUnit.NANOSECONDS);
                failedCpuTime += taskStats.getTotalCpuTime().roundTo(TimeUnit.NANOSECONDS);
            }
            if (!taskState.isDone()) {
                fullyBlocked &= taskStats.isFullyBlocked();
                blockedReasons.addAll(taskStats.getBlockedReasons());
            }
            physicalInputDataSize += taskStats.getPhysicalInputDataSize().toBytes();
            physicalInputPositions += taskStats.getPhysicalInputPositions();
            physicalInputReadTime += taskStats.getPhysicalInputReadTime().roundTo(TimeUnit.NANOSECONDS);
            internalNetworkInputDataSize += taskStats.getInternalNetworkInputDataSize().toBytes();
            internalNetworkInputPositions += taskStats.getInternalNetworkInputPositions();
            if (!this.fragment.getPartitionedSourceNodes().stream().anyMatch(TableScanNode.class::isInstance)) continue;
            rawInputDataSize += taskStats.getRawInputDataSize().toBytes();
            rawInputPositions += taskStats.getRawInputPositions();
        }
        OptionalDouble progressPercentage = OptionalDouble.empty();
        if (isScheduled && totalDrivers != 0) {
            progressPercentage = OptionalDouble.of(Math.min(100.0, (double)completedDrivers * 100.0 / (double)totalDrivers));
        }
        OptionalDouble runningPercentage = OptionalDouble.empty();
        if (isScheduled && totalDrivers != 0) {
            runningPercentage = OptionalDouble.of(Math.min(100.0, (double)runningDrivers * 100.0 / (double)totalDrivers));
        }
        return new BasicStageStats(isScheduled, failedTasks, totalDrivers, queuedDrivers, runningDrivers, completedDrivers, DataSize.succinctBytes((long)physicalInputDataSize), physicalInputPositions, new Duration((double)physicalInputReadTime, TimeUnit.NANOSECONDS).convertToMostSuccinctTimeUnit(), DataSize.succinctBytes((long)internalNetworkInputDataSize), internalNetworkInputPositions, DataSize.succinctBytes((long)rawInputDataSize), rawInputPositions, cumulativeUserMemory, failedCumulativeUserMemory, DataSize.succinctBytes((long)userMemoryReservation), DataSize.succinctBytes((long)totalMemoryReservation), new Duration((double)totalCpuTime, TimeUnit.NANOSECONDS).convertToMostSuccinctTimeUnit(), new Duration((double)failedCpuTime, TimeUnit.NANOSECONDS).convertToMostSuccinctTimeUnit(), new Duration((double)totalScheduledTime, TimeUnit.NANOSECONDS).convertToMostSuccinctTimeUnit(), new Duration((double)failedScheduledTime, TimeUnit.NANOSECONDS).convertToMostSuccinctTimeUnit(), fullyBlocked, blockedReasons, progressPercentage, runningPercentage);
    }

    public StageInfo getStageInfo(Supplier<Iterable<TaskInfo>> taskInfosSupplier) {
        Optional<StageInfo> finalStageInfo = this.finalStageInfo.get();
        if (finalStageInfo.isPresent()) {
            return finalStageInfo.get();
        }
        StageState state = this.stageState.get();
        ImmutableList taskInfos = ImmutableList.copyOf(taskInfosSupplier.get());
        int totalTasks = taskInfos.size();
        int runningTasks = 0;
        int completedTasks = 0;
        int failedTasks = 0;
        int totalDrivers = 0;
        int queuedDrivers = 0;
        int runningDrivers = 0;
        int blockedDrivers = 0;
        int completedDrivers = 0;
        double cumulativeUserMemory = 0.0;
        double failedCumulativeUserMemory = 0.0;
        long userMemoryReservation = this.currentUserMemory.get();
        long revocableMemoryReservation = this.currentRevocableMemory.get();
        long totalMemoryReservation = this.currentTotalMemory.get();
        long peakUserMemoryReservation = this.peakUserMemory.get();
        long peakRevocableMemoryReservation = this.peakRevocableMemory.get();
        long totalScheduledTime = 0L;
        long failedScheduledTime = 0L;
        long totalCpuTime = 0L;
        long failedCpuTime = 0L;
        long totalBlockedTime = 0L;
        long physicalInputDataSize = 0L;
        long failedPhysicalInputDataSize = 0L;
        long physicalInputPositions = 0L;
        long failedPhysicalInputPositions = 0L;
        long physicalInputReadTime = 0L;
        long failedPhysicalInputReadTime = 0L;
        long internalNetworkInputDataSize = 0L;
        long failedInternalNetworkInputDataSize = 0L;
        long internalNetworkInputPositions = 0L;
        long failedInternalNetworkInputPositions = 0L;
        long rawInputDataSize = 0L;
        long failedRawInputDataSize = 0L;
        long rawInputPositions = 0L;
        long failedRawInputPositions = 0L;
        long processedInputDataSize = 0L;
        long failedProcessedInputDataSize = 0L;
        long processedInputPositions = 0L;
        long failedProcessedInputPositions = 0L;
        long inputBlockedTime = 0L;
        long failedInputBlockedTime = 0L;
        long bufferedDataSize = 0L;
        ImmutableList.Builder bufferUtilizationHistograms = ImmutableList.builderWithExpectedSize((int)taskInfos.size());
        long outputDataSize = 0L;
        long failedOutputDataSize = 0L;
        long outputPositions = 0L;
        long failedOutputPositions = 0L;
        long outputBlockedTime = 0L;
        long failedOutputBlockedTime = 0L;
        long physicalWrittenDataSize = 0L;
        long failedPhysicalWrittenDataSize = 0L;
        int fullGcCount = 0;
        int fullGcTaskCount = 0;
        int minFullGcSec = 0;
        int maxFullGcSec = 0;
        int totalFullGcSec = 0;
        boolean fullyBlocked = true;
        HashSet<BlockedReason> blockedReasons = new HashSet<BlockedReason>();
        int maxTaskOperatorSummaries = 0;
        for (TaskInfo taskInfo : taskInfos) {
            boolean taskFailedOrFailing;
            TaskState taskState = taskInfo.getTaskStatus().getState();
            if (taskState.isDone()) {
                ++completedTasks;
            } else {
                ++runningTasks;
            }
            boolean bl = taskFailedOrFailing = taskState == TaskState.FAILED || taskState == TaskState.FAILING;
            if (taskFailedOrFailing) {
                ++failedTasks;
            }
            TaskStats taskStats = taskInfo.getStats();
            totalDrivers += taskStats.getTotalDrivers();
            queuedDrivers += taskStats.getQueuedDrivers();
            runningDrivers += taskStats.getRunningDrivers();
            blockedDrivers += taskStats.getBlockedDrivers();
            completedDrivers += taskStats.getCompletedDrivers();
            cumulativeUserMemory += taskStats.getCumulativeUserMemory();
            if (taskFailedOrFailing) {
                failedCumulativeUserMemory += taskStats.getCumulativeUserMemory();
            }
            totalScheduledTime += taskStats.getTotalScheduledTime().roundTo(TimeUnit.NANOSECONDS);
            totalCpuTime += taskStats.getTotalCpuTime().roundTo(TimeUnit.NANOSECONDS);
            totalBlockedTime += taskStats.getTotalBlockedTime().roundTo(TimeUnit.NANOSECONDS);
            if (taskFailedOrFailing) {
                failedScheduledTime += taskStats.getTotalScheduledTime().roundTo(TimeUnit.NANOSECONDS);
                failedCpuTime += taskStats.getTotalCpuTime().roundTo(TimeUnit.NANOSECONDS);
            }
            if (!taskState.isDone()) {
                fullyBlocked &= taskStats.isFullyBlocked();
                blockedReasons.addAll(taskStats.getBlockedReasons());
            }
            physicalInputDataSize += taskStats.getPhysicalInputDataSize().toBytes();
            physicalInputPositions += taskStats.getPhysicalInputPositions();
            physicalInputReadTime += taskStats.getPhysicalInputReadTime().roundTo(TimeUnit.NANOSECONDS);
            internalNetworkInputDataSize += taskStats.getInternalNetworkInputDataSize().toBytes();
            internalNetworkInputPositions += taskStats.getInternalNetworkInputPositions();
            rawInputDataSize += taskStats.getRawInputDataSize().toBytes();
            rawInputPositions += taskStats.getRawInputPositions();
            processedInputDataSize += taskStats.getProcessedInputDataSize().toBytes();
            processedInputPositions += taskStats.getProcessedInputPositions();
            inputBlockedTime += taskStats.getInputBlockedTime().roundTo(TimeUnit.NANOSECONDS);
            bufferedDataSize += taskInfo.getOutputBuffers().getTotalBufferedBytes();
            taskInfo.getOutputBuffers().getUtilization().ifPresent(arg_0 -> ((ImmutableList.Builder)bufferUtilizationHistograms).add(arg_0));
            outputDataSize += taskStats.getOutputDataSize().toBytes();
            outputPositions += taskStats.getOutputPositions();
            outputBlockedTime += taskStats.getOutputBlockedTime().roundTo(TimeUnit.NANOSECONDS);
            physicalWrittenDataSize += taskStats.getPhysicalWrittenDataSize().toBytes();
            if (taskFailedOrFailing) {
                failedPhysicalInputDataSize += taskStats.getPhysicalInputDataSize().toBytes();
                failedPhysicalInputPositions += taskStats.getPhysicalInputPositions();
                failedPhysicalInputReadTime += taskStats.getPhysicalInputReadTime().roundTo(TimeUnit.NANOSECONDS);
                failedInternalNetworkInputDataSize += taskStats.getInternalNetworkInputDataSize().toBytes();
                failedInternalNetworkInputPositions += taskStats.getInternalNetworkInputPositions();
                failedRawInputDataSize += taskStats.getRawInputDataSize().toBytes();
                failedRawInputPositions += taskStats.getRawInputPositions();
                failedProcessedInputDataSize += taskStats.getProcessedInputDataSize().toBytes();
                failedProcessedInputPositions += taskStats.getProcessedInputPositions();
                failedInputBlockedTime += taskStats.getInputBlockedTime().roundTo(TimeUnit.NANOSECONDS);
                failedOutputDataSize += taskStats.getOutputDataSize().toBytes();
                failedOutputPositions += taskStats.getOutputPositions();
                failedPhysicalWrittenDataSize += taskStats.getPhysicalWrittenDataSize().toBytes();
                failedOutputBlockedTime += taskStats.getOutputBlockedTime().roundTo(TimeUnit.NANOSECONDS);
            }
            fullGcCount += taskStats.getFullGcCount();
            fullGcTaskCount += taskStats.getFullGcCount() > 0 ? 1 : 0;
            int gcSec = Math.toIntExact(taskStats.getFullGcTime().roundTo(TimeUnit.SECONDS));
            totalFullGcSec += gcSec;
            minFullGcSec = Math.min(minFullGcSec, gcSec);
            maxFullGcSec = Math.max(maxFullGcSec, gcSec);
            int taskOperatorSummaries = 0;
            for (PipelineStats pipeline : taskStats.getPipelines()) {
                taskOperatorSummaries += pipeline.getOperatorSummaries().size();
            }
            maxTaskOperatorSummaries = Math.max(taskOperatorSummaries, maxTaskOperatorSummaries);
        }
        ImmutableList operatorStats = maxTaskOperatorSummaries == 0 ? ImmutableList.of() : StageStateMachine.combineTaskOperatorSummaries((List<TaskInfo>)taskInfos, maxTaskOperatorSummaries);
        StageStats stageStats = new StageStats(this.schedulingComplete.get(), this.getSplitDistribution.snapshot(), totalTasks, runningTasks, completedTasks, failedTasks, totalDrivers, queuedDrivers, runningDrivers, blockedDrivers, completedDrivers, cumulativeUserMemory, failedCumulativeUserMemory, DataSize.succinctBytes((long)userMemoryReservation), DataSize.succinctBytes((long)revocableMemoryReservation), DataSize.succinctBytes((long)totalMemoryReservation), DataSize.succinctBytes((long)peakUserMemoryReservation), DataSize.succinctBytes((long)peakRevocableMemoryReservation), Duration.succinctDuration((double)totalScheduledTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)failedScheduledTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)totalCpuTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)failedCpuTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)totalBlockedTime, (TimeUnit)TimeUnit.NANOSECONDS), fullyBlocked && runningTasks > 0, blockedReasons, DataSize.succinctBytes((long)physicalInputDataSize), DataSize.succinctBytes((long)failedPhysicalInputDataSize), physicalInputPositions, failedPhysicalInputPositions, Duration.succinctDuration((double)physicalInputReadTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)failedPhysicalInputReadTime, (TimeUnit)TimeUnit.NANOSECONDS), DataSize.succinctBytes((long)internalNetworkInputDataSize), DataSize.succinctBytes((long)failedInternalNetworkInputDataSize), internalNetworkInputPositions, failedInternalNetworkInputPositions, DataSize.succinctBytes((long)rawInputDataSize), DataSize.succinctBytes((long)failedRawInputDataSize), rawInputPositions, failedRawInputPositions, DataSize.succinctBytes((long)processedInputDataSize), DataSize.succinctBytes((long)failedProcessedInputDataSize), processedInputPositions, failedProcessedInputPositions, Duration.succinctDuration((double)inputBlockedTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)failedInputBlockedTime, (TimeUnit)TimeUnit.NANOSECONDS), DataSize.succinctBytes((long)bufferedDataSize), TDigestHistogram.merge((List)bufferUtilizationHistograms.build()), DataSize.succinctBytes((long)outputDataSize), DataSize.succinctBytes((long)failedOutputDataSize), outputPositions, failedOutputPositions, Duration.succinctDuration((double)outputBlockedTime, (TimeUnit)TimeUnit.NANOSECONDS), Duration.succinctDuration((double)failedOutputBlockedTime, (TimeUnit)TimeUnit.NANOSECONDS), DataSize.succinctBytes((long)physicalWrittenDataSize), DataSize.succinctBytes((long)failedPhysicalWrittenDataSize), new StageGcStatistics(this.stageId.getId(), totalTasks, fullGcTaskCount, minFullGcSec, maxFullGcSec, totalFullGcSec, (int)(1.0 * (double)totalFullGcSec / (double)fullGcCount)), (List<OperatorStats>)operatorStats);
        ExecutionFailureInfo failureInfo = null;
        if (state == StageState.FAILED) {
            failureInfo = this.failureCause.get();
        }
        return new StageInfo(this.stageId, state, this.fragment, this.fragment.getPartitioning().isCoordinatorOnly(), this.fragment.getTypes(), stageStats, (List<TaskInfo>)taskInfos, (List<StageInfo>)ImmutableList.of(), this.tables, failureInfo);
    }

    private static List<OperatorStats> combineTaskOperatorSummaries(List<TaskInfo> taskInfos, int maxTaskOperatorSummaries) {
        Long2ObjectOpenHashMap pipelineAndOperatorToStats = new Long2ObjectOpenHashMap(maxTaskOperatorSummaries);
        int taskInfoCount = taskInfos.size();
        LongFunction<List> statsListCreator = key -> new ArrayList(taskInfoCount);
        for (TaskInfo taskInfo : taskInfos) {
            for (PipelineStats pipeline : taskInfo.getStats().getPipelines()) {
                long pipelineKeyMask = Integer.toUnsignedLong(pipeline.getPipelineId()) << 32;
                for (OperatorStats operator : pipeline.getOperatorSummaries()) {
                    long combinedKey = pipelineKeyMask | Integer.toUnsignedLong(operator.getOperatorId());
                    ((List)pipelineAndOperatorToStats.computeIfAbsent(combinedKey, statsListCreator)).add(operator);
                }
            }
        }
        ImmutableList.Builder operatorStatsBuilder = ImmutableList.builderWithExpectedSize((int)pipelineAndOperatorToStats.size());
        for (List operators : pipelineAndOperatorToStats.values()) {
            OperatorStats stats = (OperatorStats)operators.get(0);
            if (operators.size() > 1) {
                stats = stats.add(operators.subList(1, operators.size()));
            }
            operatorStatsBuilder.add((Object)stats);
        }
        return operatorStatsBuilder.build();
    }

    public void recordGetSplitTime(long startNanos) {
        long elapsedNanos = System.nanoTime() - startNanos;
        this.getSplitDistribution.add(elapsedNanos);
        this.scheduledStats.getGetSplitTime().add((double)elapsedNanos, TimeUnit.NANOSECONDS);
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("stageId", (Object)this.stageId).add("stageState", this.stageState).toString();
    }
}

