/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.cassandra;

import com.datastax.oss.driver.api.core.CqlIdentifier;
import com.datastax.oss.driver.api.core.ProtocolVersion;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.api.core.data.GettableByIndex;
import com.datastax.oss.driver.api.core.data.TupleValue;
import com.datastax.oss.driver.api.core.data.UdtValue;
import com.datastax.oss.driver.api.core.type.DataType;
import com.datastax.oss.driver.api.core.type.ListType;
import com.datastax.oss.driver.api.core.type.MapType;
import com.datastax.oss.driver.api.core.type.SetType;
import com.datastax.oss.driver.api.core.type.TupleType;
import com.datastax.oss.driver.api.core.type.UserDefinedType;
import com.datastax.oss.protocol.internal.util.Bytes;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.net.InetAddresses;
import com.google.inject.Inject;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.cassandra.CassandraType;
import io.trino.plugin.cassandra.CassandraTypes;
import io.trino.plugin.cassandra.util.CassandraCqlUtils;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.RowValueBuilder;
import io.trino.spi.predicate.NullableValue;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.Timestamps;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.TypeSignature;
import io.trino.spi.type.TypeSignatureParameter;
import io.trino.spi.type.TypeUtils;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Supplier;

public class CassandraTypeManager {
    private final Type ipAddressType;

    @Inject
    public CassandraTypeManager(TypeManager typeManager) {
        Objects.requireNonNull(typeManager, "typeManager is null");
        this.ipAddressType = typeManager.getType(new TypeSignature("ipaddress", new TypeSignatureParameter[0]));
    }

    public Optional<CassandraType> toCassandraType(DataType dataType) {
        switch (dataType.getProtocolCode()) {
            case 1: {
                return Optional.of(CassandraTypes.ASCII);
            }
            case 2: {
                return Optional.of(CassandraTypes.BIGINT);
            }
            case 3: {
                return Optional.of(CassandraTypes.BLOB);
            }
            case 4: {
                return Optional.of(CassandraTypes.BOOLEAN);
            }
            case 5: {
                return Optional.of(CassandraTypes.COUNTER);
            }
            case 0: {
                return Optional.of(CassandraTypes.CUSTOM);
            }
            case 17: {
                return Optional.of(CassandraTypes.DATE);
            }
            case 6: {
                return Optional.of(CassandraTypes.DECIMAL);
            }
            case 7: {
                return Optional.of(CassandraTypes.DOUBLE);
            }
            case 8: {
                return Optional.of(CassandraTypes.FLOAT);
            }
            case 16: {
                return Optional.of(new CassandraType(CassandraType.Kind.INET, this.ipAddressType));
            }
            case 9: {
                return Optional.of(CassandraTypes.INT);
            }
            case 32: {
                return Optional.of(CassandraTypes.LIST);
            }
            case 33: {
                return Optional.of(CassandraTypes.MAP);
            }
            case 34: {
                return Optional.of(CassandraTypes.SET);
            }
            case 19: {
                return Optional.of(CassandraTypes.SMALLINT);
            }
            case 18: {
                return Optional.of(CassandraTypes.TIME);
            }
            case 11: {
                return Optional.of(CassandraTypes.TIMESTAMP);
            }
            case 15: {
                return Optional.of(CassandraTypes.TIMEUUID);
            }
            case 20: {
                return Optional.of(CassandraTypes.TINYINT);
            }
            case 49: {
                return this.createTypeForTuple(dataType);
            }
            case 48: {
                return this.createTypeForUserType(dataType);
            }
            case 12: {
                return Optional.of(CassandraTypes.UUID);
            }
            case 13: {
                return Optional.of(CassandraTypes.VARCHAR);
            }
            case 14: {
                return Optional.of(CassandraTypes.VARINT);
            }
        }
        return Optional.empty();
    }

    private Optional<CassandraType> createTypeForTuple(DataType dataType) {
        TupleType tupleType = (TupleType)dataType;
        List argumentTypesOptionals = (List)tupleType.getComponentTypes().stream().map(componentType -> this.toCassandraType((DataType)componentType)).collect(ImmutableList.toImmutableList());
        if (argumentTypesOptionals.stream().anyMatch(Optional::isEmpty)) {
            return Optional.empty();
        }
        List argumentTypes = (List)argumentTypesOptionals.stream().map(Optional::get).collect(ImmutableList.toImmutableList());
        RowType trinoType = RowType.anonymous((List)((List)argumentTypes.stream().map(CassandraType::getTrinoType).collect(ImmutableList.toImmutableList())));
        return Optional.of(new CassandraType(CassandraType.Kind.TUPLE, (Type)trinoType, argumentTypes));
    }

    private Optional<CassandraType> createTypeForUserType(DataType dataType) {
        UserDefinedType userDefinedType = (UserDefinedType)dataType;
        ImmutableMap.Builder argumentTypes = ImmutableMap.builder();
        List fieldNames = userDefinedType.getFieldNames();
        List fieldTypes = userDefinedType.getFieldTypes();
        if (fieldNames.size() != fieldTypes.size()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, String.format("Mismatch between the number of field names (%s) and the number of field types (%s) for the data type %s", fieldNames.size(), fieldTypes.size(), dataType));
        }
        for (int i = 0; i < fieldNames.size(); ++i) {
            Optional<CassandraType> cassandraType = this.toCassandraType((DataType)fieldTypes.get(i));
            if (cassandraType.isEmpty()) {
                return Optional.empty();
            }
            argumentTypes.put((Object)((CqlIdentifier)fieldNames.get(i)).toString(), (Object)cassandraType.get());
        }
        RowType trinoType = RowType.from((List)((List)argumentTypes.buildOrThrow().entrySet().stream().map(field -> new RowType.Field(Optional.of((String)field.getKey()), ((CassandraType)field.getValue()).getTrinoType())).collect(ImmutableList.toImmutableList())));
        return Optional.of(new CassandraType(CassandraType.Kind.UDT, (Type)trinoType, (List<CassandraType>)ImmutableList.copyOf((Collection)argumentTypes.buildOrThrow().values())));
    }

    public NullableValue getColumnValue(CassandraType cassandraType, Row row, int position) {
        return this.getColumnValue(cassandraType, (GettableByIndex)row, position, () -> row.getColumnDefinitions().get(position).getType());
    }

    public NullableValue getColumnValue(CassandraType cassandraType, GettableByIndex row, int position, Supplier<DataType> dataTypeSupplier) {
        Type trinoType = cassandraType.getTrinoType();
        if (row.isNull(position)) {
            return NullableValue.asNull((Type)trinoType);
        }
        switch (cassandraType.getKind()) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return NullableValue.of((Type)trinoType, (Object)Slices.utf8Slice((String)row.getString(position)));
            }
            case INT: {
                return NullableValue.of((Type)trinoType, (Object)row.getInt(position));
            }
            case SMALLINT: {
                return NullableValue.of((Type)trinoType, (Object)row.getShort(position));
            }
            case TINYINT: {
                return NullableValue.of((Type)trinoType, (Object)row.getByte(position));
            }
            case BIGINT: 
            case COUNTER: {
                return NullableValue.of((Type)trinoType, (Object)row.getLong(position));
            }
            case BOOLEAN: {
                return NullableValue.of((Type)trinoType, (Object)row.getBoolean(position));
            }
            case DOUBLE: {
                return NullableValue.of((Type)trinoType, (Object)row.getDouble(position));
            }
            case FLOAT: {
                return NullableValue.of((Type)trinoType, (Object)Float.floatToRawIntBits(row.getFloat(position)));
            }
            case DECIMAL: {
                return NullableValue.of((Type)trinoType, (Object)row.getBigDecimal(position).doubleValue());
            }
            case UUID: 
            case TIMEUUID: {
                return NullableValue.of((Type)trinoType, (Object)UuidType.javaUuidToTrinoUuid((UUID)row.getUuid(position)));
            }
            case TIME: {
                return NullableValue.of((Type)trinoType, (Object)(row.getLocalTime(position).toNanoOfDay() * 1000L));
            }
            case TIMESTAMP: {
                return NullableValue.of((Type)trinoType, (Object)DateTimeEncoding.packDateTimeWithZone((long)row.getInstant(position).toEpochMilli(), (TimeZoneKey)TimeZoneKey.UTC_KEY));
            }
            case DATE: {
                return NullableValue.of((Type)trinoType, (Object)row.getLocalDate(position).toEpochDay());
            }
            case INET: {
                return NullableValue.of((Type)trinoType, (Object)CassandraTypeManager.castFromVarcharToIpAddress(Slices.utf8Slice((String)InetAddresses.toAddrString((InetAddress)row.getInetAddress(position)))));
            }
            case VARINT: {
                return NullableValue.of((Type)trinoType, (Object)Slices.utf8Slice((String)row.getBigInteger(position).toString()));
            }
            case BLOB: 
            case CUSTOM: {
                return NullableValue.of((Type)trinoType, (Object)Slices.wrappedBuffer((ByteBuffer)row.getBytesUnsafe(position)));
            }
            case SET: {
                return NullableValue.of((Type)trinoType, (Object)Slices.utf8Slice((String)this.buildArrayValueFromSetType(row, position, dataTypeSupplier.get())));
            }
            case LIST: {
                return NullableValue.of((Type)trinoType, (Object)Slices.utf8Slice((String)this.buildArrayValueFromListType(row, position, dataTypeSupplier.get())));
            }
            case MAP: {
                return NullableValue.of((Type)trinoType, (Object)Slices.utf8Slice((String)this.buildMapValue(row, position, dataTypeSupplier.get())));
            }
            case TUPLE: {
                return NullableValue.of((Type)trinoType, (Object)this.buildTupleValue(cassandraType, row, position));
            }
            case UDT: {
                return NullableValue.of((Type)trinoType, (Object)this.buildUserTypeValue(cassandraType, row, position));
            }
        }
        throw new IllegalStateException("Handling of type " + this + " is not implemented");
    }

    private String buildMapValue(GettableByIndex row, int position, DataType dataType) {
        Preconditions.checkArgument((boolean)(dataType instanceof MapType), (String)"Expected to deal with an instance of %s class, got: %s", MapType.class, (Object)dataType);
        MapType mapType = (MapType)dataType;
        return this.buildMapValue((Map)row.getObject(position), mapType.getKeyType(), mapType.getValueType());
    }

    private String buildMapValue(Map<?, ?> cassandraMap, DataType keyType, DataType valueType) {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        for (Map.Entry<?, ?> entry : cassandraMap.entrySet()) {
            if (sb.length() > 1) {
                sb.append(",");
            }
            sb.append(this.objectToJson(entry.getKey(), keyType));
            sb.append(":");
            sb.append(this.objectToJson(entry.getValue(), valueType));
        }
        sb.append("}");
        return sb.toString();
    }

    private String buildArrayValueFromSetType(GettableByIndex row, int position, DataType type) {
        Preconditions.checkArgument((boolean)(type instanceof SetType), (String)"Expected to deal with an instance of %s class, got: %s", SetType.class, (Object)type);
        SetType setType = (SetType)type;
        return this.buildArrayValue((Collection)row.getObject(position), setType.getElementType());
    }

    private String buildArrayValueFromListType(GettableByIndex row, int position, DataType type) {
        Preconditions.checkArgument((boolean)(type instanceof ListType), (String)"Expected to deal with an instance of %s class, got: %s", ListType.class, (Object)type);
        ListType listType = (ListType)type;
        return this.buildArrayValue((Collection)row.getObject(position), listType.getElementType());
    }

    @VisibleForTesting
    String buildArrayValue(Collection<?> cassandraCollection, DataType elementType) {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (Object value : cassandraCollection) {
            if (sb.length() > 1) {
                sb.append(",");
            }
            sb.append(this.objectToJson(value, elementType));
        }
        sb.append("]");
        return sb.toString();
    }

    private Block buildTupleValue(CassandraType type, GettableByIndex row, int position) {
        Verify.verify((type.getKind() == CassandraType.Kind.TUPLE ? 1 : 0) != 0, (String)"Not a TUPLE type", (Object[])new Object[0]);
        TupleValue tupleValue = row.getTupleValue(position);
        return RowValueBuilder.buildRowValue((RowType)((RowType)type.getTrinoType()), fieldBuilders -> {
            int tuplePosition = 0;
            List<CassandraType> argumentTypes = type.getArgumentTypes();
            for (int i = 0; i < argumentTypes.size(); ++i) {
                CassandraType argumentType = argumentTypes.get(i);
                BlockBuilder fieldBuilder = (BlockBuilder)fieldBuilders.get(i);
                int finalTuplePosition = tuplePosition;
                NullableValue value = this.getColumnValue(argumentType, (GettableByIndex)tupleValue, tuplePosition, () -> (DataType)tupleValue.getType().getComponentTypes().get(finalTuplePosition));
                TypeUtils.writeNativeValue((Type)argumentType.getTrinoType(), (BlockBuilder)fieldBuilder, (Object)value.getValue());
                ++tuplePosition;
            }
        });
    }

    private Block buildUserTypeValue(CassandraType type, GettableByIndex row, int position) {
        Verify.verify((type.getKind() == CassandraType.Kind.UDT ? 1 : 0) != 0, (String)"Not a user defined type: %s", (Object)((Object)type.getKind()));
        UdtValue udtValue = row.getUdtValue(position);
        return RowValueBuilder.buildRowValue((RowType)((RowType)type.getTrinoType()), fieldBuilders -> {
            int tuplePosition = 0;
            List udtTypeFieldTypes = udtValue.getType().getFieldTypes();
            List<CassandraType> argumentTypes = type.getArgumentTypes();
            for (int i = 0; i < argumentTypes.size(); ++i) {
                CassandraType argumentType = argumentTypes.get(i);
                BlockBuilder fieldBuilder = (BlockBuilder)fieldBuilders.get(i);
                int finalTuplePosition = tuplePosition;
                NullableValue value = this.getColumnValue(argumentType, (GettableByIndex)udtValue, tuplePosition, () -> (DataType)udtTypeFieldTypes.get(finalTuplePosition));
                TypeUtils.writeNativeValue((Type)argumentType.getTrinoType(), (BlockBuilder)fieldBuilder, (Object)value.getValue());
                ++tuplePosition;
            }
        });
    }

    public String getColumnValueForCql(CassandraType type, Row row, int position) {
        if (row.isNull(position)) {
            return null;
        }
        switch (type.getKind()) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return CassandraCqlUtils.quoteStringLiteral(row.getString(position));
            }
            case INT: {
                return Integer.toString(row.getInt(position));
            }
            case SMALLINT: {
                return Short.toString(row.getShort(position));
            }
            case TINYINT: {
                return Byte.toString(row.getByte(position));
            }
            case BIGINT: 
            case COUNTER: {
                return Long.toString(row.getLong(position));
            }
            case BOOLEAN: {
                return Boolean.toString(row.getBool(position));
            }
            case DOUBLE: {
                return Double.toString(row.getDouble(position));
            }
            case FLOAT: {
                return Float.toString(row.getFloat(position));
            }
            case DECIMAL: {
                return row.getBigDecimal(position).toString();
            }
            case UUID: 
            case TIMEUUID: {
                return row.getUuid(position).toString();
            }
            case TIME: {
                return CassandraCqlUtils.quoteStringLiteral(row.getLocalTime(position).toString());
            }
            case TIMESTAMP: {
                return Long.toString(row.getInstant(position).toEpochMilli());
            }
            case DATE: {
                return CassandraCqlUtils.quoteStringLiteral(row.getLocalDate(position).toString());
            }
            case INET: {
                return CassandraCqlUtils.quoteStringLiteral(InetAddresses.toAddrString((InetAddress)row.getInetAddress(position)));
            }
            case VARINT: {
                return row.getBigInteger(position).toString();
            }
            case BLOB: 
            case CUSTOM: {
                return Bytes.toHexString((ByteBuffer)row.getBytesUnsafe(position));
            }
        }
        throw new IllegalStateException("Handling of type " + this + " is not implemented");
    }

    public String toCqlLiteral(CassandraType type, Object trinoNativeValue) {
        CassandraType.Kind kind = type.getKind();
        if (kind == CassandraType.Kind.DATE) {
            LocalDate date = LocalDate.ofEpochDay(Math.toIntExact((Long)trinoNativeValue));
            return CassandraCqlUtils.quoteStringLiteral(date.toString());
        }
        if (kind == CassandraType.Kind.TIME) {
            LocalTime time = LocalTime.ofNanoOfDay(Timestamps.roundDiv((long)((Long)trinoNativeValue), (long)1000L));
            return CassandraCqlUtils.quoteStringLiteral(time.toString());
        }
        if (kind == CassandraType.Kind.TIMESTAMP) {
            return String.valueOf(DateTimeEncoding.unpackMillisUtc((long)((Long)trinoNativeValue)));
        }
        String value = trinoNativeValue instanceof Slice ? ((Slice)trinoNativeValue).toStringUtf8() : trinoNativeValue.toString();
        switch (kind) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return CassandraCqlUtils.quoteStringLiteral(value);
            }
            case INET: {
                return CassandraCqlUtils.quoteStringLiteral(value.substring(1));
            }
        }
        return value;
    }

    private String objectToJson(Object cassandraValue, DataType dataType) {
        CassandraType cassandraType = this.toCassandraType(dataType).orElseThrow(() -> new IllegalStateException("Unsupported type: " + dataType));
        switch (cassandraType.getKind()) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: 
            case UUID: 
            case TIMEUUID: 
            case TIME: 
            case TIMESTAMP: 
            case DATE: 
            case INET: 
            case VARINT: 
            case TUPLE: {
                return CassandraCqlUtils.quoteStringLiteralForJson(cassandraValue.toString());
            }
            case UDT: {
                return CassandraCqlUtils.quoteStringLiteralForJson(((UdtValue)cassandraValue).getFormattedContents());
            }
            case BLOB: 
            case CUSTOM: {
                return CassandraCqlUtils.quoteStringLiteralForJson(Bytes.toHexString((ByteBuffer)((ByteBuffer)cassandraValue)));
            }
            case INT: 
            case SMALLINT: 
            case TINYINT: 
            case BIGINT: 
            case COUNTER: 
            case BOOLEAN: 
            case DOUBLE: 
            case FLOAT: 
            case DECIMAL: {
                return cassandraValue.toString();
            }
            case LIST: {
                Preconditions.checkArgument((boolean)(dataType instanceof ListType), (String)"Expected to deal with an instance of %s class, got: %s", ListType.class, (Object)dataType);
                ListType listType = (ListType)dataType;
                return this.buildArrayValue((Collection)cassandraValue, listType.getElementType());
            }
            case SET: {
                Preconditions.checkArgument((boolean)(dataType instanceof SetType), (String)"Expected to deal with an instance of %s class, got: %s", SetType.class, (Object)dataType);
                SetType setType = (SetType)dataType;
                return this.buildArrayValue((Collection)cassandraValue, setType.getElementType());
            }
            case MAP: {
                Preconditions.checkArgument((boolean)(dataType instanceof MapType), (String)"Expected to deal with an instance of %s class, got: %s", MapType.class, (Object)dataType);
                MapType mapType = (MapType)dataType;
                return this.buildMapValue((Map)cassandraValue, mapType.getKeyType(), mapType.getValueType());
            }
        }
        throw new IllegalStateException("Unsupported type: " + cassandraType);
    }

    public Object getJavaValue(CassandraType.Kind kind, Object trinoNativeValue) {
        switch (kind) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return ((Slice)trinoNativeValue).toStringUtf8();
            }
            case BIGINT: 
            case COUNTER: 
            case BOOLEAN: 
            case DOUBLE: {
                return trinoNativeValue;
            }
            case INET: {
                try {
                    return InetAddress.getByAddress(((Slice)trinoNativeValue).getBytes());
                }
                catch (UnknownHostException e) {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_CAST_ARGUMENT, "Invalid IP address binary length: " + ((Slice)trinoNativeValue).length(), (Throwable)e);
                }
            }
            case INT: 
            case SMALLINT: 
            case TINYINT: {
                return ((Long)trinoNativeValue).intValue();
            }
            case FLOAT: {
                return Float.valueOf(Float.intBitsToFloat(((Long)trinoNativeValue).intValue()));
            }
            case DECIMAL: {
                return new BigDecimal(trinoNativeValue.toString());
            }
            case TIME: {
                return LocalTime.ofNanoOfDay(Timestamps.roundDiv((long)((Long)trinoNativeValue), (long)1000L));
            }
            case TIMESTAMP: {
                return Instant.ofEpochMilli(DateTimeEncoding.unpackMillisUtc((long)((Long)trinoNativeValue)));
            }
            case DATE: {
                return LocalDate.ofEpochDay(((Long)trinoNativeValue).intValue());
            }
            case UUID: 
            case TIMEUUID: {
                return UuidType.trinoUuidToJavaUuid((Slice)((Slice)trinoNativeValue));
            }
            case BLOB: 
            case CUSTOM: 
            case TUPLE: 
            case UDT: {
                return ((Slice)trinoNativeValue).toStringUtf8();
            }
            case VARINT: {
                return new BigInteger(((Slice)trinoNativeValue).toStringUtf8());
            }
        }
        throw new IllegalStateException("Back conversion not implemented for " + this);
    }

    public boolean isSupportedPartitionKey(CassandraType.Kind kind) {
        switch (kind) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: 
            case INT: 
            case SMALLINT: 
            case TINYINT: 
            case BIGINT: 
            case BOOLEAN: 
            case DOUBLE: 
            case FLOAT: 
            case DECIMAL: 
            case UUID: 
            case TIMEUUID: 
            case TIME: 
            case TIMESTAMP: 
            case DATE: 
            case INET: {
                return true;
            }
        }
        return false;
    }

    public boolean isFullySupported(DataType dataType) {
        if (this.toCassandraType(dataType).isEmpty()) {
            return false;
        }
        if (dataType instanceof UserDefinedType) {
            UserDefinedType userDefinedType = (UserDefinedType)dataType;
            return userDefinedType.getFieldTypes().stream().allMatch(this::isFullySupported);
        }
        if (dataType instanceof MapType) {
            MapType mapType = (MapType)dataType;
            return Arrays.stream(new DataType[]{mapType.getKeyType(), mapType.getValueType()}).allMatch(this::isFullySupported);
        }
        if (dataType instanceof ListType) {
            ListType listType = (ListType)dataType;
            return this.isFullySupported(listType.getElementType());
        }
        if (dataType instanceof TupleType) {
            TupleType tupleType = (TupleType)dataType;
            return tupleType.getComponentTypes().stream().allMatch(this::isFullySupported);
        }
        if (dataType instanceof SetType) {
            SetType setType = (SetType)dataType;
            return this.isFullySupported(setType.getElementType());
        }
        return true;
    }

    public CassandraType toCassandraType(Type type, ProtocolVersion protocolVersion) {
        if (type.equals(BooleanType.BOOLEAN)) {
            return CassandraTypes.BOOLEAN;
        }
        if (type.equals(BigintType.BIGINT)) {
            return CassandraTypes.BIGINT;
        }
        if (type.equals(IntegerType.INTEGER)) {
            return CassandraTypes.INT;
        }
        if (type.equals(SmallintType.SMALLINT)) {
            return CassandraTypes.SMALLINT;
        }
        if (type.equals(TinyintType.TINYINT)) {
            return CassandraTypes.TINYINT;
        }
        if (type.equals(DoubleType.DOUBLE)) {
            return CassandraTypes.DOUBLE;
        }
        if (type.equals(RealType.REAL)) {
            return CassandraTypes.FLOAT;
        }
        if (type instanceof VarcharType) {
            return CassandraTypes.TEXT;
        }
        if (type.equals(DateType.DATE)) {
            return protocolVersion.getCode() <= ProtocolVersion.V3.getCode() ? CassandraTypes.TEXT : CassandraTypes.DATE;
        }
        if (type.equals(VarbinaryType.VARBINARY)) {
            return CassandraTypes.BLOB;
        }
        if (type.equals(TimeType.TIME_NANOS)) {
            return CassandraTypes.TIME;
        }
        if (type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MILLIS)) {
            return CassandraTypes.TIMESTAMP;
        }
        if (type.equals(UuidType.UUID)) {
            return CassandraTypes.UUID;
        }
        if (type.equals(this.ipAddressType)) {
            return new CassandraType(CassandraType.Kind.INET, this.ipAddressType);
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported type: " + type);
    }

    public boolean isIpAddressType(Type type) {
        return type.equals(this.ipAddressType);
    }

    private static Slice castFromVarcharToIpAddress(Slice slice) {
        byte[] bytes;
        byte[] address;
        try {
            address = InetAddresses.forString((String)slice.toStringUtf8()).getAddress();
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_CAST_ARGUMENT, "Cannot cast value to IPADDRESS: " + slice.toStringUtf8());
        }
        if (address.length == 4) {
            bytes = new byte[16];
            bytes[10] = -1;
            bytes[11] = -1;
            System.arraycopy(address, 0, bytes, 12, 4);
        } else if (address.length == 16) {
            bytes = address;
        } else {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Invalid InetAddress length: " + address.length);
        }
        return Slices.wrappedBuffer((byte[])bytes);
    }
}

