/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kudu.client;

import com.stumbleupon.async.Callback;
import java.util.Set;
import javax.annotation.Nonnull;
import org.apache.kudu.WireProtocol;
import org.apache.kudu.client.AsyncKuduClient;
import org.apache.kudu.client.CallResponse;
import org.apache.kudu.client.ConnectToMasterRequest;
import org.apache.kudu.client.Connection;
import org.apache.kudu.client.InvalidAuthnTokenException;
import org.apache.kudu.client.InvalidAuthzTokenException;
import org.apache.kudu.client.KuduException;
import org.apache.kudu.client.KuduMetrics;
import org.apache.kudu.client.KuduRpc;
import org.apache.kudu.client.NonRecoverableException;
import org.apache.kudu.client.RecoverableException;
import org.apache.kudu.client.RemoteTablet;
import org.apache.kudu.client.RequestTracker;
import org.apache.kudu.client.RpcOutboundMessage;
import org.apache.kudu.client.RpcTraceFrame;
import org.apache.kudu.client.Status;
import org.apache.kudu.master.Master;
import org.apache.kudu.rpc.RpcHeader;
import org.apache.kudu.shaded.com.google.common.base.Preconditions;
import org.apache.kudu.shaded.com.google.protobuf.GeneratedMessageV3;
import org.apache.kudu.shaded.com.google.protobuf.Message;
import org.apache.kudu.transactions.TxnManager;
import org.apache.kudu.tserver.Tserver;
import org.apache.kudu.util.Pair;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
class RpcProxy {
    private static final Logger LOG = LoggerFactory.getLogger(RpcProxy.class);
    private static int staticNumFail = 0;
    private static Exception staticException = null;
    @Nonnull
    private final AsyncKuduClient client;
    @Nonnull
    private final Connection connection;

    RpcProxy(AsyncKuduClient client, Connection connection) {
        this.client = Preconditions.checkNotNull(client);
        this.connection = Preconditions.checkNotNull(connection);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    static void failNextRpcs(int numFail, Exception exception) {
        Preconditions.checkNotNull(exception);
        staticNumFail = numFail;
        staticException = exception;
    }

    <R> void sendRpc(KuduRpc<R> rpc) {
        RpcProxy.sendRpc(this.client, this.connection, rpc);
    }

    static <R> void sendRpc(final AsyncKuduClient client, final Connection connection, final KuduRpc<R> rpc) {
        KuduMetrics.counter(KuduMetrics.RPC_REQUESTS_METRIC, RpcProxy.rpcTags(client, connection, rpc)).increment();
        if (rpc.attempt > 1) {
            KuduMetrics.counter(KuduMetrics.RPC_RETRIES_METRIC, RpcProxy.rpcTags(client, connection, rpc)).increment();
        }
        try {
            Set<RpcHeader.RpcFeatureFlag> features;
            if (staticNumFail > 0) {
                --staticNumFail;
                LOG.warn("Forcing a failure on sendRpc: {}", rpc);
                throw staticException;
            }
            if (!rpc.getRequiredFeatures().isEmpty() && (features = connection.getPeerFeatures()) != null && !features.contains(RpcHeader.RpcFeatureFlag.APPLICATION_FEATURE_FLAGS)) {
                throw new NonRecoverableException(Status.NotSupported("the server does not support the APPLICATION_FEATURE_FLAGS RPC feature"));
            }
            Preconditions.checkArgument(rpc.hasDeferred());
            rpc.addTrace(new RpcTraceFrame.RpcTraceFrameBuilder(rpc.method(), RpcTraceFrame.Action.SEND_TO_SERVER).serverInfo(connection.getServerInfo()).build());
            connection.enqueueMessage(RpcProxy.rpcToMessage(client, rpc), new Callback<Void, Connection.CallResponseInfo>(){

                @Override
                public Void call(Connection.CallResponseInfo callResponseInfo) throws Exception {
                    try {
                        RpcProxy.responseReceived(client, connection, rpc, callResponseInfo.response, callResponseInfo.exception);
                    }
                    catch (Exception e) {
                        rpc.errback(e);
                    }
                    return null;
                }
            });
        }
        catch (RecoverableException e) {
            client.handleRetryableError(rpc, e);
        }
        catch (Exception e) {
            rpc.errback(e);
        }
    }

    private static <R> RpcOutboundMessage rpcToMessage(AsyncKuduClient client, KuduRpc<R> rpc) {
        RpcHeader.RequestHeader.Builder headerBuilder = RpcHeader.RequestHeader.newBuilder().addAllRequiredFeatureFlags(rpc.getRequiredFeatures()).setRemoteMethod(RpcHeader.RemoteMethodPB.newBuilder().setServiceName(rpc.serviceName()).setMethodName(rpc.method()));
        if (rpc.needsAuthzToken()) {
            rpc.bindAuthzToken(client.getAuthzToken(rpc.getTable().getTableId()));
        }
        Message reqPB = rpc.createRequestPB();
        if (rpc.timeoutTracker.hasTimeout()) {
            headerBuilder.setTimeoutMillis((int)rpc.timeoutTracker.getMillisBeforeTimeout());
        }
        if (rpc.isRequestTracked()) {
            RpcHeader.RequestIdPB.Builder requestIdBuilder = RpcHeader.RequestIdPB.newBuilder();
            RequestTracker requestTracker = client.getRequestTracker();
            if (rpc.getSequenceId() == -1L) {
                rpc.setSequenceId(requestTracker.newSeqNo());
            }
            requestIdBuilder.setClientId(requestTracker.getClientId());
            requestIdBuilder.setSeqNo(rpc.getSequenceId());
            requestIdBuilder.setAttemptNo(rpc.attempt);
            requestIdBuilder.setFirstIncompleteSeqNo(requestTracker.firstIncomplete());
            headerBuilder.setRequestId(requestIdBuilder);
        }
        return new RpcOutboundMessage(headerBuilder, reqPB);
    }

    private static <R> void responseReceived(AsyncKuduClient client, Connection connection, KuduRpc<R> rpc, CallResponse response, KuduException ex) {
        long start = System.nanoTime();
        if (LOG.isTraceEnabled()) {
            if (response == null) {
                LOG.trace("{} received null response for RPC {}", (Object)connection.getLogPrefix(), rpc);
            } else {
                RpcHeader.ResponseHeader header = response.getHeader();
                LOG.trace("{} received response with rpcId {}, size {} for RPC {}", new Object[]{connection.getLogPrefix(), header.getCallId(), response.getTotalResponseSize(), rpc});
            }
        }
        KuduMetrics.counter(KuduMetrics.RPC_RESPONSE_METRIC, RpcProxy.rpcTags(client, connection, rpc)).increment();
        RpcTraceFrame.RpcTraceFrameBuilder traceBuilder = new RpcTraceFrame.RpcTraceFrameBuilder(rpc.method(), RpcTraceFrame.Action.RECEIVE_FROM_SERVER).serverInfo(connection.getServerInfo());
        if (ex != null) {
            if (ex instanceof InvalidAuthnTokenException) {
                client.handleInvalidAuthnToken(rpc);
                return;
            }
            if (ex instanceof InvalidAuthzTokenException) {
                client.handleInvalidAuthzToken(rpc, ex);
                return;
            }
            if (ex instanceof RecoverableException) {
                RpcProxy.failOrRetryRpc(client, connection, rpc, (RecoverableException)ex);
                return;
            }
            rpc.addTrace(traceBuilder.callStatus(ex.getStatus()).build());
            rpc.errback(ex);
            return;
        }
        Pair<R, Object> decoded = null;
        KuduException exception = null;
        try {
            decoded = rpc.deserialize(response, connection.getServerInfo().getUuid());
        }
        catch (KuduException e) {
            exception = e;
        }
        catch (Exception e) {
            rpc.addTrace(traceBuilder.build());
            rpc.errback(e);
            return;
        }
        if (decoded != null && decoded.getSecond() != null) {
            GeneratedMessageV3 error;
            if (decoded.getSecond() instanceof Tserver.TabletServerErrorPB) {
                error = (Tserver.TabletServerErrorPB)decoded.getSecond();
                exception = RpcProxy.dispatchTSError(client, connection, rpc, error, traceBuilder);
                if (exception == null) {
                    return;
                }
                decoded = null;
            } else if (decoded.getSecond() instanceof Master.MasterErrorPB) {
                error = (Master.MasterErrorPB)decoded.getSecond();
                exception = RpcProxy.dispatchMasterError(client, connection, rpc, (Master.MasterErrorPB)error, traceBuilder);
                if (exception == null) {
                    return;
                }
                decoded = null;
            } else if (decoded.getSecond() instanceof TxnManager.TxnManagerErrorPB) {
                error = (TxnManager.TxnManagerErrorPB)decoded.getSecond();
                exception = RpcProxy.dispatchTxnManagerError(client, connection, rpc, (TxnManager.TxnManagerErrorPB)error, traceBuilder);
                if (exception == null) {
                    return;
                }
                decoded = null;
            } else {
                rpc.addTrace(traceBuilder.build());
                exception = new NonRecoverableException(Status.NotSupported("unexpected error from server side: " + decoded.getSecond().toString()));
                rpc.errback(exception);
                return;
            }
        }
        try {
            if (decoded != null) {
                Preconditions.checkState(!(decoded.getFirst() instanceof Exception));
                if (client.isStatisticsEnabled()) {
                    rpc.updateStatistics(client.getStatistics(), decoded.getFirst());
                }
                rpc.addTrace(traceBuilder.callStatus(Status.OK()).build());
                rpc.callback(decoded.getFirst());
            } else {
                if (client.isStatisticsEnabled()) {
                    rpc.updateStatistics(client.getStatistics(), null);
                }
                rpc.addTrace(traceBuilder.callStatus(exception.getStatus()).build());
                rpc.errback(exception);
            }
        }
        catch (Exception e) {
            RpcHeader.ResponseHeader header = response.getHeader();
            Preconditions.checkNotNull(header);
            LOG.debug("{} unexpected exception {} while handling call: callId {}, RPC {}", new Object[]{connection.getLogPrefix(), e, header.getCallId(), rpc});
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("------------------<< LEAVING  DECODE <<------------------ time elapsed: {} us", (Object)((System.nanoTime() - start) / 1000L));
        }
    }

    private static KuduException dispatchTSError(AsyncKuduClient client, Connection connection, KuduRpc<?> rpc, Tserver.TabletServerErrorPB error, RpcTraceFrame.RpcTraceFrameBuilder tracer) {
        Tserver.TabletServerErrorPB.Code errCode = error.getCode();
        WireProtocol.AppStatusPB.ErrorCode errStatusCode = error.getStatus().getCode();
        Status status = Status.fromTabletServerErrorPB(error);
        if (errCode == Tserver.TabletServerErrorPB.Code.TABLET_NOT_FOUND || errCode == Tserver.TabletServerErrorPB.Code.TABLET_NOT_RUNNING) {
            client.handleTabletNotFound(rpc, new RecoverableException(status), connection.getServerInfo());
        } else if (errStatusCode == WireProtocol.AppStatusPB.ErrorCode.SERVICE_UNAVAILABLE) {
            client.handleRetryableError(rpc, new RecoverableException(status));
        } else if (errStatusCode == WireProtocol.AppStatusPB.ErrorCode.ILLEGAL_STATE || errStatusCode == WireProtocol.AppStatusPB.ErrorCode.ABORTED) {
            client.handleNotLeader(rpc, new RecoverableException(status), connection.getServerInfo());
        } else {
            return new NonRecoverableException(status);
        }
        rpc.addTrace(tracer.callStatus(status).build());
        return null;
    }

    private static KuduException dispatchMasterError(AsyncKuduClient client, Connection connection, KuduRpc<?> rpc, Master.MasterErrorPB error, RpcTraceFrame.RpcTraceFrameBuilder tracer) {
        WireProtocol.AppStatusPB.ErrorCode code = error.getStatus().getCode();
        Status status = Status.fromMasterErrorPB(error);
        if (error.getCode() == Master.MasterErrorPB.Code.NOT_THE_LEADER) {
            client.handleNotLeader(rpc, new RecoverableException(status), connection.getServerInfo());
        } else if (code == WireProtocol.AppStatusPB.ErrorCode.SERVICE_UNAVAILABLE) {
            if (rpc instanceof ConnectToMasterRequest) {
                return new RecoverableException(status);
            }
            client.handleRetryableError(rpc, new RecoverableException(status));
        } else {
            return new NonRecoverableException(status);
        }
        rpc.addTrace(tracer.callStatus(status).build());
        return null;
    }

    private static KuduException dispatchTxnManagerError(AsyncKuduClient client, Connection connection, KuduRpc<?> rpc, TxnManager.TxnManagerErrorPB pbError, RpcTraceFrame.RpcTraceFrameBuilder tracer) {
        WireProtocol.AppStatusPB.ErrorCode code = pbError.getStatus().getCode();
        Status status = Status.fromTxnManagerErrorPB(pbError);
        if (code != WireProtocol.AppStatusPB.ErrorCode.SERVICE_UNAVAILABLE) {
            return new NonRecoverableException(status);
        }
        client.handleRetryableError(rpc, new RecoverableException(status));
        rpc.addTrace(tracer.callStatus(status).build());
        return null;
    }

    private static void failOrRetryRpc(AsyncKuduClient client, Connection connection, KuduRpc<?> rpc, RecoverableException exception) {
        rpc.addTrace(new RpcTraceFrame.RpcTraceFrameBuilder(rpc.method(), RpcTraceFrame.Action.RECEIVE_FROM_SERVER).serverInfo(connection.getServerInfo()).callStatus(exception.getStatus()).build());
        RemoteTablet tablet = rpc.getTablet();
        if (tablet == null) {
            rpc.errback(exception);
            return;
        }
        if (exception instanceof InvalidAuthnTokenException) {
            client.handleInvalidAuthnToken(rpc);
        } else if (exception instanceof InvalidAuthzTokenException) {
            client.handleInvalidAuthzToken(rpc, exception);
        } else if (exception.getStatus().isServiceUnavailable()) {
            client.handleRetryableError(rpc, exception);
        } else {
            client.handleTabletNotFound(rpc, exception, connection.getServerInfo());
        }
    }

    public String toString() {
        return "RpcProxy@" + this.hashCode() + ", connection=" + this.connection;
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    Connection getConnection() {
        return this.connection;
    }

    private static String[] rpcTags(AsyncKuduClient client, Connection connection, KuduRpc<?> rpc) {
        return new String[]{"service.name", rpc.serviceName(), "method.name", rpc.method(), "server.id", connection.getServerInfo().getUuid(), "client.id", client.getClientId()};
    }
}

