/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.core.query.scheduler;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.LongAccumulator;
import org.apache.pinot.common.exception.QueryException;
import org.apache.pinot.common.metrics.ServerMetrics;
import org.apache.pinot.common.metrics.ServerQueryPhase;
import org.apache.pinot.core.query.executor.QueryExecutor;
import org.apache.pinot.core.query.request.ServerQueryRequest;
import org.apache.pinot.core.query.scheduler.OutOfCapacityException;
import org.apache.pinot.core.query.scheduler.QueryScheduler;
import org.apache.pinot.core.query.scheduler.SchedulerPriorityQueue;
import org.apache.pinot.core.query.scheduler.SchedulerQueryContext;
import org.apache.pinot.core.query.scheduler.resources.QueryExecutorService;
import org.apache.pinot.core.query.scheduler.resources.ResourceManager;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class PriorityScheduler
extends QueryScheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(PriorityScheduler.class);
    protected final SchedulerPriorityQueue _queryQueue;
    @VisibleForTesting
    protected final Semaphore _runningQueriesSemaphore;
    private final int _numRunners;
    @VisibleForTesting
    Thread _scheduler;

    public PriorityScheduler(PinotConfiguration config, ResourceManager resourceManager, QueryExecutor queryExecutor, SchedulerPriorityQueue queue, ServerMetrics metrics, LongAccumulator latestQueryTime) {
        super(config, queryExecutor, resourceManager, metrics, latestQueryTime);
        Preconditions.checkNotNull((Object)queue);
        this._queryQueue = queue;
        this._numRunners = resourceManager.getNumQueryRunnerThreads();
        this._runningQueriesSemaphore = new Semaphore(this._numRunners);
    }

    @Override
    public ListenableFuture<byte[]> submit(ServerQueryRequest queryRequest) {
        if (!this._isRunning) {
            return this.immediateErrorResponse(queryRequest, QueryException.SERVER_SCHEDULER_DOWN_ERROR);
        }
        queryRequest.getTimerContext().startNewPhaseTimer(ServerQueryPhase.SCHEDULER_WAIT);
        SchedulerQueryContext schedQueryContext = new SchedulerQueryContext(queryRequest);
        try {
            this._queryQueue.put(schedQueryContext);
        }
        catch (OutOfCapacityException e) {
            LOGGER.error("Out of capacity for table {}, message: {}", (Object)queryRequest.getTableNameWithType(), (Object)e.getMessage());
            return this.immediateErrorResponse(queryRequest, QueryException.SERVER_OUT_OF_CAPACITY_ERROR);
        }
        return schedQueryContext.getResultFuture();
    }

    @Override
    public void start() {
        super.start();
        this._scheduler = new Thread(new Runnable(){

            @Override
            public void run() {
                while (PriorityScheduler.this._isRunning) {
                    try {
                        PriorityScheduler.this._runningQueriesSemaphore.acquire();
                    }
                    catch (InterruptedException e) {
                        if (!PriorityScheduler.this._isRunning) {
                            LOGGER.info("Shutting down scheduler");
                            break;
                        }
                        LOGGER.error("Interrupt while acquiring semaphore. Exiting.", (Throwable)e);
                        break;
                    }
                    try {
                        final SchedulerQueryContext request = PriorityScheduler.this._queryQueue.take();
                        if (request == null) continue;
                        ServerQueryRequest queryRequest = request.getQueryRequest();
                        final QueryExecutorService executor = PriorityScheduler.this._resourceManager.getExecutorService(queryRequest, request.getSchedulerGroup());
                        ListenableFutureTask<byte[]> queryFutureTask = PriorityScheduler.this.createQueryFutureTask(queryRequest, executor);
                        queryFutureTask.addListener(new Runnable(){

                            @Override
                            public void run() {
                                executor.releaseWorkers();
                                request.getSchedulerGroup().endQuery();
                                PriorityScheduler.this._runningQueriesSemaphore.release();
                                PriorityScheduler.this.checkStopResourceManager();
                                if (!PriorityScheduler.this._isRunning && PriorityScheduler.this._runningQueriesSemaphore.availablePermits() == PriorityScheduler.this._numRunners) {
                                    PriorityScheduler.this._resourceManager.stop();
                                }
                            }
                        }, MoreExecutors.directExecutor());
                        request.setResultFuture((ListenableFuture<byte[]>)queryFutureTask);
                        request.getSchedulerGroup().startQuery();
                        queryRequest.getTimerContext().getPhaseTimer(ServerQueryPhase.SCHEDULER_WAIT).stopAndRecord();
                        PriorityScheduler.this._resourceManager.getQueryRunners().submit(queryFutureTask);
                    }
                    catch (Throwable t) {
                        LOGGER.error("Error in scheduler thread. This is indicative of a bug. Please report this. Server will continue with errors", t);
                    }
                }
                if (PriorityScheduler.this._isRunning) {
                    throw new RuntimeException("FATAL: Scheduler thread is quitting.....something went horribly wrong.....!!!");
                }
                PriorityScheduler.this.failAllPendingQueries();
            }
        });
        this._scheduler.setName("scheduler");
        this._scheduler.setPriority(10);
        this._scheduler.setDaemon(true);
        this._scheduler.start();
    }

    @Override
    public void stop() {
        super.stop();
        if (this._scheduler != null) {
            this._scheduler.interrupt();
        }
    }

    private synchronized void failAllPendingQueries() {
        List<SchedulerQueryContext> pending = this._queryQueue.drain();
        for (SchedulerQueryContext queryContext : pending) {
            queryContext.setResultFuture(this.immediateErrorResponse(queryContext.getQueryRequest(), QueryException.SERVER_SCHEDULER_DOWN_ERROR));
        }
    }

    private void checkStopResourceManager() {
        if (!this._isRunning && this._runningQueriesSemaphore.availablePermits() == this._numRunners) {
            this._resourceManager.stop();
        }
    }
}

