/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.util;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.io.BaseEncoding;
import io.airlift.slice.Slice;
import io.trino.filesystem.Location;
import io.trino.hdfs.FileSystemUtils;
import io.trino.hdfs.HdfsContext;
import io.trino.hdfs.HdfsEnvironment;
import io.trino.hdfs.rubix.CachingTrinoS3FileSystem;
import io.trino.hdfs.s3.TrinoS3FileSystem;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.HiveReadOnlyException;
import io.trino.plugin.hive.HiveTimestampPrecision;
import io.trino.plugin.hive.HiveType;
import io.trino.plugin.hive.TableType;
import io.trino.plugin.hive.avro.AvroRecordWriter;
import io.trino.plugin.hive.metastore.Database;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.ProtectMode;
import io.trino.plugin.hive.metastore.SemiTransactionalHiveMetastore;
import io.trino.plugin.hive.metastore.Storage;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.parquet.ParquetRecordWriter;
import io.trino.plugin.hive.type.ListTypeInfo;
import io.trino.plugin.hive.type.MapTypeInfo;
import io.trino.plugin.hive.type.PrimitiveCategory;
import io.trino.plugin.hive.type.PrimitiveTypeInfo;
import io.trino.plugin.hive.type.StructTypeInfo;
import io.trino.plugin.hive.type.TypeInfo;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.plugin.hive.util.SequenceFileRecordWriter;
import io.trino.plugin.hive.util.TextHeaderWriter;
import io.trino.plugin.hive.util.TextRecordWriter;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.Page;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.Chars;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestamp;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.viewfs.ViewFileSystem;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hive.common.type.Date;
import org.apache.hadoop.hive.common.type.HiveDecimal;
import org.apache.hadoop.hive.common.type.Timestamp;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.exec.FileSinkOperator;
import org.apache.hadoop.hive.ql.io.HiveOutputFormat;
import org.apache.hadoop.hive.serde2.SerDeException;
import org.apache.hadoop.hive.serde2.Serializer;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspectorFactory;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.PrimitiveObjectInspectorFactory;
import org.apache.hadoop.hive.serde2.typeinfo.DecimalTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.TypeInfoFactory;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.Reporter;
import org.apache.hadoop.util.Progressable;
import org.joda.time.DateTimeZone;

public final class HiveWriteUtils {
    private HiveWriteUtils() {
    }

    public static FileSinkOperator.RecordWriter createRecordWriter(Path target, JobConf conf, Properties properties, String outputFormatName, ConnectorSession session) {
        return HiveWriteUtils.createRecordWriter(target, conf, properties, outputFormatName, session, Optional.empty());
    }

    public static FileSinkOperator.RecordWriter createRecordWriter(Path target, JobConf conf, Properties properties, String outputFormatName, ConnectorSession session, Optional<TextHeaderWriter> textHeaderWriter) {
        try {
            boolean compress = HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.COMPRESSRESULT);
            if (outputFormatName.equals("org.apache.hadoop.hive.ql.io.parquet.MapredParquetOutputFormat")) {
                return ParquetRecordWriter.create(target, conf, properties, session);
            }
            if (outputFormatName.equals("org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat")) {
                return new TextRecordWriter(target, conf, properties, compress, textHeaderWriter);
            }
            if (outputFormatName.equals("org.apache.hadoop.hive.ql.io.HiveSequenceFileOutputFormat")) {
                return new SequenceFileRecordWriter(target, conf, Text.class, compress);
            }
            if (outputFormatName.equals("org.apache.hadoop.hive.ql.io.avro.AvroContainerOutputFormat")) {
                return new AvroRecordWriter(target, conf, compress, properties);
            }
            Object writer = Class.forName(outputFormatName).getConstructor(new Class[0]).newInstance(new Object[0]);
            return ((HiveOutputFormat)writer).getHiveRecordWriter(conf, target, Text.class, compress, properties, (Progressable)Reporter.NULL);
        }
        catch (IOException | ReflectiveOperationException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_DATA_ERROR, (Throwable)e);
        }
    }

    public static Serializer initializeSerializer(Configuration conf, Properties properties, String serializerName) {
        try {
            Serializer result = (Serializer)Class.forName(serializerName).getConstructor(new Class[0]).newInstance(new Object[0]);
            result.initialize(conf, properties);
            return result;
        }
        catch (ClassNotFoundException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_SERDE_NOT_FOUND, "Serializer does not exist: " + serializerName);
        }
        catch (ReflectiveOperationException | SerDeException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_WRITER_DATA_ERROR, e);
        }
    }

    public static ObjectInspector getJavaObjectInspector(Type type) {
        if (type.equals(BooleanType.BOOLEAN)) {
            return PrimitiveObjectInspectorFactory.javaBooleanObjectInspector;
        }
        if (type.equals(BigintType.BIGINT)) {
            return PrimitiveObjectInspectorFactory.javaLongObjectInspector;
        }
        if (type.equals(IntegerType.INTEGER)) {
            return PrimitiveObjectInspectorFactory.javaIntObjectInspector;
        }
        if (type.equals(SmallintType.SMALLINT)) {
            return PrimitiveObjectInspectorFactory.javaShortObjectInspector;
        }
        if (type.equals(TinyintType.TINYINT)) {
            return PrimitiveObjectInspectorFactory.javaByteObjectInspector;
        }
        if (type.equals(RealType.REAL)) {
            return PrimitiveObjectInspectorFactory.javaFloatObjectInspector;
        }
        if (type.equals(DoubleType.DOUBLE)) {
            return PrimitiveObjectInspectorFactory.javaDoubleObjectInspector;
        }
        if (type instanceof VarcharType) {
            return PrimitiveObjectInspectorFactory.writableStringObjectInspector;
        }
        if (type instanceof CharType) {
            return PrimitiveObjectInspectorFactory.writableHiveCharObjectInspector;
        }
        if (type.equals(VarbinaryType.VARBINARY)) {
            return PrimitiveObjectInspectorFactory.javaByteArrayObjectInspector;
        }
        if (type.equals(DateType.DATE)) {
            return PrimitiveObjectInspectorFactory.javaDateObjectInspector;
        }
        if (type instanceof TimestampType) {
            return PrimitiveObjectInspectorFactory.javaTimestampObjectInspector;
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return PrimitiveObjectInspectorFactory.getPrimitiveJavaObjectInspector((org.apache.hadoop.hive.serde2.typeinfo.PrimitiveTypeInfo)new DecimalTypeInfo(decimalType.getPrecision(), decimalType.getScale()));
        }
        if (type instanceof ArrayType) {
            ArrayType arrayType = (ArrayType)type;
            return ObjectInspectorFactory.getStandardListObjectInspector((ObjectInspector)HiveWriteUtils.getJavaObjectInspector(arrayType.getElementType()));
        }
        if (type instanceof MapType) {
            MapType mapType = (MapType)type;
            ObjectInspector keyObjectInspector = HiveWriteUtils.getJavaObjectInspector(mapType.getKeyType());
            ObjectInspector valueObjectInspector = HiveWriteUtils.getJavaObjectInspector(mapType.getValueType());
            return ObjectInspectorFactory.getStandardMapObjectInspector((ObjectInspector)keyObjectInspector, (ObjectInspector)valueObjectInspector);
        }
        if (type instanceof RowType) {
            return ObjectInspectorFactory.getStandardStructObjectInspector((List)((List)type.getTypeSignature().getParameters().stream().map(parameter -> (String)parameter.getNamedTypeSignature().getName().get()).collect(ImmutableList.toImmutableList())), (List)((List)type.getTypeParameters().stream().map(HiveWriteUtils::getJavaObjectInspector).collect(ImmutableList.toImmutableList())));
        }
        throw new IllegalArgumentException("unsupported type: " + type);
    }

    public static List<String> createPartitionValues(List<Type> partitionColumnTypes, Page partitionColumns, int position) {
        ImmutableList.Builder partitionValues = ImmutableList.builder();
        for (int field = 0; field < partitionColumns.getChannelCount(); ++field) {
            Object value = HiveWriteUtils.getField(DateTimeZone.UTC, partitionColumnTypes.get(field), partitionColumns.getBlock(field), position);
            if (value == null) {
                partitionValues.add((Object)"__HIVE_DEFAULT_PARTITION__");
                continue;
            }
            String valueString = value.toString();
            if (!CharMatcher.inRange((char)' ', (char)'~').matchesAllOf((CharSequence)valueString)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_PARTITION_VALUE, "Hive partition keys can only contain printable ASCII characters (0x20 - 0x7E). Invalid value: " + BaseEncoding.base16().withSeparator(" ", 2).encode(valueString.getBytes(StandardCharsets.UTF_8)));
            }
            partitionValues.add((Object)valueString);
        }
        return partitionValues.build();
    }

    public static Object getField(DateTimeZone localZone, Type type, Block block, int position) {
        if (block.isNull(position)) {
            return null;
        }
        if (BooleanType.BOOLEAN.equals((Object)type)) {
            return BooleanType.BOOLEAN.getBoolean(block, position);
        }
        if (BigintType.BIGINT.equals((Object)type)) {
            return BigintType.BIGINT.getLong(block, position);
        }
        if (IntegerType.INTEGER.equals((Object)type)) {
            return IntegerType.INTEGER.getInt(block, position);
        }
        if (SmallintType.SMALLINT.equals((Object)type)) {
            return SmallintType.SMALLINT.getShort(block, position);
        }
        if (TinyintType.TINYINT.equals((Object)type)) {
            return TinyintType.TINYINT.getByte(block, position);
        }
        if (RealType.REAL.equals((Object)type)) {
            return Float.valueOf(RealType.REAL.getFloat(block, position));
        }
        if (DoubleType.DOUBLE.equals((Object)type)) {
            return DoubleType.DOUBLE.getDouble(block, position);
        }
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            return new Text(varcharType.getSlice(block, position).getBytes());
        }
        if (type instanceof CharType) {
            CharType charType = (CharType)type;
            return new Text(Chars.padSpaces((Slice)charType.getSlice(block, position), (CharType)charType).toStringUtf8());
        }
        if (VarbinaryType.VARBINARY.equals((Object)type)) {
            return VarbinaryType.VARBINARY.getSlice(block, position).getBytes();
        }
        if (DateType.DATE.equals((Object)type)) {
            return Date.ofEpochDay((int)DateType.DATE.getInt(block, position));
        }
        if (type instanceof TimestampType) {
            TimestampType timestampType = (TimestampType)type;
            return HiveWriteUtils.getHiveTimestamp(localZone, timestampType, block, position);
        }
        if (type instanceof TimestampWithTimeZoneType) {
            Preconditions.checkArgument((boolean)type.equals(TimestampWithTimeZoneType.TIMESTAMP_TZ_MILLIS));
            return HiveWriteUtils.getHiveTimestampTz(block, position);
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return HiveWriteUtils.getHiveDecimal(decimalType, block, position);
        }
        if (type instanceof ArrayType) {
            ArrayType arrayType = (ArrayType)type;
            Type elementType = arrayType.getElementType();
            Block arrayBlock = (Block)block.getObject(position, Block.class);
            ArrayList<Object> list = new ArrayList<Object>(arrayBlock.getPositionCount());
            for (int i = 0; i < arrayBlock.getPositionCount(); ++i) {
                list.add(HiveWriteUtils.getField(localZone, elementType, arrayBlock, i));
            }
            return Collections.unmodifiableList(list);
        }
        if (type instanceof MapType) {
            MapType mapType = (MapType)type;
            Type keyType = mapType.getKeyType();
            Type valueType = mapType.getValueType();
            Block mapBlock = (Block)block.getObject(position, Block.class);
            HashMap<Object, Object> map = new HashMap<Object, Object>();
            for (int i = 0; i < mapBlock.getPositionCount(); i += 2) {
                map.put(HiveWriteUtils.getField(localZone, keyType, mapBlock, i), HiveWriteUtils.getField(localZone, valueType, mapBlock, i + 1));
            }
            return Collections.unmodifiableMap(map);
        }
        if (type instanceof RowType) {
            RowType rowType = (RowType)type;
            List fieldTypes = rowType.getTypeParameters();
            Block rowBlock = (Block)block.getObject(position, Block.class);
            HiveUtil.checkCondition(fieldTypes.size() == rowBlock.getPositionCount(), (ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Expected row value field count does not match type field count", new Object[0]);
            ArrayList<Object> row = new ArrayList<Object>(rowBlock.getPositionCount());
            for (int i = 0; i < rowBlock.getPositionCount(); ++i) {
                row.add(HiveWriteUtils.getField(localZone, (Type)fieldTypes.get(i), rowBlock, i));
            }
            return Collections.unmodifiableList(row);
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "unsupported type: " + type);
    }

    public static void checkTableIsWritable(Table table, boolean writesToNonManagedTablesEnabled) {
        if (table.getTableType().equals(TableType.MATERIALIZED_VIEW.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cannot write to Hive materialized view");
        }
        if (!writesToNonManagedTablesEnabled && !table.getTableType().equals(TableType.MANAGED_TABLE.name())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Cannot write to non-managed Hive table");
        }
        HiveWriteUtils.checkWritable(table.getSchemaTableName(), Optional.empty(), MetastoreUtil.getProtectMode(table), table.getParameters(), table.getStorage());
    }

    public static void checkPartitionIsWritable(String partitionName, Partition partition) {
        HiveWriteUtils.checkWritable(partition.getSchemaTableName(), Optional.of(partitionName), MetastoreUtil.getProtectMode(partition), partition.getParameters(), partition.getStorage());
    }

    private static void checkWritable(SchemaTableName tableName, Optional<String> partitionName, ProtectMode protectMode, Map<String, String> parameters, Storage storage) {
        String tablePartitionDescription = "Table '" + tableName + "'";
        if (partitionName.isPresent()) {
            tablePartitionDescription = tablePartitionDescription + " partition '" + partitionName.get() + "'";
        }
        MetastoreUtil.verifyOnline(tableName, partitionName, protectMode, parameters);
        if (protectMode.readOnly()) {
            throw new HiveReadOnlyException(tableName, partitionName);
        }
        if (storage.isSkewed()) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Inserting into bucketed tables with skew is not supported. %s", tablePartitionDescription));
        }
    }

    public static Location getTableDefaultLocation(HdfsContext context, SemiTransactionalHiveMetastore metastore, HdfsEnvironment hdfsEnvironment, String schemaName, String tableName) {
        Database database = metastore.getDatabase(schemaName).orElseThrow(() -> new SchemaNotFoundException(schemaName));
        return HiveWriteUtils.getTableDefaultLocation(database, context, hdfsEnvironment, schemaName, tableName);
    }

    public static Location getTableDefaultLocation(Database database, HdfsContext context, HdfsEnvironment hdfsEnvironment, String schemaName, String tableName) {
        String location = database.getLocation().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location is not set", schemaName)));
        Path databasePath = new Path(location);
        if (!HiveWriteUtils.isS3FileSystem(context, hdfsEnvironment, databasePath)) {
            if (!HiveWriteUtils.pathExists(context, hdfsEnvironment, databasePath)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location does not exist: %s", schemaName, databasePath));
            }
            if (!HiveWriteUtils.isDirectory(context, hdfsEnvironment, databasePath)) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_DATABASE_LOCATION_ERROR, String.format("Database '%s' location is not a directory: %s", schemaName, databasePath));
            }
        }
        Location databaseLocation = Location.of((String)databasePath.toString());
        return databaseLocation.appendPath(HiveUtil.escapeTableName(tableName));
    }

    public static boolean pathExists(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            return hdfsEnvironment.getFileSystem(context, path).exists(path);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    public static boolean isS3FileSystem(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            FileSystem fileSystem = FileSystemUtils.getRawFileSystem((FileSystem)hdfsEnvironment.getFileSystem(context, path));
            return fileSystem instanceof TrinoS3FileSystem || fileSystem.getClass().getName().equals("com.amazon.ws.emr.hadoop.fs.EmrFileSystem") || fileSystem instanceof CachingTrinoS3FileSystem;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    public static boolean isViewFileSystem(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            return FileSystemUtils.getRawFileSystem((FileSystem)hdfsEnvironment.getFileSystem(context, path)) instanceof ViewFileSystem;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    private static boolean isDirectory(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            return hdfsEnvironment.getFileSystem(context, path).isDirectory(path);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking path: " + path, (Throwable)e);
        }
    }

    public static boolean isHdfsEncrypted(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            FileSystem fileSystem = FileSystemUtils.getRawFileSystem((FileSystem)hdfsEnvironment.getFileSystem(context, path));
            if (fileSystem instanceof DistributedFileSystem) {
                return ((DistributedFileSystem)fileSystem).getEZForPath(path) != null;
            }
            return false;
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed checking encryption status for path: " + path, (Throwable)e);
        }
    }

    public static boolean isFileCreatedByQuery(String fileName, String queryId) {
        return fileName.startsWith(queryId) || fileName.endsWith(queryId);
    }

    public static Location createTemporaryPath(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path targetPath, String temporaryStagingDirectoryPath) {
        String temporaryPrefix = temporaryStagingDirectoryPath.replace("${USER}", context.getIdentity().getUser());
        if (HiveWriteUtils.isViewFileSystem(context, hdfsEnvironment, targetPath)) {
            temporaryPrefix = ".hive-staging";
        }
        Path temporaryRoot = new Path(targetPath, temporaryPrefix);
        Path temporaryPath = new Path(temporaryRoot, UUID.randomUUID().toString());
        HiveWriteUtils.createDirectory(context, hdfsEnvironment, temporaryPath);
        if (hdfsEnvironment.isNewFileInheritOwnership()) {
            HiveWriteUtils.setDirectoryOwner(context, hdfsEnvironment, temporaryPath, targetPath);
        }
        return Location.of((String)temporaryPath.toString());
    }

    private static void setDirectoryOwner(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path, Path targetPath) {
        try {
            FileStatus fileStatus;
            FileSystem fileSystem = hdfsEnvironment.getFileSystem(context, path);
            if (!fileSystem.exists(targetPath)) {
                Path parent = targetPath.getParent();
                if (!fileSystem.exists(parent)) {
                    return;
                }
                fileStatus = fileSystem.getFileStatus(parent);
            } else {
                fileStatus = fileSystem.getFileStatus(targetPath);
            }
            String owner = fileStatus.getOwner();
            String group = fileStatus.getGroup();
            fileSystem.setOwner(path, owner, group);
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, String.format("Failed to set owner on %s based on %s", path, targetPath), (Throwable)e);
        }
    }

    public static void createDirectory(HdfsContext context, HdfsEnvironment hdfsEnvironment, Path path) {
        try {
            if (!hdfsEnvironment.getFileSystem(context, path).mkdirs(path, (FsPermission)hdfsEnvironment.getNewDirectoryPermissions().orElse(null))) {
                throw new IOException("mkdirs returned false");
            }
        }
        catch (IOException e) {
            throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed to create directory: " + path, (Throwable)e);
        }
        if (hdfsEnvironment.getNewDirectoryPermissions().isPresent()) {
            try {
                hdfsEnvironment.getFileSystem(context, path).setPermission(path, (FsPermission)hdfsEnvironment.getNewDirectoryPermissions().get());
            }
            catch (IOException e) {
                throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_FILESYSTEM_ERROR, "Failed to set permission on directory: " + path, (Throwable)e);
            }
        }
    }

    public static void checkedDelete(FileSystem fileSystem, Path file, boolean recursive) throws IOException {
        try {
            if (!fileSystem.delete(file, recursive) && fileSystem.exists(file)) {
                throw new IOException("Failed to delete " + file);
            }
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    public static boolean isWritableType(HiveType hiveType) {
        return HiveWriteUtils.isWritableType(hiveType.getTypeInfo());
    }

    private static boolean isWritableType(TypeInfo typeInfo) {
        switch (typeInfo.getCategory()) {
            case PRIMITIVE: {
                PrimitiveCategory primitiveCategory = ((PrimitiveTypeInfo)typeInfo).getPrimitiveCategory();
                return HiveWriteUtils.isWritablePrimitiveType(primitiveCategory);
            }
            case MAP: {
                MapTypeInfo mapTypeInfo = (MapTypeInfo)typeInfo;
                return HiveWriteUtils.isWritableType(mapTypeInfo.getMapKeyTypeInfo()) && HiveWriteUtils.isWritableType(mapTypeInfo.getMapValueTypeInfo());
            }
            case LIST: {
                ListTypeInfo listTypeInfo = (ListTypeInfo)typeInfo;
                return HiveWriteUtils.isWritableType(listTypeInfo.getListElementTypeInfo());
            }
            case STRUCT: {
                StructTypeInfo structTypeInfo = (StructTypeInfo)typeInfo;
                return structTypeInfo.getAllStructFieldTypeInfos().stream().allMatch(HiveWriteUtils::isWritableType);
            }
        }
        return false;
    }

    private static boolean isWritablePrimitiveType(PrimitiveCategory primitiveCategory) {
        switch (primitiveCategory) {
            case BOOLEAN: 
            case LONG: 
            case INT: 
            case SHORT: 
            case BYTE: 
            case FLOAT: 
            case DOUBLE: 
            case STRING: 
            case DATE: 
            case TIMESTAMP: 
            case BINARY: 
            case DECIMAL: 
            case VARCHAR: 
            case CHAR: {
                return true;
            }
        }
        return false;
    }

    public static List<ObjectInspector> getRowColumnInspectors(List<Type> types) {
        return types.stream().map(HiveWriteUtils::getRowColumnInspector).collect(Collectors.toList());
    }

    public static ObjectInspector getRowColumnInspector(Type type) {
        if (type.equals(BooleanType.BOOLEAN)) {
            return PrimitiveObjectInspectorFactory.writableBooleanObjectInspector;
        }
        if (type.equals(BigintType.BIGINT)) {
            return PrimitiveObjectInspectorFactory.writableLongObjectInspector;
        }
        if (type.equals(IntegerType.INTEGER)) {
            return PrimitiveObjectInspectorFactory.writableIntObjectInspector;
        }
        if (type.equals(SmallintType.SMALLINT)) {
            return PrimitiveObjectInspectorFactory.writableShortObjectInspector;
        }
        if (type.equals(TinyintType.TINYINT)) {
            return PrimitiveObjectInspectorFactory.writableByteObjectInspector;
        }
        if (type.equals(RealType.REAL)) {
            return PrimitiveObjectInspectorFactory.writableFloatObjectInspector;
        }
        if (type.equals(DoubleType.DOUBLE)) {
            return PrimitiveObjectInspectorFactory.writableDoubleObjectInspector;
        }
        if (type instanceof VarcharType) {
            VarcharType varcharType = (VarcharType)type;
            if (varcharType.isUnbounded()) {
                return PrimitiveObjectInspectorFactory.writableStringObjectInspector;
            }
            if (varcharType.getBoundedLength() <= 65535) {
                return PrimitiveObjectInspectorFactory.getPrimitiveWritableObjectInspector((org.apache.hadoop.hive.serde2.typeinfo.PrimitiveTypeInfo)TypeInfoFactory.getVarcharTypeInfo((int)varcharType.getBoundedLength()));
            }
        }
        if (type instanceof CharType) {
            CharType charType = (CharType)type;
            int charLength = charType.getLength();
            return PrimitiveObjectInspectorFactory.getPrimitiveWritableObjectInspector((org.apache.hadoop.hive.serde2.typeinfo.PrimitiveTypeInfo)TypeInfoFactory.getCharTypeInfo((int)charLength));
        }
        if (type.equals(VarbinaryType.VARBINARY)) {
            return PrimitiveObjectInspectorFactory.writableBinaryObjectInspector;
        }
        if (type.equals(DateType.DATE)) {
            return PrimitiveObjectInspectorFactory.writableDateObjectInspector;
        }
        if (type instanceof TimestampType) {
            return PrimitiveObjectInspectorFactory.writableTimestampObjectInspector;
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            return PrimitiveObjectInspectorFactory.getPrimitiveWritableObjectInspector((org.apache.hadoop.hive.serde2.typeinfo.PrimitiveTypeInfo)new DecimalTypeInfo(decimalType.getPrecision(), decimalType.getScale()));
        }
        if (HiveUtil.isStructuralType(type)) {
            return HiveWriteUtils.getJavaObjectInspector(type);
        }
        throw new IllegalArgumentException("unsupported type: " + type);
    }

    public static HiveDecimal getHiveDecimal(DecimalType decimalType, Block block, int position) {
        BigInteger unscaledValue = decimalType.isShort() ? BigInteger.valueOf(decimalType.getLong(block, position)) : ((Int128)decimalType.getObject(block, position)).toBigInteger();
        return HiveDecimal.create((BigInteger)unscaledValue, (int)decimalType.getScale());
    }

    private static Timestamp getHiveTimestamp(DateTimeZone localZone, TimestampType type, Block block, int position) {
        long epochSeconds;
        int nanosOfMicro;
        long epochMicros;
        Verify.verify((type.getPrecision() <= HiveTimestampPrecision.MAX.getPrecision() ? 1 : 0) != 0, (String)"Timestamp precision too high for Hive", (Object[])new Object[0]);
        if (type.isShort()) {
            epochMicros = type.getLong(block, position);
            nanosOfMicro = 0;
        } else {
            LongTimestamp timestamp = (LongTimestamp)type.getObject(block, position);
            epochMicros = timestamp.getEpochMicros();
            nanosOfMicro = timestamp.getPicosOfMicro() / 1000;
        }
        if (DateTimeZone.UTC.equals((Object)localZone)) {
            epochSeconds = Math.floorDiv(epochMicros, 1000000);
        } else {
            long localEpochMillis = Math.floorDiv(epochMicros, 1000);
            long utcEpochMillis = localZone.convertLocalToUTC(localEpochMillis, false);
            epochSeconds = Math.floorDiv(utcEpochMillis, 1000);
        }
        int microsOfSecond = Math.floorMod(epochMicros, 1000000);
        int nanosOfSecond = microsOfSecond * 1000 + nanosOfMicro;
        return Timestamp.ofEpochSecond((long)epochSeconds, (int)nanosOfSecond);
    }

    private static Timestamp getHiveTimestampTz(Block block, int position) {
        long epochMillis = DateTimeEncoding.unpackMillisUtc((long)block.getLong(position, 0));
        return Timestamp.ofEpochMilli((long)epochMillis);
    }
}

