/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.bigquery;

import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.storage.v1.ArrowSerializationOptions;
import com.google.cloud.bigquery.storage.v1.BigQueryReadClient;
import com.google.cloud.bigquery.storage.v1.CreateReadSessionRequest;
import com.google.cloud.bigquery.storage.v1.DataFormat;
import com.google.cloud.bigquery.storage.v1.ReadSession;
import com.google.protobuf.ByteString;
import dev.failsafe.Failsafe;
import dev.failsafe.Policy;
import dev.failsafe.RetryPolicy;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.plugin.bigquery.BigQueryClient;
import io.trino.plugin.bigquery.BigQueryClientFactory;
import io.trino.plugin.bigquery.BigQueryReadClientFactory;
import io.trino.plugin.bigquery.BigQueryUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.channels.ReadableByteChannel;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.arrow.vector.ipc.ReadChannel;
import org.apache.arrow.vector.ipc.message.MessageSerializer;
import org.apache.arrow.vector.util.ByteArrayReadableSeekableByteChannel;

public class ReadSessionCreator {
    private static final Logger log = Logger.get(ReadSessionCreator.class);
    private final BigQueryClientFactory bigQueryClientFactory;
    private final BigQueryReadClientFactory bigQueryReadClientFactory;
    private final boolean viewEnabled;
    private final boolean arrowSerializationEnabled;
    private final Duration viewExpiration;
    private final int maxCreateReadSessionRetries;

    public ReadSessionCreator(BigQueryClientFactory bigQueryClientFactory, BigQueryReadClientFactory bigQueryReadClientFactory, boolean viewEnabled, boolean arrowSerializationEnabled, Duration viewExpiration, int maxCreateReadSessionRetries) {
        this.bigQueryClientFactory = bigQueryClientFactory;
        this.bigQueryReadClientFactory = bigQueryReadClientFactory;
        this.viewEnabled = viewEnabled;
        this.arrowSerializationEnabled = arrowSerializationEnabled;
        this.viewExpiration = viewExpiration;
        this.maxCreateReadSessionRetries = maxCreateReadSessionRetries;
    }

    public ReadSession create(ConnectorSession session, TableId remoteTable, List<String> selectedFields, Optional<String> filter, int parallelism) {
        BigQueryClient client = this.bigQueryClientFactory.create(session);
        TableInfo tableDetails = client.getTable(remoteTable).orElseThrow(() -> new TableNotFoundException(new SchemaTableName(remoteTable.getDataset(), remoteTable.getTable())));
        TableInfo actualTable = this.getActualTable(client, tableDetails, selectedFields);
        List filteredSelectedFields = selectedFields.stream().map(BigQueryUtil::toBigQueryColumnName).collect(Collectors.toList());
        try (BigQueryReadClient bigQueryReadClient = this.bigQueryReadClientFactory.create(session);){
            ReadSession.TableReadOptions.Builder readOptions = ReadSession.TableReadOptions.newBuilder().addAllSelectedFields(filteredSelectedFields);
            filter.ifPresent(arg_0 -> ((ReadSession.TableReadOptions.Builder)readOptions).setRowRestriction(arg_0));
            DataFormat format = DataFormat.AVRO;
            if (this.arrowSerializationEnabled) {
                format = DataFormat.ARROW;
                readOptions.setArrowSerializationOptions(ArrowSerializationOptions.newBuilder().setBufferCompression(ArrowSerializationOptions.CompressionCodec.ZSTD).build());
            }
            CreateReadSessionRequest createReadSessionRequest = CreateReadSessionRequest.newBuilder().setParent("projects/" + client.getParentProjectId()).setReadSession(ReadSession.newBuilder().setDataFormat(format).setTable(this.toTableResourceName(actualTable.getTableId())).setReadOptions(readOptions)).setMaxStreamCount(parallelism).build();
            ReadSession readSession = (ReadSession)Failsafe.with((Policy)RetryPolicy.builder().withMaxRetries(this.maxCreateReadSessionRetries).withBackoff(10L, 500L, ChronoUnit.MILLIS).onRetry(event -> log.debug("Request failed, retrying: %s", new Object[]{event.getLastException()})).abortOn(failure -> !BigQueryUtil.isRetryable(failure)).build(), (Policy[])new RetryPolicy[0]).get(() -> bigQueryReadClient.createReadSession(createReadSessionRequest));
            return readSession;
        }
    }

    public String getSchemaAsString(ReadSession readSession) {
        if (this.arrowSerializationEnabled) {
            return ReadSessionCreator.deserializeArrowSchema(readSession.getArrowSchema().getSerializedSchema());
        }
        return readSession.getAvroSchema().getSchema();
    }

    private static String deserializeArrowSchema(ByteString serializedSchema) {
        try {
            return MessageSerializer.deserializeSchema((ReadChannel)new ReadChannel((ReadableByteChannel)new ByteArrayReadableSeekableByteChannel(serializedSchema.toByteArray()))).toJson();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    String toTableResourceName(TableId tableId) {
        return String.format("projects/%s/datasets/%s/tables/%s", tableId.getProject(), tableId.getDataset(), tableId.getTable());
    }

    private TableInfo getActualTable(BigQueryClient client, TableInfo remoteTable, List<String> requiredColumns) {
        TableDefinition tableDefinition = remoteTable.getDefinition();
        TableDefinition.Type tableType = tableDefinition.getType();
        if (tableType == TableDefinition.Type.TABLE || tableType == TableDefinition.Type.SNAPSHOT) {
            return remoteTable;
        }
        if (tableType == TableDefinition.Type.VIEW) {
            if (!this.viewEnabled) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Views are not enabled. You can enable views by setting '%s' to true. Notice additional cost may occur.", "bigquery.views-enabled"));
            }
            return client.getCachedTable(this.viewExpiration, remoteTable, requiredColumns);
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, String.format("Table type '%s' of table '%s.%s' is not supported", tableType, remoteTable.getTableId().getDataset(), remoteTable.getTableId().getTable()));
    }
}

