/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.geospatial;

import com.esri.core.geometry.Envelope;
import com.esri.core.geometry.Point;
import com.esri.core.geometry.ogc.OGCGeometry;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.geospatial.GeometryUtils;
import io.trino.geospatial.serde.GeometrySerde;
import io.trino.plugin.geospatial.BingTile;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.BufferedRowValueBuilder;
import io.trino.spi.function.Description;
import io.trino.spi.function.ScalarFunction;
import io.trino.spi.function.SqlType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RowType;
import java.util.List;

public final class BingTileFunctions {
    private static final int TILE_PIXELS = 256;
    private static final double MAX_LATITUDE = 85.05112878;
    private static final double MIN_LATITUDE = -85.05112878;
    private static final double MIN_LONGITUDE = -180.0;
    private static final double MAX_LONGITUDE = 180.0;
    private static final double EARTH_RADIUS_KM = 6371.01;
    private static final int OPTIMIZED_TILING_MIN_ZOOM_LEVEL = 10;
    private static final Block EMPTY_TILE_ARRAY = BigintType.BIGINT.createFixedSizeBlockBuilder(0).build();
    private static final String LATITUDE_OUT_OF_RANGE = "Latitude must be between -85.05112878 and 85.05112878";
    private static final String LATITUDE_SPAN_OUT_OF_RANGE = String.format("Latitude span for the geometry must be in [%.2f, %.2f] range", -85.05112878, 85.05112878);
    private static final String LONGITUDE_OUT_OF_RANGE = "Longitude must be between -180.0 and 180.0";
    private static final String LONGITUDE_SPAN_OUT_OF_RANGE = String.format("Longitude span for the geometry must be in [%.2f, %.2f] range", -180.0, 180.0);
    private static final String QUAD_KEY_EMPTY = "QuadKey must not be empty string";
    private static final String QUAD_KEY_TOO_LONG = "QuadKey must be 23 characters or less";
    private static final String ZOOM_LEVEL_TOO_SMALL = "Zoom level must be > 0";
    private static final String ZOOM_LEVEL_TOO_LARGE = "Zoom level must be <= 23";

    private BingTileFunctions() {
    }

    @Description(value="Creates a Bing tile from XY coordinates and zoom level")
    @ScalarFunction(value="bing_tile")
    @SqlType(value="BingTile")
    public static long toBingTile(@SqlType(value="integer") long tileX, @SqlType(value="integer") long tileY, @SqlType(value="integer") long zoomLevel) {
        BingTileFunctions.checkZoomLevel(zoomLevel);
        BingTileFunctions.checkCoordinate(tileX, zoomLevel);
        BingTileFunctions.checkCoordinate(tileY, zoomLevel);
        return BingTile.fromCoordinates(Math.toIntExact(tileX), Math.toIntExact(tileY), Math.toIntExact(zoomLevel)).encode();
    }

    @Description(value="Given a Bing tile, returns its QuadKey")
    @ScalarFunction(value="bing_tile_quadkey")
    @SqlType(value="varchar")
    public static Slice toQuadKey(@SqlType(value="BingTile") long input) {
        return Slices.utf8Slice((String)BingTile.decode(input).toQuadKey());
    }

    @Description(value="Given a Bing tile, returns zoom level of the tile")
    @ScalarFunction(value="bing_tile_zoom_level")
    @SqlType(value="tinyint")
    public static long bingTileZoomLevel(@SqlType(value="BingTile") long input) {
        return BingTile.decode(input).getZoomLevel();
    }

    @Description(value="Creates a Bing tile from a QuadKey")
    @ScalarFunction(value="bing_tile")
    @SqlType(value="BingTile")
    public static long toBingTile(@SqlType(value="varchar") Slice quadKey) {
        BingTileFunctions.checkQuadKey(quadKey);
        return BingTile.fromQuadKey(quadKey.toStringUtf8()).encode();
    }

    @Description(value="Given a (latitude, longitude) point, returns the containing Bing tile at the specified zoom level")
    @ScalarFunction(value="bing_tile_at")
    @SqlType(value="BingTile")
    public static long bingTileAt(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="integer") long zoomLevel) {
        BingTileFunctions.checkLatitude(latitude, LATITUDE_OUT_OF_RANGE);
        BingTileFunctions.checkLongitude(longitude, LONGITUDE_OUT_OF_RANGE);
        BingTileFunctions.checkZoomLevel(zoomLevel);
        return BingTileFunctions.latitudeLongitudeToTile(latitude, longitude, Math.toIntExact(zoomLevel)).encode();
    }

    @Description(value="Given a (longitude, latitude) point, returns the surrounding Bing tiles at the specified zoom level")
    @ScalarFunction(value="bing_tiles_around")
    @SqlType(value="array(BingTile)")
    public static Block bingTilesAround(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="integer") long zoomLevel) {
        BingTileFunctions.checkLatitude(latitude, LATITUDE_OUT_OF_RANGE);
        BingTileFunctions.checkLongitude(longitude, LONGITUDE_OUT_OF_RANGE);
        BingTileFunctions.checkZoomLevel(zoomLevel);
        long mapSize = BingTileFunctions.mapSize(Math.toIntExact(zoomLevel));
        long maxTileIndex = mapSize / 256L - 1L;
        int tileX = BingTileFunctions.longitudeToTileX(longitude, mapSize);
        int tileY = BingTileFunctions.longitudeToTileY(latitude, mapSize);
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, 9);
        for (int i = -1; i <= 1; ++i) {
            for (int j = -1; j <= 1; ++j) {
                int x = tileX + i;
                int y = tileY + j;
                if (x < 0 || (long)x > maxTileIndex || y < 0 || (long)y > maxTileIndex) continue;
                BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(x, y, Math.toIntExact(zoomLevel)).encode());
            }
        }
        return blockBuilder.build();
    }

    @Description(value="Given a (latitude, longitude) point, a radius in kilometers and a zoom level, returns a minimum set of Bing tiles at specified zoom level that cover a circle of specified radius around the specified point.")
    @ScalarFunction(value="bing_tiles_around")
    @SqlType(value="array(BingTile)")
    public static Block bingTilesAround(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="integer") long zoomLevelAsLong, @SqlType(value="double") double radiusInKm) {
        BingTile tile;
        int y;
        boolean include;
        int x;
        BingTileFunctions.checkLatitude(latitude, LATITUDE_OUT_OF_RANGE);
        BingTileFunctions.checkLongitude(longitude, LONGITUDE_OUT_OF_RANGE);
        BingTileFunctions.checkZoomLevel(zoomLevelAsLong);
        BingTileFunctions.checkCondition(radiusInKm >= 0.0, "Radius must be >= 0", new Object[0]);
        BingTileFunctions.checkCondition(radiusInKm <= 1000.0, "Radius must be <= 1,000 km", new Object[0]);
        int zoomLevel = Math.toIntExact(zoomLevelAsLong);
        long mapSize = BingTileFunctions.mapSize(zoomLevel);
        int maxTileIndex = (int)(mapSize / 256L) - 1;
        int tileY = BingTileFunctions.longitudeToTileY(latitude, mapSize);
        int tileX = BingTileFunctions.longitudeToTileX(longitude, mapSize);
        double topLatitude = BingTileFunctions.addDistanceToLatitude(latitude, radiusInKm, 0.0);
        BingTile topTile = BingTileFunctions.latitudeLongitudeToTile(topLatitude, longitude, zoomLevel);
        double bottomLatitude = BingTileFunctions.addDistanceToLatitude(latitude, radiusInKm, 180.0);
        BingTile bottomTile = BingTileFunctions.latitudeLongitudeToTile(bottomLatitude, longitude, zoomLevel);
        double leftLongitude = BingTileFunctions.addDistanceToLongitude(latitude, longitude, radiusInKm, 270.0);
        BingTile leftTile = BingTileFunctions.latitudeLongitudeToTile(latitude, leftLongitude, zoomLevel);
        double rightLongitude = BingTileFunctions.addDistanceToLongitude(latitude, longitude, radiusInKm, 90.0);
        BingTile rightTile = BingTileFunctions.latitudeLongitudeToTile(latitude, rightLongitude, zoomLevel);
        boolean wrapAroundX = rightTile.getX() < leftTile.getX();
        int tileCountX = wrapAroundX ? rightTile.getX() + maxTileIndex - leftTile.getX() + 2 : rightTile.getX() - leftTile.getX() + 1;
        int tileCountY = bottomTile.getY() - topTile.getY() + 1;
        int totalTileCount = tileCountX * tileCountY;
        BingTileFunctions.checkCondition(totalTileCount <= 1000000, "The number of tiles covering input rectangle exceeds the limit of 1M. Number of tiles: %d. Radius: %.1f km. Zoom level: %d.", totalTileCount, radiusInKm, zoomLevel);
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, totalTileCount);
        for (int i = 0; i < tileCountX; ++i) {
            x = (leftTile.getX() + i) % (maxTileIndex + 1);
            BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(x, tileY, zoomLevel).encode());
        }
        for (int y2 = topTile.getY(); y2 <= bottomTile.getY(); ++y2) {
            if (y2 == tileY) continue;
            BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(tileX, y2, zoomLevel).encode());
        }
        GreatCircleDistanceToPoint distanceToCenter = new GreatCircleDistanceToPoint(latitude, longitude);
        x = rightTile.getX();
        while (x != tileX) {
            include = false;
            for (y = topTile.getY(); y < tileY; ++y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point bottomLeftCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX(), tile.getY() + 1, tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, bottomLeftCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            include = false;
            for (y = bottomTile.getY(); y > tileY; --y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point topLeftCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX(), tile.getY(), tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, topLeftCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            x = x == 0 ? maxTileIndex : x - 1;
        }
        x = leftTile.getX();
        while (x != tileX) {
            include = false;
            for (y = topTile.getY(); y < tileY; ++y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point bottomRightCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX() + 1, tile.getY() + 1, tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, bottomRightCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            include = false;
            for (y = bottomTile.getY(); y > tileY; --y) {
                tile = BingTile.fromCoordinates(x, y, zoomLevel);
                if (include) {
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                    continue;
                }
                Point topRightCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX() + 1, tile.getY(), tile.getZoomLevel());
                if (!BingTileFunctions.withinDistance(distanceToCenter, radiusInKm, topRightCorner)) continue;
                include = true;
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            x = (x + 1) % (maxTileIndex + 1);
        }
        return blockBuilder.build();
    }

    @Description(value="Given a Bing tile, returns the polygon representation of the tile")
    @ScalarFunction(value="bing_tile_polygon")
    @SqlType(value="Geometry")
    public static Slice bingTilePolygon(@SqlType(value="BingTile") long input) {
        BingTile tile = BingTile.decode(input);
        return GeometrySerde.serialize((Envelope)BingTileFunctions.tileToEnvelope(tile));
    }

    @Description(value="Given a geometry and a zoom level, returns the minimum set of Bing tiles that fully covers that geometry")
    @ScalarFunction(value="geometry_to_bing_tiles")
    @SqlType(value="array(BingTile)")
    public static Block geometryToBingTiles(@SqlType(value="Geometry") Slice input, @SqlType(value="integer") long zoomLevelInput) {
        BingTileFunctions.checkZoomLevel(zoomLevelInput);
        int zoomLevel = Math.toIntExact(zoomLevelInput);
        OGCGeometry ogcGeometry = GeometrySerde.deserialize((Slice)input);
        if (ogcGeometry.isEmpty()) {
            return EMPTY_TILE_ARRAY;
        }
        Envelope envelope = GeometryUtils.getEnvelope((OGCGeometry)ogcGeometry);
        BingTileFunctions.checkLatitude(envelope.getYMin(), LATITUDE_SPAN_OUT_OF_RANGE);
        BingTileFunctions.checkLatitude(envelope.getYMax(), LATITUDE_SPAN_OUT_OF_RANGE);
        BingTileFunctions.checkLongitude(envelope.getXMin(), LONGITUDE_SPAN_OUT_OF_RANGE);
        BingTileFunctions.checkLongitude(envelope.getXMax(), LONGITUDE_SPAN_OUT_OF_RANGE);
        boolean pointOrRectangle = GeometryUtils.isPointOrRectangle((OGCGeometry)ogcGeometry, (Envelope)envelope);
        BingTile leftUpperTile = BingTileFunctions.latitudeLongitudeToTile(envelope.getYMax(), envelope.getXMin(), zoomLevel);
        BingTile rightLowerTile = BingTileFunctions.getTileCoveringLowerRightCorner(envelope, zoomLevel);
        long tileCount = (long)(rightLowerTile.getX() - leftUpperTile.getX() + 1) * (long)(rightLowerTile.getY() - leftUpperTile.getY() + 1);
        BingTileFunctions.checkGeometryToBingTilesLimits(ogcGeometry, envelope, pointOrRectangle, tileCount, zoomLevel);
        BlockBuilder blockBuilder = BigintType.BIGINT.createBlockBuilder(null, Math.toIntExact(tileCount));
        if (pointOrRectangle || zoomLevel <= 10) {
            for (int x = leftUpperTile.getX(); x <= rightLowerTile.getX(); ++x) {
                for (int y = leftUpperTile.getY(); y <= rightLowerTile.getY(); ++y) {
                    BingTile tile = BingTile.fromCoordinates(x, y, zoomLevel);
                    if (!pointOrRectangle && GeometryUtils.disjoint((Envelope)BingTileFunctions.tileToEnvelope(tile), (OGCGeometry)ogcGeometry)) continue;
                    BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
                }
            }
        } else {
            BingTile[] tiles;
            for (BingTile tile : tiles = BingTileFunctions.getTilesInBetween(leftUpperTile, rightLowerTile, 10)) {
                BingTileFunctions.appendIntersectingSubtiles(ogcGeometry, zoomLevel, tile, blockBuilder);
            }
        }
        return blockBuilder.build();
    }

    private static BingTile getTileCoveringLowerRightCorner(Envelope envelope, int zoomLevel) {
        BingTile tile = BingTileFunctions.latitudeLongitudeToTile(envelope.getYMin(), envelope.getXMax(), zoomLevel);
        int deltaX = 0;
        int deltaY = 0;
        Point upperLeftCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX(), tile.getY(), tile.getZoomLevel());
        if (upperLeftCorner.getX() == envelope.getXMax()) {
            deltaX = -1;
        }
        if (upperLeftCorner.getY() == envelope.getYMin()) {
            deltaY = -1;
        }
        if (deltaX != 0 || deltaY != 0) {
            return BingTile.fromCoordinates(tile.getX() + deltaX, tile.getY() + deltaY, tile.getZoomLevel());
        }
        return tile;
    }

    private static void checkGeometryToBingTilesLimits(OGCGeometry ogcGeometry, Envelope envelope, boolean pointOrRectangle, long tileCount, int zoomLevel) {
        if (pointOrRectangle) {
            BingTileFunctions.checkCondition(tileCount <= 1000000L, "The number of tiles covering input rectangle exceeds the limit of 1M. Number of tiles: %d. Rectangle: xMin=%.2f, yMin=%.2f, xMax=%.2f, yMax=%.2f. Zoom level: %d.", tileCount, envelope.getXMin(), envelope.getYMin(), envelope.getXMax(), envelope.getYMax(), zoomLevel);
        } else {
            BingTileFunctions.checkCondition((long)((int)tileCount) == tileCount, "The zoom level is too high to compute a set of covering Bing tiles.", new Object[0]);
            long complexity = 0L;
            try {
                complexity = Math.multiplyExact(tileCount, GeometryUtils.getPointCount((OGCGeometry)ogcGeometry));
            }
            catch (ArithmeticException e) {
                BingTileFunctions.checkCondition(false, "The zoom level is too high or the geometry is too complex to compute a set of covering Bing tiles. Please use a lower zoom level or convert the geometry to its bounding box using the ST_Envelope function.", new Object[0]);
            }
            BingTileFunctions.checkCondition(complexity <= 25000000L, "The zoom level is too high or the geometry is too complex to compute a set of covering Bing tiles. Please use a lower zoom level or convert the geometry to its bounding box using the ST_Envelope function.", new Object[0]);
        }
    }

    private static double addDistanceToLongitude(@SqlType(value="double") double latitude, @SqlType(value="double") double longitude, @SqlType(value="double") double radiusInKm, @SqlType(value="double") double bearing) {
        double latitudeInRadians = Math.toRadians(latitude);
        double longitudeInRadians = Math.toRadians(longitude);
        double bearingInRadians = Math.toRadians(bearing);
        double radiusRatio = radiusInKm / 6371.01;
        double newLongitude = Math.toDegrees(longitudeInRadians + Math.atan2(Math.sin(bearingInRadians) * Math.sin(radiusRatio) * Math.cos(latitudeInRadians), Math.cos(radiusRatio) - Math.sin(latitudeInRadians) * Math.sin(latitudeInRadians)));
        if (newLongitude > 180.0) {
            return -180.0 + (newLongitude - 180.0);
        }
        if (newLongitude < -180.0) {
            return 180.0 + (newLongitude - -180.0);
        }
        return newLongitude;
    }

    private static double addDistanceToLatitude(@SqlType(value="double") double latitude, @SqlType(value="double") double radiusInKm, @SqlType(value="double") double bearing) {
        double latitudeInRadians = Math.toRadians(latitude);
        double bearingInRadians = Math.toRadians(bearing);
        double radiusRatio = radiusInKm / 6371.01;
        double newLatitude = Math.toDegrees(Math.asin(Math.sin(latitudeInRadians) * Math.cos(radiusRatio) + Math.cos(latitudeInRadians) * Math.sin(radiusRatio) * Math.cos(bearingInRadians)));
        if (newLatitude > 85.05112878) {
            return 85.05112878;
        }
        if (newLatitude < -85.05112878) {
            return -85.05112878;
        }
        return newLatitude;
    }

    private static BingTile[] getTilesInBetween(BingTile leftUpperTile, BingTile rightLowerTile, int zoomLevel) {
        Preconditions.checkArgument((leftUpperTile.getZoomLevel() == rightLowerTile.getZoomLevel() ? 1 : 0) != 0);
        Preconditions.checkArgument((leftUpperTile.getZoomLevel() > zoomLevel ? 1 : 0) != 0);
        int divisor = 1 << leftUpperTile.getZoomLevel() - zoomLevel;
        int minX = leftUpperTile.getX() / divisor;
        int maxX = rightLowerTile.getX() / divisor;
        int minY = leftUpperTile.getY() / divisor;
        int maxY = rightLowerTile.getY() / divisor;
        BingTile[] tiles = new BingTile[(maxX - minX + 1) * (maxY - minY + 1)];
        int index = 0;
        for (int x = minX; x <= maxX; ++x) {
            for (int y = minY; y <= maxY; ++y) {
                tiles[index] = BingTile.fromCoordinates(x, y, 10);
                ++index;
            }
        }
        return tiles;
    }

    private static void appendIntersectingSubtiles(OGCGeometry ogcGeometry, int zoomLevel, BingTile tile, BlockBuilder blockBuilder) {
        int tileZoomLevel = tile.getZoomLevel();
        Preconditions.checkArgument((tileZoomLevel <= zoomLevel ? 1 : 0) != 0);
        Envelope tileEnvelope = BingTileFunctions.tileToEnvelope(tile);
        if (tileZoomLevel == zoomLevel) {
            if (!GeometryUtils.disjoint((Envelope)tileEnvelope, (OGCGeometry)ogcGeometry)) {
                BigintType.BIGINT.writeLong(blockBuilder, tile.encode());
            }
            return;
        }
        if (GeometryUtils.contains((OGCGeometry)ogcGeometry, (Envelope)tileEnvelope)) {
            int subTileCount = 1 << zoomLevel - tileZoomLevel;
            int minX = subTileCount * tile.getX();
            int minY = subTileCount * tile.getY();
            for (int x = minX; x < minX + subTileCount; ++x) {
                for (int y = minY; y < minY + subTileCount; ++y) {
                    BigintType.BIGINT.writeLong(blockBuilder, BingTile.fromCoordinates(x, y, zoomLevel).encode());
                }
            }
            return;
        }
        if (GeometryUtils.disjoint((Envelope)tileEnvelope, (OGCGeometry)ogcGeometry)) {
            return;
        }
        int minX = 2 * tile.getX();
        int minY = 2 * tile.getY();
        int nextZoomLevel = tileZoomLevel + 1;
        Verify.verify((nextZoomLevel <= 23 ? 1 : 0) != 0);
        for (int x = minX; x < minX + 2; ++x) {
            for (int y = minY; y < minY + 2; ++y) {
                BingTileFunctions.appendIntersectingSubtiles(ogcGeometry, zoomLevel, BingTile.fromCoordinates(x, y, nextZoomLevel), blockBuilder);
            }
        }
    }

    private static Point tileXYToLatitudeLongitude(int tileX, int tileY, int zoomLevel) {
        long mapSize = BingTileFunctions.mapSize(zoomLevel);
        double x = BingTileFunctions.clip(tileX * 256, 0.0, mapSize) / (double)mapSize - 0.5;
        double y = 0.5 - BingTileFunctions.clip(tileY * 256, 0.0, mapSize) / (double)mapSize;
        double latitude = 90.0 - 360.0 * Math.atan(Math.exp(-y * 2.0 * Math.PI)) / Math.PI;
        double longitude = 360.0 * x;
        return new Point(longitude, latitude);
    }

    private static BingTile latitudeLongitudeToTile(double latitude, double longitude, int zoomLevel) {
        long mapSize = BingTileFunctions.mapSize(zoomLevel);
        int tileX = BingTileFunctions.longitudeToTileX(longitude, mapSize);
        int tileY = BingTileFunctions.longitudeToTileY(latitude, mapSize);
        return BingTile.fromCoordinates(tileX, tileY, zoomLevel);
    }

    private static int longitudeToTileX(double longitude, long mapSize) {
        double x = (longitude + 180.0) / 360.0;
        return BingTileFunctions.axisToCoordinates(x, mapSize);
    }

    private static int longitudeToTileY(double latitude, long mapSize) {
        double sinLatitude = Math.sin(latitude * Math.PI / 180.0);
        double y = 0.5 - Math.log((1.0 + sinLatitude) / (1.0 - sinLatitude)) / (Math.PI * 4);
        return BingTileFunctions.axisToCoordinates(y, mapSize);
    }

    private static int axisToCoordinates(double axis, long mapSize) {
        int tileAxis = (int)BingTileFunctions.clip(axis * (double)mapSize, 0.0, mapSize - 1L);
        return tileAxis / 256;
    }

    private static Envelope tileToEnvelope(BingTile tile) {
        Point upperLeftCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX(), tile.getY(), tile.getZoomLevel());
        Point lowerRightCorner = BingTileFunctions.tileXYToLatitudeLongitude(tile.getX() + 1, tile.getY() + 1, tile.getZoomLevel());
        return new Envelope(upperLeftCorner.getX(), lowerRightCorner.getY(), lowerRightCorner.getX(), upperLeftCorner.getY());
    }

    private static void checkZoomLevel(long zoomLevel) {
        BingTileFunctions.checkCondition(zoomLevel > 0L, ZOOM_LEVEL_TOO_SMALL, new Object[0]);
        BingTileFunctions.checkCondition(zoomLevel <= 23L, ZOOM_LEVEL_TOO_LARGE, new Object[0]);
    }

    private static void checkCoordinate(long coordinate, long zoomLevel) {
        BingTileFunctions.checkCondition(coordinate >= 0L && coordinate < (long)(1 << (int)zoomLevel), "XY coordinates for a Bing tile at zoom level %s must be within [0, %s) range", zoomLevel, 1 << (int)zoomLevel);
    }

    private static void checkQuadKey(@SqlType(value="varchar") Slice quadkey) {
        BingTileFunctions.checkCondition(quadkey.length() > 0, QUAD_KEY_EMPTY, new Object[0]);
        BingTileFunctions.checkCondition(quadkey.length() <= 23, QUAD_KEY_TOO_LONG, new Object[0]);
    }

    private static void checkLatitude(double latitude, String errorMessage) {
        BingTileFunctions.checkCondition(latitude >= -85.05112878 && latitude <= 85.05112878, errorMessage, new Object[0]);
    }

    private static void checkLongitude(double longitude, String errorMessage) {
        BingTileFunctions.checkCondition(longitude >= -180.0 && longitude <= 180.0, errorMessage, new Object[0]);
    }

    private static boolean withinDistance(GreatCircleDistanceToPoint distanceFunction, double maxDistance, Point point) {
        return distanceFunction.distance(point.getY(), point.getX()) <= maxDistance;
    }

    private static void checkCondition(boolean condition, String formatString, Object ... args) {
        if (!condition) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format(formatString, args));
        }
    }

    private static double clip(double n, double minValue, double maxValue) {
        return Math.min(Math.max(n, minValue), maxValue);
    }

    private static long mapSize(int zoomLevel) {
        return 256L << zoomLevel;
    }

    private static final class GreatCircleDistanceToPoint {
        private double sinLatitude;
        private double cosLatitude;
        private double radianLongitude;

        private GreatCircleDistanceToPoint(double latitude, double longitude) {
            double radianLatitude = Math.toRadians(latitude);
            this.sinLatitude = Math.sin(radianLatitude);
            this.cosLatitude = Math.cos(radianLatitude);
            this.radianLongitude = Math.toRadians(longitude);
        }

        public double distance(double latitude2, double longitude2) {
            double radianLatitude2 = Math.toRadians(latitude2);
            double sin2 = Math.sin(radianLatitude2);
            double cos2 = Math.cos(radianLatitude2);
            double deltaLongitude = this.radianLongitude - Math.toRadians(longitude2);
            double cosDeltaLongitude = Math.cos(deltaLongitude);
            double t1 = cos2 * Math.sin(deltaLongitude);
            double t2 = this.cosLatitude * sin2 - this.sinLatitude * cos2 * cosDeltaLongitude;
            double t3 = this.sinLatitude * sin2 + this.cosLatitude * cos2 * cosDeltaLongitude;
            return Math.atan2(Math.sqrt(t1 * t1 + t2 * t2), t3) * 6371.01;
        }
    }

    @Description(value="Given a Bing tile, returns XY coordinates of the tile")
    @ScalarFunction(value="bing_tile_coordinates")
    public static final class BingTileCoordinatesFunction {
        private static final RowType BING_TILE_COORDINATES_ROW_TYPE = RowType.anonymous((List)ImmutableList.of((Object)IntegerType.INTEGER, (Object)IntegerType.INTEGER));
        private final BufferedRowValueBuilder rowValueBuilder = BufferedRowValueBuilder.createBuffered((RowType)BING_TILE_COORDINATES_ROW_TYPE);

        @SqlType(value="row(x integer,y integer)")
        public Block bingTileCoordinates(@SqlType(value="BingTile") long input) {
            BingTile tile = BingTile.decode(input);
            return this.rowValueBuilder.build(fields -> {
                IntegerType.INTEGER.writeLong((BlockBuilder)fields.get(0), (long)tile.getX());
                IntegerType.INTEGER.writeLong((BlockBuilder)fields.get(1), (long)tile.getY());
            });
        }
    }
}

