/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.segment.local.segment.store;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.io.FileUtils;
import org.apache.pinot.segment.local.segment.store.IndexEntry;
import org.apache.pinot.segment.local.segment.store.IndexKey;
import org.apache.pinot.segment.local.segment.store.TextIndexUtils;
import org.apache.pinot.segment.spi.index.metadata.SegmentMetadataImpl;
import org.apache.pinot.segment.spi.memory.PinotDataBuffer;
import org.apache.pinot.segment.spi.store.ColumnIndexDirectory;
import org.apache.pinot.segment.spi.store.ColumnIndexType;
import org.apache.pinot.segment.spi.store.ColumnIndexUtils;
import org.apache.pinot.spi.env.CommonsConfigurationUtils;
import org.apache.pinot.spi.utils.ReadMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SingleFileIndexDirectory
extends ColumnIndexDirectory {
    private static final Logger LOGGER = LoggerFactory.getLogger(SingleFileIndexDirectory.class);
    private static final long MAGIC_MARKER = -2401053088876085587L;
    private static final int MAGIC_MARKER_SIZE_BYTES = 8;
    private static final int MAX_ALLOCATION_SIZE = 0x7D000000;
    private final File _segmentDirectory;
    private SegmentMetadataImpl _segmentMetadata;
    private final ReadMode _readMode;
    private final File _indexFile;
    private final Map<IndexKey, IndexEntry> _columnEntries;
    private final List<PinotDataBuffer> _allocBuffers;
    private PinotDataBuffer _starTreeIndexDataBuffer;
    private boolean _shouldCleanupRemovedIndices;

    public SingleFileIndexDirectory(File segmentDirectory, SegmentMetadataImpl segmentMetadata, ReadMode readMode) throws IOException, ConfigurationException {
        Preconditions.checkNotNull((Object)segmentDirectory);
        Preconditions.checkNotNull((Object)readMode);
        Preconditions.checkNotNull((Object)segmentMetadata);
        Preconditions.checkArgument((boolean)segmentDirectory.exists(), (Object)("SegmentDirectory: " + segmentDirectory.toString() + " does not exist"));
        Preconditions.checkArgument((boolean)segmentDirectory.isDirectory(), (Object)("SegmentDirectory: " + segmentDirectory.toString() + " is not a directory"));
        this._segmentDirectory = segmentDirectory;
        this._segmentMetadata = segmentMetadata;
        this._readMode = readMode;
        this._indexFile = new File(segmentDirectory, "columns.psf");
        if (!this._indexFile.exists()) {
            this._indexFile.createNewFile();
        }
        this._columnEntries = new HashMap<IndexKey, IndexEntry>(this._segmentMetadata.getAllColumns().size());
        this._allocBuffers = new ArrayList<PinotDataBuffer>();
        this.load();
    }

    public void setSegmentMetadata(SegmentMetadataImpl segmentMetadata) {
        this._segmentMetadata = segmentMetadata;
    }

    public PinotDataBuffer getBuffer(String column, ColumnIndexType type) throws IOException {
        return this.checkAndGetIndexBuffer(column, type);
    }

    public PinotDataBuffer newBuffer(String column, ColumnIndexType type, long sizeBytes) throws IOException {
        return this.allocNewBufferInternal(column, type, sizeBytes, type.name().toLowerCase() + ".create");
    }

    public boolean hasIndexFor(String column, ColumnIndexType type) {
        if (type == ColumnIndexType.TEXT_INDEX) {
            return TextIndexUtils.hasTextIndex(this._segmentDirectory, column);
        }
        IndexKey key = new IndexKey(column, type);
        return this._columnEntries.containsKey(key);
    }

    private PinotDataBuffer checkAndGetIndexBuffer(String column, ColumnIndexType type) {
        IndexKey key = new IndexKey(column, type);
        IndexEntry entry = this._columnEntries.get(key);
        if (entry == null || entry._buffer == null) {
            throw new RuntimeException("Could not find index for column: " + column + ", type: " + type + ", segment: " + this._segmentDirectory.toString());
        }
        return entry._buffer;
    }

    private PinotDataBuffer allocNewBufferInternal(String column, ColumnIndexType indexType, long size, String context) throws IOException {
        IndexKey key = new IndexKey(column, indexType);
        this.checkKeyNotPresent(key);
        String allocContext = this.allocationContext(key) + context;
        IndexEntry entry = new IndexEntry(key);
        entry._startOffset = this._indexFile.length();
        entry._size = size + 8L;
        PinotDataBuffer appendBuffer = PinotDataBuffer.mapFile((File)this._indexFile, (boolean)false, (long)entry._startOffset, (long)entry._size, (ByteOrder)ByteOrder.BIG_ENDIAN, (String)allocContext);
        LOGGER.debug("Allotted buffer for key: {}, startOffset: {}, size: {}", new Object[]{key, entry._startOffset, entry._size});
        appendBuffer.putLong(0, -2401053088876085587L);
        this._allocBuffers.add(appendBuffer);
        entry._buffer = appendBuffer.view(8L, entry._size);
        this._columnEntries.put(key, entry);
        this.persistIndexMap(entry);
        return entry._buffer;
    }

    private void checkKeyNotPresent(IndexKey key) {
        if (this._columnEntries.containsKey(key)) {
            throw new RuntimeException("Attempt to re-create an existing index for key: " + key.toString() + ", for segmentDirectory: " + this._segmentDirectory.getAbsolutePath());
        }
    }

    private void validateMagicMarker(PinotDataBuffer buffer, long startOffset) {
        long actualMarkerValue = buffer.getLong(startOffset);
        if (actualMarkerValue != -2401053088876085587L) {
            LOGGER.error("Missing magic marker in index file: {} at position: {}", (Object)this._indexFile, (Object)startOffset);
            throw new RuntimeException("Inconsistent data read. Index data file " + this._indexFile.toString() + " is possibly corrupted");
        }
    }

    private void load() throws IOException, ConfigurationException {
        this.loadMap();
        this.mapBufferEntries();
        if (this._segmentMetadata.getStarTreeV2MetadataList() != null) {
            this.loadStarTreeIndex();
        }
    }

    private void loadStarTreeIndex() throws IOException {
        File indexFile = new File(this._segmentDirectory, "star_tree_index");
        this._starTreeIndexDataBuffer = this._readMode == ReadMode.heap ? PinotDataBuffer.loadFile((File)indexFile, (long)0L, (long)indexFile.length(), (ByteOrder)ByteOrder.BIG_ENDIAN, (String)"Star-tree V2 data buffer") : PinotDataBuffer.mapFile((File)indexFile, (boolean)true, (long)0L, (long)indexFile.length(), (ByteOrder)ByteOrder.BIG_ENDIAN, (String)"Star-tree V2 data buffer");
    }

    private void loadMap() throws ConfigurationException {
        File mapFile = new File(this._segmentDirectory, "index_map");
        PropertiesConfiguration mapConfig = CommonsConfigurationUtils.fromFile((File)mapFile);
        for (String string : CommonsConfigurationUtils.getKeys((Configuration)mapConfig)) {
            String[] parsedKeys = ColumnIndexUtils.parseIndexMapKeys((String)string, (String)this._segmentDirectory.getPath());
            IndexKey indexKey = new IndexKey(parsedKeys[0], ColumnIndexType.getValue((String)parsedKeys[1]));
            IndexEntry entry = this._columnEntries.get(indexKey);
            if (entry == null) {
                entry = new IndexEntry(indexKey);
                this._columnEntries.put(indexKey, entry);
            }
            if (parsedKeys[2].equals("startOffset")) {
                entry._startOffset = mapConfig.getLong(string);
                continue;
            }
            if (parsedKeys[2].equals("size")) {
                entry._size = mapConfig.getLong(string);
                continue;
            }
            throw new ConfigurationException("Invalid map file key: " + string + ", segmentDirectory: " + this._segmentDirectory.toString());
        }
        for (Map.Entry entry : this._columnEntries.entrySet()) {
            IndexEntry entry2 = (IndexEntry)entry.getValue();
            if (entry2._size >= 0L && entry2._startOffset >= 0L) continue;
            throw new ConfigurationException("Invalid map entry for key: " + ((IndexKey)entry.getKey()).toString() + ", segment: " + this._segmentDirectory.toString());
        }
    }

    private void mapBufferEntries() throws IOException {
        TreeMap<Long, IndexEntry> indexStartMap = new TreeMap<Long, IndexEntry>();
        for (Map.Entry<IndexKey, IndexEntry> columnEntry : this._columnEntries.entrySet()) {
            long startOffset = columnEntry.getValue()._startOffset;
            indexStartMap.put(startOffset, columnEntry.getValue());
        }
        long runningSize = 0L;
        ArrayList<Long> offsetAccum = new ArrayList<Long>();
        for (Map.Entry offsetEntry : indexStartMap.entrySet()) {
            IndexEntry entry = (IndexEntry)offsetEntry.getValue();
            if ((runningSize += entry._size) >= 0x7D000000L && !offsetAccum.isEmpty()) {
                this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetEntry.getKey());
                runningSize = entry._size;
                offsetAccum.clear();
            }
            offsetAccum.add((Long)offsetEntry.getKey());
        }
        if (!offsetAccum.isEmpty()) {
            this.mapAndSliceFile(indexStartMap, offsetAccum, (Long)offsetAccum.get(0) + runningSize);
        }
    }

    private void mapAndSliceFile(SortedMap<Long, IndexEntry> startOffsets, List<Long> offsetAccum, long endOffset) throws IOException {
        Preconditions.checkNotNull(startOffsets);
        Preconditions.checkNotNull(offsetAccum);
        Preconditions.checkArgument((!offsetAccum.isEmpty() ? 1 : 0) != 0);
        long fromFilePos = offsetAccum.get(0);
        long size = endOffset - fromFilePos;
        String context = this.allocationContext(this._indexFile, "single_file_index.rw.." + String.valueOf(fromFilePos) + "." + String.valueOf(size));
        PinotDataBuffer buffer = this._readMode == ReadMode.heap ? PinotDataBuffer.loadFile((File)this._indexFile, (long)fromFilePos, (long)size, (ByteOrder)ByteOrder.BIG_ENDIAN, (String)context) : PinotDataBuffer.mapFile((File)this._indexFile, (boolean)true, (long)fromFilePos, (long)size, (ByteOrder)ByteOrder.BIG_ENDIAN, (String)context);
        this._allocBuffers.add(buffer);
        long prevSlicePoint = 0L;
        for (Long fileOffset : offsetAccum) {
            IndexEntry entry = (IndexEntry)startOffsets.get(fileOffset);
            long endSlicePoint = prevSlicePoint + entry._size;
            this.validateMagicMarker(buffer, prevSlicePoint);
            entry._buffer = buffer.view(prevSlicePoint + 8L, endSlicePoint);
            prevSlicePoint = endSlicePoint;
        }
    }

    private void persistIndexMap(IndexEntry entry) throws IOException {
        File mapFile = new File(this._segmentDirectory, "index_map");
        try (PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(mapFile, true)));){
            SingleFileIndexDirectory.persistIndexMap(entry, writer);
        }
    }

    private String allocationContext(IndexKey key) {
        return ((Object)((Object)this)).getClass().getSimpleName() + key.toString();
    }

    private void cleanupRemovedIndices() throws IOException {
        File tmpIdxFile = new File(this._segmentDirectory, "columns.psf.tmp");
        List<IndexEntry> retained = SingleFileIndexDirectory.copyIndices(this._indexFile, tmpIdxFile, new TreeMap<IndexKey, IndexEntry>(this._columnEntries));
        FileUtils.deleteQuietly((File)this._indexFile);
        Preconditions.checkState((boolean)tmpIdxFile.renameTo(this._indexFile), (String)"Failed to rename temp index file: %s to original index file: %s", (Object)tmpIdxFile, (Object)this._indexFile);
        File mapFile = new File(this._segmentDirectory, "index_map");
        FileUtils.deleteQuietly((File)mapFile);
        try (PrintWriter writer = new PrintWriter(new BufferedWriter(new FileWriter(mapFile)));){
            SingleFileIndexDirectory.persistIndexMaps(retained, writer);
        }
    }

    public void close() throws IOException {
        for (PinotDataBuffer buf : this._allocBuffers) {
            buf.close();
        }
        if (this._starTreeIndexDataBuffer != null) {
            this._starTreeIndexDataBuffer.close();
        }
        if (this._shouldCleanupRemovedIndices) {
            this.cleanupRemovedIndices();
        }
        this._columnEntries.clear();
        this._allocBuffers.clear();
    }

    public void removeIndex(String columnName, ColumnIndexType indexType) {
        if (indexType == ColumnIndexType.TEXT_INDEX) {
            TextIndexUtils.cleanupTextIndex(this._segmentDirectory, columnName);
            return;
        }
        if (this._columnEntries.remove(new IndexKey(columnName, indexType)) != null) {
            this._shouldCleanupRemovedIndices = true;
        }
    }

    public Set<String> getColumnsWithIndex(ColumnIndexType type) {
        HashSet<String> columns = new HashSet<String>();
        if (type == ColumnIndexType.TEXT_INDEX) {
            for (String column : this._segmentMetadata.getAllColumns()) {
                if (!TextIndexUtils.hasTextIndex(this._segmentDirectory, column)) continue;
                columns.add(column);
            }
            return columns;
        }
        for (IndexKey indexKey : this._columnEntries.keySet()) {
            if (indexKey._type != type) continue;
            columns.add(indexKey._name);
        }
        return columns;
    }

    public PinotDataBuffer getStarTreeIndex() throws IOException {
        return this._starTreeIndexDataBuffer;
    }

    public InputStream getStarTreeIndexMap() throws IOException {
        return new FileInputStream(new File(this._segmentDirectory, "star_tree_index_map"));
    }

    public String toString() {
        return this._segmentDirectory.toString() + "/" + this._indexFile.toString();
    }

    @VisibleForTesting
    static List<IndexEntry> copyIndices(File srcFile, File destFile, Map<IndexKey, IndexEntry> indicesToCopy) throws IOException {
        ArrayList<IndexEntry> retained = new ArrayList<IndexEntry>();
        long nextOffset = 0L;
        try (FileChannel srcCh = new RandomAccessFile(srcFile, "r").getChannel();
             FileChannel dstCh = new RandomAccessFile(destFile, "rw").getChannel();){
            for (IndexEntry index : indicesToCopy.values()) {
                srcCh.transferTo(index._startOffset, index._size, dstCh);
                retained.add(new IndexEntry(index._key, nextOffset, index._size));
                nextOffset += index._size;
            }
        }
        return retained;
    }

    private static String getKey(String column, String indexName, boolean isStartOffset) {
        return column + "." + indexName + "." + (isStartOffset ? "startOffset" : "size");
    }

    @VisibleForTesting
    static void persistIndexMaps(List<IndexEntry> entries, PrintWriter writer) {
        for (IndexEntry entry : entries) {
            SingleFileIndexDirectory.persistIndexMap(entry, writer);
        }
    }

    private static void persistIndexMap(IndexEntry entry, PrintWriter writer) {
        String colName = entry._key._name;
        String idxType = entry._key._type.getIndexName();
        String startKey = SingleFileIndexDirectory.getKey(colName, idxType, true);
        StringBuilder sb = new StringBuilder();
        sb.append(startKey).append(" = ").append(entry._startOffset);
        writer.println(sb);
        String endKey = SingleFileIndexDirectory.getKey(colName, idxType, false);
        sb = new StringBuilder();
        sb.append(endKey).append(" = ").append(entry._size);
        writer.println(sb);
    }
}

