/*
 * Decompiled with CFR 0.152.
 */
package io.trino.orc;

import com.google.common.base.Joiner;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.errorprone.annotations.FormatMethod;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.orc.CachingOrcDataSource;
import io.trino.orc.MemoryOrcDataSource;
import io.trino.orc.NameBasedFieldMapper;
import io.trino.orc.OrcColumn;
import io.trino.orc.OrcCorruptionException;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcDataSourceId;
import io.trino.orc.OrcDecompressor;
import io.trino.orc.OrcPredicate;
import io.trino.orc.OrcReaderOptions;
import io.trino.orc.OrcRecordReader;
import io.trino.orc.OrcWriteValidation;
import io.trino.orc.metadata.ColumnMetadata;
import io.trino.orc.metadata.CompressionKind;
import io.trino.orc.metadata.ExceptionWrappingMetadataReader;
import io.trino.orc.metadata.Footer;
import io.trino.orc.metadata.Metadata;
import io.trino.orc.metadata.OrcColumnId;
import io.trino.orc.metadata.OrcMetadataReader;
import io.trino.orc.metadata.OrcType;
import io.trino.orc.metadata.PostScript;
import io.trino.orc.stream.OrcChunkLoader;
import io.trino.orc.stream.OrcInputStream;
import io.trino.spi.Page;
import io.trino.spi.type.Type;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.joda.time.DateTimeZone;

public class OrcReader {
    public static final int MAX_BATCH_SIZE = 8192;
    public static final int INITIAL_BATCH_SIZE = 1;
    public static final int BATCH_SIZE_GROWTH_FACTOR = 2;
    private static final Logger log = Logger.get(OrcReader.class);
    private static final int CURRENT_MAJOR_VERSION = 0;
    private static final int CURRENT_MINOR_VERSION = 12;
    private static final int EXPECTED_FOOTER_SIZE = 16384;
    private final OrcDataSource orcDataSource;
    private final ExceptionWrappingMetadataReader metadataReader;
    private final OrcReaderOptions options;
    private final PostScript.HiveWriterVersion hiveWriterVersion;
    private final int bufferSize;
    private final CompressionKind compressionKind;
    private final Optional<OrcDecompressor> decompressor;
    private final Footer footer;
    private final Metadata metadata;
    private final OrcColumn rootColumn;
    private final Optional<OrcWriteValidation> writeValidation;

    public static Optional<OrcReader> createOrcReader(OrcDataSource orcDataSource, OrcReaderOptions options) throws IOException {
        return OrcReader.createOrcReader(orcDataSource, options, Optional.empty());
    }

    private static Optional<OrcReader> createOrcReader(OrcDataSource orcDataSource, OrcReaderOptions options, Optional<OrcWriteValidation> writeValidation) throws IOException {
        long estimatedFileSize = (orcDataSource = OrcReader.wrapWithCacheIfTiny(orcDataSource, options.getTinyStripeThreshold())).getEstimatedSize();
        if (estimatedFileSize > 0L && estimatedFileSize <= (long)PostScript.MAGIC.length()) {
            throw new OrcCorruptionException(orcDataSource.getId(), "Invalid file size %s", estimatedFileSize);
        }
        long expectedReadSize = Math.min(estimatedFileSize, 16384L);
        Slice fileTail = orcDataSource.readTail(Math.toIntExact(expectedReadSize));
        if (fileTail.length() == 0) {
            return Optional.empty();
        }
        return Optional.of(new OrcReader(orcDataSource, options, writeValidation, fileTail));
    }

    private OrcReader(OrcDataSource orcDataSource, OrcReaderOptions options, Optional<OrcWriteValidation> writeValidation, Slice fileTail) throws IOException {
        PostScript postScript;
        this.options = Objects.requireNonNull(options, "options is null");
        this.orcDataSource = orcDataSource;
        this.metadataReader = new ExceptionWrappingMetadataReader(orcDataSource.getId(), new OrcMetadataReader(options));
        this.writeValidation = Objects.requireNonNull(writeValidation, "writeValidation is null");
        short postScriptSize = fileTail.getUnsignedByte(fileTail.length() - 1);
        if (postScriptSize >= fileTail.length()) {
            throw new OrcCorruptionException(orcDataSource.getId(), "Invalid postscript length %s", postScriptSize);
        }
        try {
            postScript = this.metadataReader.readPostScript((InputStream)fileTail.slice(fileTail.length() - 1 - postScriptSize, (int)postScriptSize).getInput());
        }
        catch (OrcCorruptionException e) {
            try {
                Slice headerMagic = orcDataSource.readFully(0L, PostScript.MAGIC.length());
                if (!PostScript.MAGIC.equals((Object)headerMagic)) {
                    throw new OrcCorruptionException(orcDataSource.getId(), "Not an ORC file");
                }
            }
            catch (IOException headerMagic) {
                // empty catch block
            }
            throw e;
        }
        OrcReader.checkOrcVersion(orcDataSource, postScript.getVersion());
        this.validateWrite(validation -> validation.getVersion().equals(postScript.getVersion()), "Unexpected version", new Object[0]);
        this.bufferSize = Math.toIntExact(postScript.getCompressionBlockSize());
        this.compressionKind = postScript.getCompression();
        this.decompressor = OrcDecompressor.createOrcDecompressor(orcDataSource.getId(), this.compressionKind, this.bufferSize);
        this.validateWrite(validation -> validation.getCompression() == this.compressionKind, "Unexpected compression", new Object[0]);
        this.hiveWriterVersion = postScript.getHiveWriterVersion();
        int footerSize = Math.toIntExact(postScript.getFooterLength());
        int metadataSize = Math.toIntExact(postScript.getMetadataLength());
        int completeFooterSize = footerSize + metadataSize + postScriptSize + 1;
        Slice completeFooterSlice = completeFooterSize > fileTail.length() ? orcDataSource.readTail(completeFooterSize) : fileTail.slice(fileTail.length() - completeFooterSize, completeFooterSize);
        Slice metadataSlice = completeFooterSlice.slice(0, metadataSize);
        try (OrcInputStream metadataInputStream = new OrcInputStream(OrcChunkLoader.create(orcDataSource.getId(), metadataSlice, this.decompressor, AggregatedMemoryContext.newSimpleAggregatedMemoryContext()));){
            this.metadata = this.metadataReader.readMetadata(this.hiveWriterVersion, metadataInputStream);
        }
        Slice footerSlice = completeFooterSlice.slice(metadataSize, footerSize);
        try (OrcInputStream footerInputStream = new OrcInputStream(OrcChunkLoader.create(orcDataSource.getId(), footerSlice, this.decompressor, AggregatedMemoryContext.newSimpleAggregatedMemoryContext()));){
            this.footer = this.metadataReader.readFooter(this.hiveWriterVersion, footerInputStream);
        }
        if (this.footer.getTypes().size() == 0) {
            throw new OrcCorruptionException(orcDataSource.getId(), "File has no columns");
        }
        this.rootColumn = OrcReader.createOrcColumn("", "", new OrcColumnId(0), this.footer.getTypes(), orcDataSource.getId());
        this.validateWrite(validation -> validation.getColumnNames().equals(this.getColumnNames()), "Unexpected column names", new Object[0]);
        this.validateWrite(validation -> validation.getRowGroupMaxRowCount() == this.footer.getRowsInRowGroup().orElse(0), "Unexpected rows in group", new Object[0]);
        if (writeValidation.isPresent()) {
            writeValidation.get().validateMetadata(orcDataSource.getId(), this.footer.getUserMetadata());
            writeValidation.get().validateFileStatistics(orcDataSource.getId(), this.footer.getFileStats());
            writeValidation.get().validateStripeStatistics(orcDataSource.getId(), this.footer.getStripes(), this.metadata.getStripeStatsList());
        }
    }

    public List<String> getColumnNames() {
        return this.footer.getTypes().get(OrcColumnId.ROOT_COLUMN).getFieldNames();
    }

    public Footer getFooter() {
        return this.footer;
    }

    public Metadata getMetadata() {
        return this.metadata;
    }

    public OrcColumn getRootColumn() {
        return this.rootColumn;
    }

    public int getBufferSize() {
        return this.bufferSize;
    }

    public CompressionKind getCompressionKind() {
        return this.compressionKind;
    }

    public OrcRecordReader createRecordReader(List<OrcColumn> readColumns, List<Type> readTypes, OrcPredicate predicate, DateTimeZone legacyFileTimeZone, AggregatedMemoryContext memoryUsage, int initialBatchSize, Function<Exception, RuntimeException> exceptionTransform) throws OrcCorruptionException {
        return this.createRecordReader(readColumns, readTypes, Collections.nCopies(readColumns.size(), OrcReader.fullyProjectedLayout()), predicate, 0L, this.orcDataSource.getEstimatedSize(), legacyFileTimeZone, memoryUsage, initialBatchSize, exceptionTransform, NameBasedFieldMapper::create);
    }

    public OrcRecordReader createRecordReader(List<OrcColumn> readColumns, List<Type> readTypes, List<ProjectedLayout> readLayouts, OrcPredicate predicate, long offset, long length, DateTimeZone legacyFileTimeZone, AggregatedMemoryContext memoryUsage, int initialBatchSize, Function<Exception, RuntimeException> exceptionTransform, FieldMapperFactory fieldMapperFactory) throws OrcCorruptionException {
        return new OrcRecordReader(Objects.requireNonNull(readColumns, "readColumns is null"), Objects.requireNonNull(readTypes, "readTypes is null"), Objects.requireNonNull(readLayouts, "readLayouts is null"), Objects.requireNonNull(predicate, "predicate is null"), this.footer.getNumberOfRows(), this.footer.getStripes(), this.footer.getFileStats(), this.metadata.getStripeStatsList(), this.orcDataSource, offset, length, this.footer.getTypes(), this.decompressor, this.footer.getRowsInRowGroup(), Objects.requireNonNull(legacyFileTimeZone, "legacyFileTimeZone is null"), this.hiveWriterVersion, this.metadataReader, this.options, this.footer.getUserMetadata(), memoryUsage, this.writeValidation, initialBatchSize, exceptionTransform, fieldMapperFactory);
    }

    private static OrcDataSource wrapWithCacheIfTiny(OrcDataSource dataSource, DataSize maxCacheSize) throws IOException {
        if (dataSource instanceof MemoryOrcDataSource || dataSource instanceof CachingOrcDataSource) {
            return dataSource;
        }
        if (dataSource.getEstimatedSize() > maxCacheSize.toBytes()) {
            return dataSource;
        }
        Slice data = dataSource.readTail(Math.toIntExact(dataSource.getEstimatedSize()));
        dataSource.close();
        return new MemoryOrcDataSource(dataSource.getId(), data);
    }

    private static OrcColumn createOrcColumn(String parentStreamName, String fieldName, OrcColumnId columnId, ColumnMetadata<OrcType> types, OrcDataSourceId orcDataSourceId) {
        String path = fieldName.isEmpty() ? parentStreamName : parentStreamName + "." + fieldName;
        OrcType orcType = types.get(columnId);
        Object nestedColumns = ImmutableList.of();
        if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.STRUCT) {
            nestedColumns = (List)IntStream.range(0, orcType.getFieldCount()).mapToObj(fieldId -> OrcReader.createOrcColumn(path, orcType.getFieldName(fieldId), orcType.getFieldTypeIndex(fieldId), types, orcDataSourceId)).collect(ImmutableList.toImmutableList());
        } else if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.LIST) {
            nestedColumns = ImmutableList.of((Object)OrcReader.createOrcColumn(path, "item", orcType.getFieldTypeIndex(0), types, orcDataSourceId));
        } else if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.MAP) {
            nestedColumns = ImmutableList.of((Object)OrcReader.createOrcColumn(path, "key", orcType.getFieldTypeIndex(0), types, orcDataSourceId), (Object)OrcReader.createOrcColumn(path, "value", orcType.getFieldTypeIndex(1), types, orcDataSourceId));
        } else if (orcType.getOrcTypeKind() == OrcType.OrcTypeKind.UNION) {
            nestedColumns = (List)IntStream.range(0, orcType.getFieldCount()).mapToObj(fieldId -> OrcReader.createOrcColumn(path, "field" + fieldId, orcType.getFieldTypeIndex(fieldId), types, orcDataSourceId)).collect(ImmutableList.toImmutableList());
        }
        return new OrcColumn(path, columnId, fieldName, orcType.getOrcTypeKind(), orcDataSourceId, (List<OrcColumn>)nestedColumns, orcType.getAttributes());
    }

    private static void checkOrcVersion(OrcDataSource orcDataSource, List<Integer> version) {
        if (version.size() >= 1) {
            int major = version.get(0);
            int minor = 0;
            if (version.size() > 1) {
                minor = version.get(1);
            }
            if (major > 0 || major == 0 && minor > 12) {
                log.warn("ORC file %s was written by a newer Hive version %s. This file may not be readable by this version of Hive (%s.%s).", new Object[]{orcDataSource, Joiner.on((char)'.').join(version), 0, 12});
            }
        }
    }

    @FormatMethod
    private void validateWrite(Predicate<OrcWriteValidation> test, String messageFormat, Object ... args) throws OrcCorruptionException {
        if (this.writeValidation.isPresent() && !test.test(this.writeValidation.get())) {
            throw new OrcCorruptionException(this.orcDataSource.getId(), "Write validation failed: " + messageFormat, args);
        }
    }

    static void validateFile(OrcWriteValidation writeValidation, OrcDataSource input, List<Type> readTypes) throws OrcCorruptionException {
        try {
            OrcReader orcReader = OrcReader.createOrcReader(input, new OrcReaderOptions(), Optional.of(writeValidation)).orElseThrow(() -> new OrcCorruptionException(input.getId(), "File is empty"));
            try (OrcRecordReader orcRecordReader = orcReader.createRecordReader(orcReader.getRootColumn().getNestedColumns(), readTypes, OrcPredicate.TRUE, DateTimeZone.UTC, AggregatedMemoryContext.newSimpleAggregatedMemoryContext(), 1, exception -> {
                Throwables.throwIfUnchecked((Throwable)exception);
                return new RuntimeException((Throwable)exception);
            });){
                Page page = orcRecordReader.nextPage();
                while (page != null) {
                    page.getLoadedPage();
                    page = orcRecordReader.nextPage();
                }
            }
        }
        catch (IOException e) {
            throw new OrcCorruptionException(e, input.getId(), "Validation failed", new Object[0]);
        }
    }

    public static ProjectedLayout fullyProjectedLayout() {
        return orcColumn -> OrcReader.fullyProjectedLayout();
    }

    public static interface ProjectedLayout {
        public ProjectedLayout getFieldLayout(OrcColumn var1);
    }

    public static interface FieldMapperFactory {
        public FieldMapper create(OrcColumn var1);
    }

    public static interface FieldMapper {
        public OrcColumn get(String var1);
    }

    public static class NameBasedProjectedLayout
    implements ProjectedLayout {
        private final Optional<Map<String, ProjectedLayout>> fieldLayouts;

        private NameBasedProjectedLayout(Optional<Map<String, ProjectedLayout>> fieldLayouts) {
            this.fieldLayouts = Objects.requireNonNull(fieldLayouts, "fieldLayouts is null");
        }

        @Override
        public ProjectedLayout getFieldLayout(OrcColumn orcColumn) {
            String name = orcColumn.getColumnName().toLowerCase(Locale.ENGLISH);
            if (this.fieldLayouts.isPresent()) {
                return this.fieldLayouts.get().get(name);
            }
            return OrcReader.fullyProjectedLayout();
        }

        public static ProjectedLayout createProjectedLayout(OrcColumn root, List<List<String>> dereferences) {
            if (dereferences.stream().map(List::size).anyMatch(Predicate.isEqual(0))) {
                return OrcReader.fullyProjectedLayout();
            }
            Map dereferencesByField = dereferences.stream().collect(Collectors.groupingBy(sequence -> (String)sequence.get(0), Collectors.mapping(sequence -> sequence.subList(1, sequence.size()), Collectors.toList())));
            ImmutableMap.Builder fieldLayouts = ImmutableMap.builder();
            for (OrcColumn nestedColumn : root.getNestedColumns()) {
                String fieldName = nestedColumn.getColumnName().toLowerCase(Locale.ENGLISH);
                if (!dereferencesByField.containsKey(fieldName)) continue;
                fieldLayouts.put((Object)fieldName, (Object)NameBasedProjectedLayout.createProjectedLayout(nestedColumn, dereferencesByField.get(fieldName)));
            }
            return new NameBasedProjectedLayout(Optional.of(fieldLayouts.buildOrThrow()));
        }
    }
}

