/*
 * Decompiled with CFR 0.152.
 */
package io.micronaut.http.client.netty;

import io.micronaut.core.annotation.Internal;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.http.client.HttpClientConfiguration;
import io.micronaut.http.client.exceptions.HttpClientException;
import io.micronaut.http.client.netty.BlockHint;
import io.micronaut.http.client.netty.ConnectionManager;
import io.micronaut.http.client.netty.PoolSink;
import java.util.Deque;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.slf4j.Logger;
import reactor.core.publisher.Sinks;

@Internal
abstract class PoolResizer {
    private final Logger log;
    private final HttpClientConfiguration.ConnectionPoolConfiguration connectionPoolConfiguration;
    private final AtomicReference<WorkState> state = new AtomicReference<WorkState>(WorkState.IDLE);
    private final AtomicInteger pendingConnectionCount = new AtomicInteger(0);
    private final Deque<PoolSink<ConnectionManager.PoolHandle>> pendingRequests = new ConcurrentLinkedDeque<PoolSink<ConnectionManager.PoolHandle>>();
    private final List<ResizerConnection> http1Connections = new CopyOnWriteArrayList<ResizerConnection>();
    private final List<ResizerConnection> http2Connections = new CopyOnWriteArrayList<ResizerConnection>();

    PoolResizer(Logger log, HttpClientConfiguration.ConnectionPoolConfiguration connectionPoolConfiguration) {
        this.log = log;
        this.connectionPoolConfiguration = connectionPoolConfiguration;
    }

    private void dirty() {
        WorkState endState;
        WorkState before = this.state.getAndUpdate(ws -> {
            if (ws == WorkState.IDLE) {
                return WorkState.ACTIVE_WITHOUT_PENDING_WORK;
            }
            return WorkState.ACTIVE_WITH_PENDING_WORK;
        });
        if (before != WorkState.IDLE) {
            return;
        }
        do {
            try {
                this.doSomeWork();
            }
            catch (Throwable t) {
                this.state.set(WorkState.IDLE);
                throw t;
            }
        } while ((endState = this.state.updateAndGet(ws -> {
            if (ws == WorkState.ACTIVE_WITH_PENDING_WORK) {
                return WorkState.ACTIVE_WITHOUT_PENDING_WORK;
            }
            return WorkState.IDLE;
        })) != WorkState.IDLE);
    }

    private void doSomeWork() {
        PoolSink<ConnectionManager.PoolHandle> toDispatch;
        BlockHint blockedPendingRequests = null;
        while ((toDispatch = this.pendingRequests.pollFirst()) != null) {
            boolean dispatched = false;
            for (ResizerConnection c : this.http2Connections) {
                if (!this.dispatchSafe(c, toDispatch)) continue;
                dispatched = true;
                break;
            }
            if (!dispatched) {
                for (ResizerConnection c : this.http1Connections) {
                    if (!this.dispatchSafe(c, toDispatch)) continue;
                    dispatched = true;
                    break;
                }
            }
            if (dispatched) continue;
            this.pendingRequests.addFirst(toDispatch);
            blockedPendingRequests = BlockHint.combine(blockedPendingRequests, toDispatch.getBlockHint());
            break;
        }
        int pendingRequestCount = this.pendingRequests.size();
        int pendingConnectionCount = this.pendingConnectionCount.get();
        int http1ConnectionCount = this.http1Connections.size();
        int http2ConnectionCount = this.http2Connections.size();
        if (pendingRequestCount == 0) {
            return;
        }
        int connectionsToOpen = pendingRequestCount - pendingConnectionCount;
        connectionsToOpen = Math.min(connectionsToOpen, this.connectionPoolConfiguration.getMaxPendingConnections() - pendingConnectionCount);
        if (http1ConnectionCount > 0) {
            connectionsToOpen = Math.min(connectionsToOpen, this.connectionPoolConfiguration.getMaxConcurrentHttp1Connections() - http1ConnectionCount);
        }
        if (http2ConnectionCount > 0) {
            connectionsToOpen = Math.min(connectionsToOpen, this.connectionPoolConfiguration.getMaxConcurrentHttp2Connections() - http2ConnectionCount);
        }
        if (connectionsToOpen > 0) {
            this.pendingConnectionCount.addAndGet(connectionsToOpen);
            for (int i = 0; i < connectionsToOpen; ++i) {
                try {
                    this.openNewConnection(blockedPendingRequests);
                    continue;
                }
                catch (Exception e) {
                    try {
                        this.onNewConnectionFailure(e);
                        continue;
                    }
                    catch (Exception f) {
                        this.log.error("Internal error", (Throwable)f);
                    }
                }
            }
            this.dirty();
        }
    }

    private boolean dispatchSafe(ResizerConnection connection, PoolSink<ConnectionManager.PoolHandle> toDispatch) {
        try {
            return connection.dispatch(toDispatch);
        }
        catch (Exception e) {
            try {
                if (toDispatch.tryEmitError(e) != Sinks.EmitResult.OK) {
                    this.log.debug("Failure during connection dispatch operation, but dispatch request was already complete.", (Throwable)e);
                }
            }
            catch (Exception f) {
                this.log.error("Internal error", (Throwable)f);
            }
            return true;
        }
    }

    abstract void openNewConnection(@Nullable BlockHint var1) throws Exception;

    static boolean incrementWithLimit(AtomicInteger variable, int limit) {
        int old;
        do {
            if ((old = variable.get()) < limit) continue;
            return false;
        } while (!variable.compareAndSet(old, old + 1));
        return true;
    }

    void onNewConnectionFailure(@Nullable Throwable error) throws Exception {
        this.pendingConnectionCount.decrementAndGet();
        this.dirty();
    }

    final void onNewConnectionEstablished1(ResizerConnection connection) {
        this.http1Connections.add(connection);
        this.pendingConnectionCount.decrementAndGet();
        this.dirty();
    }

    final void onNewConnectionEstablished2(ResizerConnection connection) {
        this.http2Connections.add(connection);
        this.pendingConnectionCount.decrementAndGet();
        this.dirty();
    }

    final void onConnectionInactive1(ResizerConnection connection) {
        this.http1Connections.remove(connection);
        this.dirty();
    }

    final void onConnectionInactive2(ResizerConnection connection) {
        this.http2Connections.remove(connection);
        this.dirty();
    }

    final void addPendingRequest(PoolSink<ConnectionManager.PoolHandle> sink) {
        int maxPendingAcquires = this.connectionPoolConfiguration.getMaxPendingAcquires();
        if (maxPendingAcquires != Integer.MAX_VALUE && this.pendingRequests.size() >= maxPendingAcquires) {
            sink.tryEmitError(new HttpClientException("Cannot acquire connection, exceeded max pending acquires configuration"));
            return;
        }
        this.pendingRequests.addLast(sink);
        this.dirty();
    }

    @Nullable
    final Sinks.One<ConnectionManager.PoolHandle> pollPendingRequest() {
        Sinks.One req = this.pendingRequests.pollFirst();
        if (req != null) {
            this.dirty();
        }
        return req;
    }

    final void markConnectionAvailable() {
        this.dirty();
    }

    final void forEachConnection(Consumer<ResizerConnection> c) {
        for (ResizerConnection http1Connection : this.http1Connections) {
            c.accept(http1Connection);
        }
        for (ResizerConnection http2Connection : this.http2Connections) {
            c.accept(http2Connection);
        }
    }

    private static enum WorkState {
        IDLE,
        ACTIVE_WITH_PENDING_WORK,
        ACTIVE_WITHOUT_PENDING_WORK;

    }

    static abstract class ResizerConnection {
        ResizerConnection() {
        }

        abstract boolean dispatch(PoolSink<ConnectionManager.PoolHandle> var1) throws Exception;
    }
}

