/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.health;

import com.codahale.metrics.Counter;
import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheckRegistryListener;
import io.dropwizard.health.HealthCheckConfiguration;
import io.dropwizard.health.HealthCheckScheduler;
import io.dropwizard.health.HealthCheckType;
import io.dropwizard.health.HealthStateAggregator;
import io.dropwizard.health.HealthStateListener;
import io.dropwizard.health.HealthStateView;
import io.dropwizard.health.HealthStatusChecker;
import io.dropwizard.health.Schedule;
import io.dropwizard.health.ScheduledHealthCheck;
import io.dropwizard.health.ShutdownNotifier;
import io.dropwizard.health.State;
import io.dropwizard.util.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HealthCheckManager
implements HealthCheckRegistryListener,
HealthStatusChecker,
ShutdownNotifier,
HealthStateListener,
HealthStateAggregator {
    private static final Logger LOGGER = LoggerFactory.getLogger(HealthCheckManager.class);
    private final AtomicBoolean isAppAlive = new AtomicBoolean(true);
    private final AtomicBoolean isAppHealthy = new AtomicBoolean(false);
    private final AtomicInteger unhealthyCriticalHealthChecks = new AtomicInteger();
    private final AtomicInteger unhealthyCriticalAliveChecks = new AtomicInteger();
    @Nonnull
    private final HealthCheckScheduler scheduler;
    @Nonnull
    private final Map<String, HealthCheckConfiguration> configs;
    @Nonnull
    private final Collection<HealthStateListener> healthStateListeners;
    @Nonnull
    private final MetricRegistry metrics;
    private final Duration shutdownWaitPeriod;
    private final boolean initialOverallState;
    @Nonnull
    private final String aggregateHealthyName;
    @Nonnull
    private final String aggregateUnhealthyName;
    @Nonnull
    private Map<String, ScheduledHealthCheck> checks;
    private volatile boolean shuttingDown = false;

    public HealthCheckManager(List<HealthCheckConfiguration> configs, HealthCheckScheduler scheduler, MetricRegistry metrics, Duration shutdownWaitPeriod, boolean initialOverallState, Collection<HealthStateListener> healthStateListeners) {
        this.configs = configs.stream().collect(Collectors.toMap(HealthCheckConfiguration::getName, Function.identity()));
        this.scheduler = Objects.requireNonNull(scheduler);
        this.metrics = Objects.requireNonNull(metrics);
        this.shutdownWaitPeriod = shutdownWaitPeriod;
        this.initialOverallState = initialOverallState;
        this.checks = new HashMap<String, ScheduledHealthCheck>();
        this.healthStateListeners = Objects.requireNonNull(healthStateListeners);
        this.aggregateHealthyName = MetricRegistry.name((String)"health", (String[])new String[]{"aggregate", "healthy"});
        this.aggregateUnhealthyName = MetricRegistry.name((String)"health", (String[])new String[]{"aggregate", "unhealthy"});
        metrics.register(this.aggregateHealthyName, (Metric)((Gauge)this::calculateNumberOfHealthyChecks));
        metrics.register(this.aggregateUnhealthyName, (Metric)((Gauge)this::calculateNumberOfUnhealthyChecks));
    }

    void setChecks(Map<String, ScheduledHealthCheck> checks) {
        this.checks = checks;
    }

    public void onHealthCheckAdded(String name, HealthCheck healthCheck) {
        HealthCheckConfiguration config = this.configs.get(name);
        if (config == null) {
            LOGGER.debug("ignoring registered health check that isn't configured: name={}", (Object)name);
            return;
        }
        Schedule schedule = config.getSchedule();
        HealthCheckType type = config.getType();
        boolean critical = type == HealthCheckType.ALIVE || config.isCritical();
        boolean initialState = config.isInitialState();
        State state = new State(name, schedule.getFailureAttempts(), schedule.getSuccessAttempts(), initialState, this);
        Counter healthyCheckCounter = this.metrics.counter(MetricRegistry.name((String)"health", (String[])new String[]{name, "healthy"}));
        Counter unhealthyCheckCounter = this.metrics.counter(MetricRegistry.name((String)"health", (String[])new String[]{name, "unhealthy"}));
        ScheduledHealthCheck check = new ScheduledHealthCheck(name, type, critical, healthCheck, schedule, state, healthyCheckCounter, unhealthyCheckCounter);
        this.checks.put(name, check);
        if (!initialState && critical) {
            this.handleCriticalHealthChange(check, false);
        }
        this.scheduler.scheduleInitial(check);
    }

    public void onHealthCheckRemoved(String name, HealthCheck healthCheck) {
        this.scheduler.unschedule(name);
    }

    @Override
    public void onStateChanged(String name, boolean isNowHealthy) {
        LOGGER.debug("health check changed state: name={} state={}", (Object)name, (Object)isNowHealthy);
        ScheduledHealthCheck check = this.checks.get(name);
        if (check == null) {
            LOGGER.error("State changed for unconfigured health check: name={} state={}", (Object)name, (Object)isNowHealthy);
            return;
        }
        if (check.isCritical()) {
            this.handleCriticalHealthChange(check, isNowHealthy);
        } else {
            this.handleNonCriticalHealthChange(check, isNowHealthy);
        }
        this.scheduler.schedule(check, isNowHealthy);
        this.healthStateListeners.forEach(listener -> {
            try {
                listener.onStateChanged(name, isNowHealthy);
            }
            catch (RuntimeException e) {
                LOGGER.warn("Exception thrown for healthCheckName: {} from Health State listener onStateChanged: {}", new Object[]{name, listener, e});
            }
        });
        this.healthStateListeners.forEach(listener -> listener.onStateChanged(name, isNowHealthy));
    }

    protected void initializeAppHealth() {
        this.isAppHealthy.set(this.initialOverallState);
    }

    private long calculateNumberOfHealthyChecks() {
        return this.checks.values().stream().filter(ScheduledHealthCheck::isHealthy).count();
    }

    private long calculateNumberOfUnhealthyChecks() {
        return this.checks.values().stream().filter(check -> !check.isHealthy()).count();
    }

    private void handleCriticalHealthChange(ScheduledHealthCheck healthCheck, boolean isNowHealthy) {
        if (isNowHealthy) {
            LOGGER.info("A critical dependency is now healthy: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
            switch (healthCheck.getType()) {
                case ALIVE: {
                    this.updateCriticalStatus(this.isAppAlive, this.unhealthyCriticalAliveChecks.decrementAndGet());
                    return;
                }
                case READY: {
                    if (!this.shuttingDown) {
                        this.updateCriticalStatus(this.isAppHealthy, this.unhealthyCriticalHealthChecks.decrementAndGet());
                    } else {
                        LOGGER.info("Status change is ignored during shutdown: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
                    }
                    return;
                }
            }
        } else {
            HealthCheckConfiguration healthCheckConfiguration = this.configs.get(healthCheck.getName());
            if (healthCheckConfiguration != null && !healthCheckConfiguration.isInitialState() && !healthCheck.isPreviouslyRecovered()) {
                LOGGER.warn("A critical unhealthy initialized dependency has not yet recovered: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
            } else {
                LOGGER.error("A critical dependency is now unhealthy: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
            }
            switch (healthCheck.getType()) {
                case ALIVE: {
                    this.updateCriticalStatus(this.isAppAlive, this.unhealthyCriticalAliveChecks.incrementAndGet());
                    return;
                }
                case READY: {
                    this.updateCriticalStatus(this.isAppHealthy, this.unhealthyCriticalHealthChecks.incrementAndGet());
                    return;
                }
            }
        }
        LOGGER.warn("Unexpected health check type: type={}", (Object)healthCheck.getType());
    }

    private void updateCriticalStatus(AtomicBoolean status, int count) {
        status.set(count == 0);
        LOGGER.debug("current status: unhealthy-critical={}", (Object)count);
    }

    private void handleNonCriticalHealthChange(ScheduledHealthCheck healthCheck, boolean isNowHealthy) {
        if (isNowHealthy) {
            LOGGER.info("A non-critical dependency is now healthy: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
            if (!this.isHealthy() && this.checks.values().stream().filter(ScheduledHealthCheck::isCritical).allMatch(ScheduledHealthCheck::isHealthy)) {
                this.isAppHealthy.compareAndSet(false, true);
            }
        } else {
            HealthCheckConfiguration healthCheckConfiguration = this.configs.get(healthCheck.getName());
            if (healthCheckConfiguration != null && !healthCheckConfiguration.isInitialState() && !healthCheck.isPreviouslyRecovered()) {
                LOGGER.info("A non-critical unhealthy initialized dependency has not yet recovered: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
            } else {
                LOGGER.warn("A non-critical dependency is now unhealthy: name={}, type={}", (Object)healthCheck.getName(), (Object)healthCheck.getType());
            }
        }
    }

    String getAggregateHealthyName() {
        return this.aggregateHealthyName;
    }

    String getAggregateUnhealthyName() {
        return this.aggregateUnhealthyName;
    }

    @Override
    public boolean isHealthy() {
        return this.isAppAlive.get() && this.isAppHealthy.get();
    }

    @Override
    public boolean isHealthy(@Nullable String type) {
        if (HealthCheckType.ALIVE.name().equalsIgnoreCase(type)) {
            return this.isAppAlive.get();
        }
        return this.isHealthy();
    }

    @Override
    public void notifyShutdownStarted() throws Exception {
        this.shuttingDown = true;
        LOGGER.info("delayed shutdown: started (waiting {})", (Object)this.shutdownWaitPeriod);
        this.isAppHealthy.set(false);
        Thread.sleep(this.shutdownWaitPeriod.toMilliseconds());
        LOGGER.info("delayed shutdown: finished");
    }

    @Override
    public void onHealthyCheck(String healthCheckName) {
        this.healthStateListeners.forEach(listener -> {
            try {
                listener.onHealthyCheck(healthCheckName);
            }
            catch (RuntimeException e) {
                LOGGER.warn("Exception thrown for healthCheckName: {} from Health State listener onHealthyCheck: {}", new Object[]{healthCheckName, listener, e});
            }
        });
    }

    @Override
    public void onUnhealthyCheck(String healthCheckName) {
        this.healthStateListeners.forEach(listener -> {
            try {
                listener.onUnhealthyCheck(healthCheckName);
            }
            catch (RuntimeException e) {
                LOGGER.warn("Exception thrown for healthCheckName: {} from Health State listener onUnhealthyCheck: {}", new Object[]{healthCheckName, listener, e});
            }
        });
    }

    @Override
    @Nonnull
    public Collection<HealthStateView> healthStateViews() {
        return this.checks.values().stream().map(ScheduledHealthCheck::view).collect(Collectors.toList());
    }

    @Override
    @Nonnull
    public Optional<HealthStateView> healthStateView(@Nonnull String name) {
        return Optional.ofNullable(this.checks.get(name)).map(ScheduledHealthCheck::view);
    }
}

