/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.rest;

import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileWatcher
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(FileWatcher.class);
    private volatile boolean shutdown;
    private final WatchService watchService;
    private final Path file;
    private final Callback callback;
    private final ExecutorService executorService = Executors.newSingleThreadExecutor(r -> {
        Thread thread = new Thread(r, "file-watcher");
        thread.setDaemon(true);
        return thread;
    });

    public FileWatcher(Path file, Callback callback) throws IOException {
        this.file = file;
        this.watchService = FileSystems.getDefault().newWatchService();
        file.getParent().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.OVERFLOW);
        this.callback = callback;
    }

    public static void onFileChange(Path file, Callback callback) throws IOException {
        log.info("Constructing a new watch service: " + file);
        FileWatcher fileWatcher = new FileWatcher(file, callback);
        fileWatcher.executorService.submit(fileWatcher);
    }

    @Override
    public void run() {
        log.info("Running file watcher service thread");
        try {
            while (!this.shutdown) {
                try {
                    this.handleNextWatchNotification();
                }
                catch (InterruptedException e) {
                    throw e;
                }
                catch (ClosedWatchServiceException e) {
                    this.shutdown = true;
                }
                catch (Exception e) {
                    log.info("Watch service caught exception, will continue:" + e);
                }
            }
        }
        catch (InterruptedException e) {
            log.info("Ending watch due to interrupt");
        }
    }

    private void handleNextWatchNotification() throws InterruptedException {
        log.debug("Waiting for watch key to be signalled: " + this.file);
        WatchKey key = this.watchService.take();
        log.info("Watch key signalled");
        boolean runCallback = false;
        for (WatchEvent<?> event : key.pollEvents()) {
            WatchEvent.Kind<?> kind = event.kind();
            if (kind == StandardWatchEventKinds.OVERFLOW) {
                log.debug("Watch event is OVERFLOW");
                continue;
            }
            if (event.context() != null && !(event.context() instanceof Path)) {
                throw new ClassCastException("Expected `event.context()` to be an instance of " + Path.class + ", but it is " + event.context().getClass());
            }
            Path context = (Path)event.context();
            Path changed = this.file.getParent().resolve(context);
            log.info("Watch event is " + event.kind() + ": " + context + " => " + changed);
            if (changed.equals(this.file)) {
                if (Files.exists(changed, new LinkOption[0])) {
                    log.debug("Watch resolved path exists: " + this.file);
                    runCallback = true;
                    continue;
                }
                log.debug("Watch resolved path does not exist: " + this.file);
                continue;
            }
            log.debug("Watch resolved path is not the same");
        }
        key.reset();
        if (runCallback) {
            try {
                this.callback.run();
            }
            catch (Exception e) {
                log.warn("Hit exception in callback on file watcher", (Throwable)e);
            }
        }
    }

    public void shutdown() {
        this.shutdown = true;
        try {
            this.watchService.close();
            this.executorService.shutdown();
        }
        catch (IOException e) {
            log.info("Error closing watch service", (Throwable)e);
        }
    }

    public static interface Callback {
        public void run() throws Exception;
    }
}

