/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.jdbc.source;

import io.confluent.connect.jdbc.source.JdbcSourceConnectorConfig;
import io.confluent.connect.jdbc.source.TimestampIncrementingOffset;
import io.confluent.connect.jdbc.util.ColumnId;
import io.confluent.connect.jdbc.util.DateTimeUtils;
import io.confluent.connect.jdbc.util.ExpressionBuilder;
import io.confluent.connect.jdbc.util.LruCache;
import java.math.BigDecimal;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.stream.Collectors;
import org.apache.kafka.connect.data.Field;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.errors.DataException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimestampIncrementingCriteria {
    protected static final BigDecimal LONG_MAX_VALUE_AS_BIGDEC = new BigDecimal(Long.MAX_VALUE);
    protected final Logger log = LoggerFactory.getLogger(this.getClass());
    protected final List<ColumnId> timestampColumns;
    protected final ColumnId incrementingColumn;
    protected final TimeZone timeZone;
    private final LruCache<Schema, List<String>> caseAdjustedTimestampColumns;

    public TimestampIncrementingCriteria(ColumnId incrementingColumn, List<ColumnId> timestampColumns, TimeZone timeZone) {
        this.timestampColumns = timestampColumns != null ? timestampColumns : Collections.emptyList();
        this.incrementingColumn = incrementingColumn;
        this.timeZone = timeZone;
        this.caseAdjustedTimestampColumns = timestampColumns != null ? new LruCache(16) : null;
    }

    protected boolean hasTimestampColumns() {
        return !this.timestampColumns.isEmpty();
    }

    protected boolean hasIncrementedColumn() {
        return this.incrementingColumn != null;
    }

    public void whereClause(ExpressionBuilder builder) {
        if (this.hasTimestampColumns() && this.hasIncrementedColumn()) {
            this.timestampIncrementingWhereClause(builder);
        } else if (this.hasTimestampColumns()) {
            this.timestampWhereClause(builder);
        } else if (this.hasIncrementedColumn()) {
            this.incrementingWhereClause(builder);
        }
    }

    public void setQueryParameters(PreparedStatement stmt, CriteriaValues values) throws SQLException {
        if (this.hasTimestampColumns() && this.hasIncrementedColumn()) {
            this.setQueryParametersTimestampIncrementing(stmt, values);
        } else if (this.hasTimestampColumns()) {
            this.setQueryParametersTimestamp(stmt, values);
        } else if (this.hasIncrementedColumn()) {
            this.setQueryParametersIncrementing(stmt, values);
        }
    }

    protected void setQueryParametersTimestampIncrementing(PreparedStatement stmt, CriteriaValues values) throws SQLException {
        Timestamp beginTime = values.beginTimestampValue();
        Timestamp endTime = values.endTimestampValue();
        Long incOffset = values.lastIncrementedValue();
        stmt.setTimestamp(1, endTime, DateTimeUtils.getTimeZoneCalendar(this.timeZone));
        stmt.setTimestamp(2, beginTime, DateTimeUtils.getTimeZoneCalendar(this.timeZone));
        stmt.setLong(3, incOffset);
        stmt.setTimestamp(4, beginTime, DateTimeUtils.getTimeZoneCalendar(this.timeZone));
        this.log.debug("Executing prepared statement with start time value = {} end time = {} and incrementing value = {}", new Object[]{DateTimeUtils.formatTimestamp(beginTime, this.timeZone), DateTimeUtils.formatTimestamp(endTime, this.timeZone), incOffset});
    }

    protected void setQueryParametersIncrementing(PreparedStatement stmt, CriteriaValues values) throws SQLException {
        Long incOffset = values.lastIncrementedValue();
        stmt.setLong(1, incOffset);
        this.log.debug("Executing prepared statement with incrementing value = {}", (Object)incOffset);
    }

    protected void setQueryParametersTimestamp(PreparedStatement stmt, CriteriaValues values) throws SQLException {
        Timestamp beginTime = values.beginTimestampValue();
        Timestamp endTime = values.endTimestampValue();
        stmt.setTimestamp(1, beginTime, DateTimeUtils.getTimeZoneCalendar(this.timeZone));
        stmt.setTimestamp(2, endTime, DateTimeUtils.getTimeZoneCalendar(this.timeZone));
        this.log.debug("Executing prepared statement with timestamp value = {} end time = {}", (Object)DateTimeUtils.formatTimestamp(beginTime, this.timeZone), (Object)DateTimeUtils.formatTimestamp(endTime, this.timeZone));
    }

    public TimestampIncrementingOffset extractValues(Schema schema, Struct record, TimestampIncrementingOffset previousOffset, JdbcSourceConnectorConfig.TimestampGranularity timestampGranularity) {
        Timestamp extractedTimestamp = null;
        if (this.hasTimestampColumns()) {
            extractedTimestamp = this.extractOffsetTimestamp(schema, record, timestampGranularity);
            assert (previousOffset == null || previousOffset.getTimestampOffset() != null && previousOffset.getTimestampOffset().compareTo(extractedTimestamp) <= 0);
        }
        Long extractedId = null;
        if (this.hasIncrementedColumn()) {
            extractedId = this.extractOffsetIncrementedId(schema, record);
            assert (previousOffset == null || previousOffset.getIncrementingOffset() == -1L || extractedId > previousOffset.getIncrementingOffset() || this.hasTimestampColumns());
        }
        return new TimestampIncrementingOffset(extractedTimestamp, extractedId);
    }

    protected Timestamp extractOffsetTimestamp(Schema schema, Struct record, JdbcSourceConnectorConfig.TimestampGranularity timestampGranularity) {
        this.caseAdjustedTimestampColumns.computeIfAbsent(schema, this::findCaseSensitiveTimestampColumns);
        for (String timestampColumn : (List)this.caseAdjustedTimestampColumns.get(schema)) {
            Timestamp ts = timestampGranularity.toTimestamp.apply(record.get(timestampColumn), this.timeZone);
            if (ts == null) continue;
            return ts;
        }
        return null;
    }

    protected Long extractOffsetIncrementedId(Schema schema, Struct record) {
        Long extractedId;
        Field field = schema.field(this.incrementingColumn.name());
        if (field == null) {
            throw new DataException("Incrementing column " + this.incrementingColumn.name() + " not found in " + schema.fields().stream().map(Field::name).collect(Collectors.joining(",")));
        }
        Schema incrementingColumnSchema = field.schema();
        Object incrementingColumnValue = record.get(this.incrementingColumn.name());
        if (incrementingColumnValue == null) {
            throw new ConnectException("Null value for incrementing column of type: " + incrementingColumnSchema.type());
        }
        if (this.isIntegralPrimitiveType(incrementingColumnValue)) {
            extractedId = ((Number)incrementingColumnValue).longValue();
        } else if (incrementingColumnSchema.name() != null && incrementingColumnSchema.name().equals("org.apache.kafka.connect.data.Decimal")) {
            extractedId = this.extractDecimalId(incrementingColumnValue);
        } else {
            throw new ConnectException("Invalid type for incrementing column: " + incrementingColumnSchema.type());
        }
        this.log.trace("Extracted incrementing column value: {}", (Object)extractedId);
        return extractedId;
    }

    protected Long extractDecimalId(Object incrementingColumnValue) {
        BigDecimal decimal = (BigDecimal)incrementingColumnValue;
        if (decimal.compareTo(LONG_MAX_VALUE_AS_BIGDEC) > 0) {
            throw new ConnectException("Decimal value for incrementing column exceeded Long.MAX_VALUE");
        }
        if (decimal.scale() != 0) {
            throw new ConnectException("Scale of Decimal value for incrementing column must be 0");
        }
        return decimal.longValue();
    }

    protected boolean isIntegralPrimitiveType(Object incrementingColumnValue) {
        return incrementingColumnValue instanceof Long || incrementingColumnValue instanceof Integer || incrementingColumnValue instanceof Short || incrementingColumnValue instanceof Byte;
    }

    protected String coalesceTimestampColumns(ExpressionBuilder builder) {
        if (this.timestampColumns.size() == 1) {
            builder.append(this.timestampColumns.get(0));
        } else {
            builder.append("COALESCE(");
            builder.appendList().delimitedBy(",").of(this.timestampColumns);
            builder.append(")");
        }
        return builder.toString();
    }

    protected void timestampIncrementingWhereClause(ExpressionBuilder builder) {
        builder.append(" WHERE ");
        this.coalesceTimestampColumns(builder);
        builder.append(" < ? AND ((");
        this.coalesceTimestampColumns(builder);
        builder.append(" = ? AND ");
        builder.append(this.incrementingColumn);
        builder.append(" > ?");
        builder.append(") OR ");
        this.coalesceTimestampColumns(builder);
        builder.append(" > ?)");
        builder.append(" ORDER BY ");
        this.coalesceTimestampColumns(builder);
        builder.append(",");
        builder.append(this.incrementingColumn);
        builder.append(" ASC");
    }

    protected void incrementingWhereClause(ExpressionBuilder builder) {
        builder.append(" WHERE ");
        builder.append(this.incrementingColumn);
        builder.append(" > ?");
        builder.append(" ORDER BY ");
        builder.append(this.incrementingColumn);
        builder.append(" ASC");
    }

    protected void timestampWhereClause(ExpressionBuilder builder) {
        builder.append(" WHERE ");
        this.coalesceTimestampColumns(builder);
        builder.append(" > ? AND ");
        this.coalesceTimestampColumns(builder);
        builder.append(" < ? ORDER BY ");
        this.coalesceTimestampColumns(builder);
        builder.append(" ASC");
    }

    private List<String> findCaseSensitiveTimestampColumns(Schema schema) {
        Map<String, List<String>> caseInsensitiveColumns = schema.fields().stream().map(Field::name).collect(Collectors.groupingBy(String::toLowerCase));
        ArrayList<String> result = new ArrayList<String>();
        for (ColumnId timestampColumn : this.timestampColumns) {
            String columnName = timestampColumn.name();
            if (schema.field(columnName) != null) {
                this.log.trace("Timestamp column name {} case-sensitively matches column read from database", (Object)columnName);
                result.add(columnName);
                continue;
            }
            this.log.debug("Timestamp column name {} not found in columns read from database; falling back to a case-insensitive search", (Object)columnName);
            List<String> caseInsensitiveMatches = caseInsensitiveColumns.get(columnName.toLowerCase());
            if (caseInsensitiveMatches == null || caseInsensitiveMatches.isEmpty()) {
                throw new DataException("Timestamp column " + columnName + " not found in " + schema.fields().stream().map(Field::name).collect(Collectors.joining(",")));
            }
            if (caseInsensitiveMatches.size() > 1) {
                throw new DataException("Timestamp column " + columnName + " not found in columns read from database: " + schema.fields().stream().map(Field::name).collect(Collectors.joining(",")) + ". Could not fall back to case-insensitively selecting a column because there were multiple columns whose names case-insensitively matched the specified name: " + String.join((CharSequence)",", caseInsensitiveMatches) + ". To force the connector to choose between these columns, specify a value for the timestamp column configuration property that matches the desired column case-sensitively.");
            }
            String caseAdjustedColumnName = caseInsensitiveMatches.get(0);
            this.log.debug("Falling back on column {} for user-specified timestamp column {} (this is the only column that case-insensitively matches)", (Object)caseAdjustedColumnName, (Object)columnName);
            result.add(caseAdjustedColumnName);
        }
        return result;
    }

    public static interface CriteriaValues {
        public Timestamp beginTimestampValue() throws SQLException;

        public Timestamp endTimestampValue() throws SQLException;

        public Long lastIncrementedValue() throws SQLException;
    }
}

