/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.license;

import io.confluent.license.InvalidLicenseException;
import io.confluent.license.License;
import io.confluent.license.LicenseManager;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class LicenseManagers {
    public static final LicenseManagers INSTANCE = new LicenseManagers(Executors::newSingleThreadScheduledExecutor);
    private static final Logger log = LoggerFactory.getLogger(LicenseManagers.class);
    private final AtomicReference<ScheduledExecutorService> executor = new AtomicReference<Object>(null);
    private final Map<LicenseManager, RunningLicenseManager> registered = new HashMap<LicenseManager, RunningLicenseManager>();
    private final Supplier<ScheduledExecutorService> execServiceFactory;

    LicenseManagers(Supplier<ScheduledExecutorService> execServiceFactory) {
        Objects.requireNonNull(execServiceFactory);
        this.execServiceFactory = execServiceFactory;
    }

    synchronized ScheduledFuture<?> start(LicenseManager licenseManager) {
        Objects.nonNull(licenseManager);
        this.executor.updateAndGet(existing -> existing != null ? existing : this.execServiceFactory.get());
        RunningLicenseManager running = this.registered.get(licenseManager);
        if (running == null) {
            running = new RunningLicenseManager(licenseManager, this.executor.get(), this::cancelled);
            this.registered.put(licenseManager, running);
            licenseManager.doStart();
        }
        return running;
    }

    synchronized boolean stop(LicenseManager licenseManager) {
        RunningLicenseManager running;
        boolean stopped = false;
        if (licenseManager != null && (running = this.registered.remove(licenseManager)) != null) {
            running.cancel(true);
            stopped = true;
        }
        if (this.registered.isEmpty()) {
            this.shutdown();
        }
        return stopped;
    }

    synchronized void shutdown() {
        if (this.executor.get() != null) {
            for (RunningLicenseManager running : new ArrayList<RunningLicenseManager>(this.registered.values())) {
                this.stop(running.licenseManager);
            }
            this.registered.clear();
            ScheduledExecutorService service = this.executor.getAndSet(null);
            if (service != null) {
                service.shutdownNow();
            }
        }
    }

    protected void cancelled(LicenseManager licenseManager) {
        if (licenseManager != null) {
            this.registered.remove(licenseManager);
        }
    }

    protected boolean isRunning() {
        return this.executor.get() != null;
    }

    protected static class RunningLicenseManager
    implements ScheduledFuture<Object> {
        private final ScheduledExecutorService executor;
        private final Consumer<LicenseManager> cancelListener;
        final LicenseManager licenseManager;
        final ScheduledFuture<?> future;
        private boolean loggedError = false;

        RunningLicenseManager(LicenseManager mgr, ScheduledExecutorService executor, Consumer<LicenseManager> cancelListener) {
            this.licenseManager = mgr;
            this.executor = executor;
            this.cancelListener = cancelListener;
            this.future = executor.scheduleAtFixedRate(this::checkLicense, 1L, 1L, TimeUnit.DAYS);
        }

        protected void checkLicense() {
            try {
                long timeRemainingMillis;
                if (this.executor.isTerminated()) {
                    return;
                }
                License license = null;
                try {
                    license = this.licenseManager.registerOrValidateLicense("");
                    timeRemainingMillis = license.timeRemaining(TimeUnit.MILLISECONDS);
                    this.loggedError = false;
                }
                catch (InvalidLicenseException e) {
                    if (!this.loggedError) {
                        log.error("Stored license was not valid. Please register a valid Confluent license. Will check again every 15 seconds until a valid license is found, but this log message will not repeat", (Throwable)e);
                        this.loggedError = true;
                    }
                    timeRemainingMillis = TimeUnit.SECONDS.toMillis(15L);
                }
                long millisBeforeNextCheck = this.millisForNextLicenseCheck(timeRemainingMillis);
                if (millisBeforeNextCheck > 0L) {
                    try {
                        log.debug("Scheduling next license check in {}", (Object)this.durationOfMillis(millisBeforeNextCheck));
                        this.executor.schedule(this::checkLicense, millisBeforeNextCheck, TimeUnit.MILLISECONDS);
                    }
                    catch (RejectedExecutionException e) {
                        log.error("Unable to schedule next license check in {}", (Object)this.durationOfMillis(millisBeforeNextCheck));
                    }
                }
            }
            catch (Throwable t) {
                log.error("Failed to validate the Confluent license or schedule the next license check", t);
            }
        }

        protected String durationOfMillis(long millis) {
            long timeInSeconds = TimeUnit.MILLISECONDS.toSeconds(millis);
            long sec = timeInSeconds % 60L;
            long minutes = timeInSeconds % 3600L / 60L;
            long hours = timeInSeconds % 86400L / 3600L;
            return String.format("%s:%s:%s", hours, minutes, sec);
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return this.future.getDelay(unit);
        }

        @Override
        public int compareTo(Delayed o) {
            return this.future.compareTo(o);
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            boolean result = this.future.cancel(mayInterruptIfRunning);
            this.licenseManager.doStop();
            this.cancelListener.accept(this.licenseManager);
            return result;
        }

        @Override
        public boolean isCancelled() {
            return this.future.isCancelled();
        }

        @Override
        public boolean isDone() {
            return this.future.isDone();
        }

        @Override
        public Object get() throws InterruptedException, ExecutionException {
            return this.future.get();
        }

        @Override
        public Object get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            return this.future.get(timeout, unit);
        }

        public int hashCode() {
            return this.future.hashCode();
        }

        public boolean equals(Object obj) {
            return obj == this || obj == this.future;
        }

        protected long millisForNextLicenseCheck(long timeRemainingMillis) {
            if (timeRemainingMillis <= 0L) {
                return TimeUnit.SECONDS.toMillis(15L);
            }
            if (timeRemainingMillis < TimeUnit.MINUTES.toMillis(5L)) {
                return TimeUnit.SECONDS.toMillis(15L);
            }
            if (timeRemainingMillis < TimeUnit.MINUTES.toMillis(60L)) {
                return TimeUnit.MINUTES.toMillis(5L);
            }
            if (timeRemainingMillis < TimeUnit.HOURS.toMillis(24L)) {
                return TimeUnit.HOURS.toMillis(1L);
            }
            return -1L;
        }
    }
}

