/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.spi;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.Digest;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.spi.CertificateExtensionsUtils;
import eu.europa.esig.dss.spi.DSSASN1Utils;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.x509.ResponderId;
import eu.europa.esig.dss.spi.x509.revocation.RevocationToken;
import eu.europa.esig.dss.utils.Utils;
import java.io.IOException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.esf.OtherHash;
import org.bouncycastle.asn1.ocsp.BasicOCSPResponse;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.ocsp.OCSPResponse;
import org.bouncycastle.asn1.ocsp.OCSPResponseStatus;
import org.bouncycastle.asn1.ocsp.ResponderID;
import org.bouncycastle.asn1.ocsp.ResponseBytes;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.CertificateID;
import org.bouncycastle.cert.ocsp.OCSPException;
import org.bouncycastle.cert.ocsp.OCSPResp;
import org.bouncycastle.cert.ocsp.RespID;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.bouncycastle.operator.DigestCalculator;
import org.bouncycastle.operator.DigestCalculatorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaDigestCalculatorProviderBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DSSRevocationUtils {
    private static final Logger LOG = LoggerFactory.getLogger(DSSRevocationUtils.class);
    private static JcaDigestCalculatorProviderBuilder jcaDigestCalculatorProviderBuilder = new JcaDigestCalculatorProviderBuilder();

    private DSSRevocationUtils() {
    }

    public static BasicOCSPResp getBasicOcspResp(ASN1Sequence asn1Sequence) {
        BasicOCSPResp basicOCSPResp = null;
        try {
            BasicOCSPResponse basicOcspResponse = BasicOCSPResponse.getInstance((Object)asn1Sequence);
            basicOCSPResp = new BasicOCSPResp(basicOcspResponse);
        }
        catch (Exception e) {
            LOG.warn("Impossible to create BasicOCSPResp from ASN1Sequence!", (Throwable)e);
        }
        return basicOCSPResp;
    }

    public static OCSPResp getOcspResp(ASN1Sequence asn1Sequence) {
        OCSPResp ocspResp = null;
        try {
            OCSPResponse ocspResponse = OCSPResponse.getInstance((Object)asn1Sequence);
            ocspResp = new OCSPResp(ocspResponse);
        }
        catch (Exception e) {
            LOG.warn("Impossible to create OCSPResp from ASN1Sequence!", (Throwable)e);
        }
        return ocspResp;
    }

    public static BasicOCSPResp fromRespToBasic(OCSPResp ocspResp) {
        BasicOCSPResp basicOCSPResp = null;
        try {
            Object responseObject = ocspResp.getResponseObject();
            if (responseObject instanceof BasicOCSPResp) {
                basicOCSPResp = (BasicOCSPResp)responseObject;
            } else {
                LOG.warn("Unknown OCSP response type: {}", responseObject.getClass());
            }
        }
        catch (OCSPException e) {
            LOG.warn("Impossible to process OCSPResp!", (Throwable)e);
        }
        return basicOCSPResp;
    }

    public static OCSPResp fromBasicToResp(BasicOCSPResp basicOCSPResp) {
        try {
            byte[] encoded = basicOCSPResp.getEncoded();
            return DSSRevocationUtils.fromBasicToResp(encoded);
        }
        catch (IOException e) {
            throw new DSSException(String.format("Unable to convert BasicOCSPResp to OCSPResp : %s", e.getMessage()), (Throwable)e);
        }
    }

    public static byte[] getEncodedFromBasicResp(BasicOCSPResp basicOCSPResp) {
        try {
            if (basicOCSPResp != null) {
                OCSPResp ocspResp = DSSRevocationUtils.fromBasicToResp(basicOCSPResp);
                return ocspResp.getEncoded();
            }
            throw new DSSException("Empty OCSP response");
        }
        catch (IOException e) {
            throw new DSSException(String.format("OCSP encoding error : %s", e.getMessage()), (Throwable)e);
        }
    }

    public static OCSPResp fromBasicToResp(byte[] basicOCSPRespBinary) {
        OCSPResponseStatus responseStatus = new OCSPResponseStatus(0);
        DEROctetString derBasicOCSPResp = new DEROctetString(basicOCSPRespBinary);
        ResponseBytes responseBytes = new ResponseBytes(OCSPObjectIdentifiers.id_pkix_ocsp_basic, (ASN1OctetString)derBasicOCSPResp);
        OCSPResponse ocspResponse = new OCSPResponse(responseStatus, responseBytes);
        return new OCSPResp(ocspResponse);
    }

    public static DigestAlgorithm getUsedDigestAlgorithm(SingleResp singleResp) {
        return DigestAlgorithm.forOID((String)singleResp.getCertID().getHashAlgOID().getId());
    }

    public static boolean matches(CertificateID certId, SingleResp singleResp) {
        CertificateID singleRespCertID = singleResp.getCertID();
        ASN1ObjectIdentifier singleRespCertIDHashAlgOID = singleRespCertID.getHashAlgOID();
        byte[] singleRespCertIDIssuerKeyHash = singleRespCertID.getIssuerKeyHash();
        byte[] singleRespCertIDIssuerNameHash = singleRespCertID.getIssuerNameHash();
        BigInteger singleRespCertIDSerialNumber = singleRespCertID.getSerialNumber();
        ASN1ObjectIdentifier certIdHashAlgOID = certId.getHashAlgOID();
        byte[] certIdIssuerKeyHash = certId.getIssuerKeyHash();
        byte[] certIdIssuerNameHash = certId.getIssuerNameHash();
        BigInteger certIdSerialNumber = certId.getSerialNumber();
        return singleRespCertIDHashAlgOID.equals((ASN1Primitive)certIdHashAlgOID) && Arrays.equals(singleRespCertIDIssuerKeyHash, certIdIssuerKeyHash) && Arrays.equals(singleRespCertIDIssuerNameHash, certIdIssuerNameHash) && singleRespCertIDSerialNumber.equals(certIdSerialNumber);
    }

    public static CertificateID getOCSPCertificateID(CertificateToken cert, CertificateToken issuerCert, DigestAlgorithm digestAlgorithm) {
        try {
            BigInteger serialNumber = cert.getSerialNumber();
            DigestCalculator digestCalculator = DSSRevocationUtils.getDigestCalculator(digestAlgorithm);
            X509CertificateHolder x509CertificateHolder = DSSASN1Utils.getX509CertificateHolder(issuerCert);
            return new CertificateID(digestCalculator, x509CertificateHolder, serialNumber);
        }
        catch (OCSPException e) {
            throw new DSSException("Unable to create CertificateID", (Throwable)e);
        }
    }

    public static DigestCalculator getDigestCalculator(DigestAlgorithm digestAlgorithm) {
        try {
            DigestCalculatorProvider digestCalculatorProvider = jcaDigestCalculatorProviderBuilder.build();
            return digestCalculatorProvider.get(new AlgorithmIdentifier(new ASN1ObjectIdentifier(digestAlgorithm.getOid()), (ASN1Encodable)DERNull.INSTANCE));
        }
        catch (OperatorCreationException e) {
            throw new DSSException(String.format("Unable to create a DigestCalculator instance. DigestAlgorithm %s is not supported", digestAlgorithm.name()), (Throwable)e);
        }
    }

    public static BasicOCSPResp loadOCSPBase64Encoded(String base64Encoded) throws IOException {
        byte[] derEncoded = Utils.fromBase64((String)base64Encoded);
        return DSSRevocationUtils.loadOCSPFromBinaries(derEncoded);
    }

    public static BasicOCSPResp loadOCSPFromBinaries(byte[] binaries) throws IOException {
        OCSPResp ocspResp = new OCSPResp(binaries);
        return DSSRevocationUtils.fromRespToBasic(ocspResp);
    }

    public static byte[] getEncoded(OCSPResp ocspResp) {
        try {
            return ocspResp.getEncoded();
        }
        catch (IOException e) {
            throw new DSSException(String.format("Unable to get binaries of OCSPResp : %s", e.getMessage()), (Throwable)e);
        }
    }

    public static ResponderId getDSSResponderId(RespID respID) {
        ResponderID responderID = respID.toASN1Primitive();
        return DSSRevocationUtils.getDSSResponderId(responderID);
    }

    public static ResponderId getDSSResponderId(ResponderID responderID) {
        return new ResponderId(DSSASN1Utils.toX500Principal(responderID.getName()), responderID.getKeyHash());
    }

    public static List<String> getCRLRevocationTokenKeys(CertificateToken certificateToken) {
        ArrayList<String> revocationKeys = new ArrayList<String>();
        List<String> crlUrls = CertificateExtensionsUtils.getCRLAccessUrls(certificateToken);
        for (String crlUrl : crlUrls) {
            revocationKeys.add(DSSRevocationUtils.getCRLRevocationTokenKey(crlUrl));
        }
        return revocationKeys;
    }

    public static String getCRLRevocationTokenKey(String crlUrl) {
        return DSSUtils.getSHA1Digest(crlUrl);
    }

    public static List<String> getOcspRevocationTokenKeys(CertificateToken certificateToken) {
        ArrayList<String> revocationKeys = new ArrayList<String>();
        List<String> ocspUrls = CertificateExtensionsUtils.getOCSPAccessUrls(certificateToken);
        for (String ocspUrl : ocspUrls) {
            revocationKeys.add(DSSRevocationUtils.getOcspRevocationKey(certificateToken, ocspUrl));
        }
        return revocationKeys;
    }

    public static String getOcspRevocationKey(CertificateToken certificateToken, String ocspUrl) {
        return DSSUtils.getSHA1Digest(certificateToken.getEntityKey() + ":" + ocspUrl);
    }

    public static SingleResp getLatestSingleResponse(BasicOCSPResp basicResponse, CertificateToken certificate, CertificateToken issuer) {
        List<SingleResp> singleResponses = DSSRevocationUtils.getSingleResponses(basicResponse, certificate, issuer);
        if (Utils.isCollectionEmpty(singleResponses)) {
            return null;
        }
        if (singleResponses.size() == 1) {
            return singleResponses.get(0);
        }
        return DSSRevocationUtils.getLatestSingleRespInList(singleResponses);
    }

    private static SingleResp getLatestSingleRespInList(List<SingleResp> singleResponses) {
        Date latest = null;
        SingleResp latestResp = null;
        for (SingleResp singleResp : singleResponses) {
            Date thisUpdate = singleResp.getThisUpdate();
            if (latest != null && !thisUpdate.after(latest)) continue;
            latestResp = singleResp;
            latest = thisUpdate;
        }
        return latestResp;
    }

    public static List<SingleResp> getSingleResponses(BasicOCSPResp basicResponse, CertificateToken certificate, CertificateToken issuer) {
        SingleResp[] responses;
        ArrayList<SingleResp> result = new ArrayList<SingleResp>();
        for (SingleResp singleResp : responses = DSSRevocationUtils.getSingleResps(basicResponse)) {
            DigestAlgorithm usedDigestAlgorithm = DSSRevocationUtils.getUsedDigestAlgorithm(singleResp);
            CertificateID certId = DSSRevocationUtils.getOCSPCertificateID(certificate, issuer, usedDigestAlgorithm);
            if (!DSSRevocationUtils.matches(certId, singleResp)) continue;
            result.add(singleResp);
        }
        return result;
    }

    private static SingleResp[] getSingleResps(BasicOCSPResp basicResponse) {
        try {
            return basicResponse.getResponses();
        }
        catch (Exception e) {
            LOG.warn("Unable to extract SingleResp(s) : {}", (Object)e.getMessage());
            return new SingleResp[0];
        }
    }

    public static Digest getDigest(OtherHash otherHash) {
        if (otherHash != null) {
            DigestAlgorithm digestAlgorithm = DigestAlgorithm.forOID((String)otherHash.getHashAlgorithm().getAlgorithm().getId());
            byte[] digestValue = otherHash.getHashValue();
            return new Digest(digestAlgorithm, digestValue);
        }
        return null;
    }

    public static boolean checkIssuerValidAtRevocationProductionTime(RevocationToken<?> revocationToken, CertificateToken issuerCertificateToken) {
        return issuerCertificateToken != null && issuerCertificateToken.isValidOn(revocationToken.getProductionDate());
    }
}

