/*
 * Decompiled with CFR 0.152.
 */
package ai.vespa.feed.client.impl;

import ai.vespa.feed.client.DocumentId;
import ai.vespa.feed.client.FeedClient;
import ai.vespa.feed.client.FeedException;
import ai.vespa.feed.client.HttpResponse;
import ai.vespa.feed.client.OperationParameters;
import ai.vespa.feed.client.OperationStats;
import ai.vespa.feed.client.Result;
import ai.vespa.feed.client.ResultException;
import ai.vespa.feed.client.ResultParseException;
import ai.vespa.feed.client.impl.Cluster;
import ai.vespa.feed.client.impl.DryrunCluster;
import ai.vespa.feed.client.impl.FeedClientBuilderImpl;
import ai.vespa.feed.client.impl.HttpRequest;
import ai.vespa.feed.client.impl.HttpRequestStrategy;
import ai.vespa.feed.client.impl.JettyCluster;
import ai.vespa.feed.client.impl.RequestStrategy;
import ai.vespa.feed.client.impl.ResultImpl;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonFactoryBuilder;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.StreamReadConstraints;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.LongSupplier;
import java.util.function.Supplier;

class HttpFeedClient
implements FeedClient {
    private static final Duration maxTimeout = Duration.ofMinutes(15L);
    private static final JsonFactory jsonParserFactory = ((JsonFactoryBuilder)new JsonFactoryBuilder().streamReadConstraints(StreamReadConstraints.builder().maxStringLength(Integer.MAX_VALUE).build())).build();
    private final Map<String, Supplier<String>> requestHeaders;
    private final RequestStrategy requestStrategy;
    private final AtomicBoolean closed = new AtomicBoolean();
    private final boolean speedTest;
    private final LongSupplier nanoClock;

    HttpFeedClient(FeedClientBuilderImpl builder) throws IOException {
        this(builder, builder.dryrun ? () -> new DryrunCluster() : () -> new JettyCluster(builder));
    }

    HttpFeedClient(FeedClientBuilderImpl builder, ClusterFactory clusterFactory) throws IOException {
        this(builder, clusterFactory, new HttpRequestStrategy(builder, clusterFactory));
    }

    HttpFeedClient(FeedClientBuilderImpl builder, ClusterFactory clusterFactory, RequestStrategy requestStrategy) throws IOException {
        this.requestHeaders = new HashMap<String, Supplier<String>>(builder.requestHeaders);
        this.requestStrategy = requestStrategy;
        this.speedTest = builder.speedTest;
        this.nanoClock = builder.nanoClock;
        this.verifyConnection(builder, clusterFactory);
    }

    public CompletableFuture<Result> put(DocumentId documentId, String documentJson, OperationParameters params) {
        return this.send("POST", documentId, Objects.requireNonNull(documentJson), params);
    }

    public CompletableFuture<Result> update(DocumentId documentId, String updateJson, OperationParameters params) {
        return this.send("PUT", documentId, Objects.requireNonNull(updateJson), params);
    }

    public CompletableFuture<Result> remove(DocumentId documentId, OperationParameters params) {
        return this.send("DELETE", documentId, null, params);
    }

    public OperationStats stats() {
        return this.requestStrategy.stats();
    }

    public void resetStats() {
        this.requestStrategy.resetStats();
    }

    public FeedClient.CircuitBreaker.State circuitBreakerState() {
        return this.requestStrategy.circuitBreakerState();
    }

    public void close(boolean graceful) {
        this.closed.set(true);
        if (graceful) {
            this.requestStrategy.await();
        }
        this.requestStrategy.destroy();
    }

    private CompletableFuture<Result> send(String method, DocumentId documentId, String operationJson, OperationParameters params) {
        if (this.closed.get()) {
            throw new IllegalStateException("Client is closed");
        }
        HttpRequest request = new HttpRequest(method, HttpFeedClient.getPath(documentId), HttpFeedClient.getQuery(params, this.speedTest), this.requestHeaders, operationJson == null ? null : operationJson.getBytes(StandardCharsets.UTF_8), params.timeout().orElse(maxTimeout), this.nanoClock);
        CompletableFuture<Result> promise = new CompletableFuture<Result>();
        ((CompletableFuture)this.requestStrategy.enqueue(documentId, request).thenApply(response -> HttpFeedClient.toResult(request, response, documentId))).whenComplete((result, thrown) -> {
            if (thrown != null) {
                while (thrown instanceof CompletionException) {
                    thrown = thrown.getCause();
                }
                promise.completeExceptionally((Throwable)thrown);
            } else {
                promise.complete((Result)result);
            }
        });
        return promise;
    }

    private void verifyConnection(FeedClientBuilderImpl builder, ClusterFactory clusterFactory) throws IOException {
        Instant start = Instant.now();
        try (Cluster cluster = clusterFactory.create();){
            HttpRequest request = new HttpRequest("POST", HttpFeedClient.getPath(DocumentId.of((String)"feeder", (String)"handshake", (String)"dummy")), HttpFeedClient.getQuery(OperationParameters.empty(), true), this.requestHeaders, null, Duration.ofSeconds(15L), this.nanoClock);
            CompletableFuture<HttpResponse> future = new CompletableFuture<HttpResponse>();
            cluster.dispatch(request, future);
            HttpResponse response = future.get(20L, TimeUnit.SECONDS);
            if (response.code() != 200) {
                String message;
                if (response.body() != null) {
                    switch (response.contentType()) {
                        case "application/json": {
                            message = HttpFeedClient.parseMessage(response.body());
                            break;
                        }
                        case "text/plain": {
                            message = new String(response.body(), StandardCharsets.UTF_8);
                            break;
                        }
                        default: {
                            message = response.toString();
                            break;
                        }
                    }
                } else {
                    message = response.toString();
                }
                if (response.code() == 400 && "Could not read document, no document?".equals(message)) {
                    if (builder.speedTest) {
                        throw new FeedException("server does not support speed test; upgrade to a newer version");
                    }
                    return;
                }
                throw new FeedException("server responded non-OK to handshake: " + message);
            }
        }
        catch (ExecutionException e) {
            Duration duration = Duration.between(start, Instant.now());
            throw new FeedException("failed handshake with server after " + String.valueOf(duration) + ": " + String.valueOf(e.getCause()), e.getCause());
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new FeedException("interrupted during handshake with server", (Throwable)e);
        }
        catch (TimeoutException e) {
            throw new FeedException("timed out during handshake with server", (Throwable)e);
        }
    }

    private static String parseMessage(byte[] json) {
        try {
            return HttpFeedClient.parse(null, (byte[])json).message;
        }
        catch (Exception e) {
            return new String(json, StandardCharsets.UTF_8);
        }
    }

    static Result.Type toResultType(Outcome outcome) {
        switch (outcome) {
            case success: {
                return Result.Type.success;
            }
            case conditionNotMet: {
                return Result.Type.conditionNotMet;
            }
        }
        throw new IllegalArgumentException("No corresponding result type for '" + String.valueOf((Object)outcome) + "'");
    }

    static MessageAndTrace parse(DocumentId documentId, byte[] json) {
        String message = null;
        String trace = null;
        try (JsonParser parser = jsonParserFactory.createParser(json);){
            String name;
            if (parser.nextToken() != JsonToken.START_OBJECT) {
                throw new ResultParseException(documentId, "Expected '" + String.valueOf(JsonToken.START_OBJECT) + "', but found '" + String.valueOf(parser.currentToken()) + "' in: " + new String(json, StandardCharsets.UTF_8));
            }
            block19: while ((name = parser.nextFieldName()) != null) {
                switch (name) {
                    case "message": {
                        message = parser.nextTextValue();
                        continue block19;
                    }
                    case "trace": {
                        if (parser.nextToken() != JsonToken.START_ARRAY) {
                            throw new ResultParseException(documentId, "Expected 'trace' to be an array, but got '" + String.valueOf(parser.currentToken()) + "' in: " + new String(json, StandardCharsets.UTF_8));
                        }
                        int start = (int)parser.currentTokenLocation().getByteOffset();
                        int depth = 1;
                        while (depth > 0) {
                            switch (parser.nextToken()) {
                                case START_ARRAY: {
                                    ++depth;
                                    break;
                                }
                                case END_ARRAY: {
                                    --depth;
                                }
                            }
                        }
                        int end = (int)parser.currentTokenLocation().getByteOffset() + 1;
                        trace = new String(json, start, end - start, StandardCharsets.UTF_8);
                        continue block19;
                    }
                }
                parser.nextToken();
            }
            if (parser.currentToken() != JsonToken.END_OBJECT) {
                throw new ResultParseException(documentId, "Expected '" + String.valueOf(JsonToken.END_OBJECT) + "', but found '" + String.valueOf(parser.currentToken()) + "' in: " + new String(json, StandardCharsets.UTF_8));
            }
        }
        catch (IOException e) {
            throw new ResultParseException(documentId, (Throwable)e);
        }
        return new MessageAndTrace(message, trace);
    }

    static Result toResult(HttpRequest request, HttpResponse response, DocumentId documentId) {
        Outcome outcome;
        switch (response.code()) {
            case 200: {
                outcome = Outcome.success;
                break;
            }
            case 412: {
                outcome = Outcome.conditionNotMet;
                break;
            }
            case 502: 
            case 504: 
            case 507: {
                outcome = Outcome.vespaFailure;
                break;
            }
            default: {
                outcome = Outcome.transportFailure;
            }
        }
        MessageAndTrace mat = HttpFeedClient.parse(documentId, response.body());
        if (outcome == Outcome.transportFailure) {
            throw new FeedException(documentId, "Status " + response.code() + " executing '" + String.valueOf(request) + "': " + (mat.message == null ? new String(response.body(), StandardCharsets.UTF_8) : mat.message));
        }
        if (outcome == Outcome.vespaFailure) {
            throw new ResultException(documentId, mat.message, mat.trace);
        }
        return new ResultImpl(HttpFeedClient.toResultType(outcome), documentId, mat.message, mat.trace);
    }

    static String getPath(DocumentId documentId) {
        StringJoiner path = new StringJoiner("/", "/", "");
        path.add("document");
        path.add("v1");
        path.add(HttpFeedClient.encode(documentId.namespace()));
        path.add(HttpFeedClient.encode(documentId.documentType()));
        if (documentId.number().isPresent()) {
            path.add("number");
            path.add(Long.toUnsignedString(documentId.number().getAsLong()));
        } else if (documentId.group().isPresent()) {
            path.add("group");
            path.add(HttpFeedClient.encode((String)documentId.group().get()));
        } else {
            path.add("docid");
        }
        path.add(HttpFeedClient.encode(documentId.userSpecific()));
        return path.toString();
    }

    static String encode(String raw) {
        try {
            return URLEncoder.encode(raw, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            throw new IllegalStateException(e);
        }
    }

    static String getQuery(OperationParameters params, boolean speedTest) {
        StringJoiner query = new StringJoiner("&", "?", "").setEmptyValue("");
        if (params.createIfNonExistent()) {
            query.add("create=true");
        }
        params.testAndSetCondition().ifPresent(condition -> query.add("condition=" + HttpFeedClient.encode(condition)));
        params.route().ifPresent(route -> query.add("route=" + HttpFeedClient.encode(route)));
        params.tracelevel().ifPresent(tracelevel -> query.add("tracelevel=" + tracelevel));
        if (speedTest) {
            query.add("dryRun=true");
        }
        return query.toString();
    }

    static interface ClusterFactory {
        public Cluster create() throws IOException;
    }

    private static class MessageAndTrace {
        final String message;
        final String trace;

        MessageAndTrace(String message, String trace) {
            this.message = message;
            this.trace = trace;
        }
    }

    private static enum Outcome {
        success,
        conditionNotMet,
        vespaFailure,
        transportFailure;

    }
}

