/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.search.logging;

import ai.vespa.validation.Validation;
import com.yahoo.search.logging.LoggerEntry;
import com.yahoo.vespa.defaults.Defaults;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

public class Spooler {
    private static final Logger log = Logger.getLogger(Spooler.class.getName());
    private static final Path defaultSpoolPath = Path.of(Defaults.getDefaults().underVespaHome("var/spool/vespa/events"), new String[0]);
    private static final Comparator<File> ordering = new TimestampCompare();
    private static final int defaultMaxEntriesPerFile = 100;
    static final Duration maxDelayAfterFirstWrite = Duration.ofSeconds(5L);
    private static final int maxFilesToRead = 50;
    private Path processingPath;
    private Path readyPath;
    private Path failuresPath;
    private Path successesPath;
    AtomicInteger entryCounter = new AtomicInteger(0);
    AtomicLong fileNameBase = new AtomicLong(0L);
    AtomicInteger fileCounter = new AtomicInteger(0);
    private final Path spoolPath;
    private final int maxEntriesPerFile;
    private final Clock clock;
    private final AtomicReference<Instant> firstWriteTimestamp = new AtomicReference();
    private final boolean keepSuccessFiles;
    private final int maxFailures;
    private final Map<File, Integer> failures = new ConcurrentHashMap<File, Integer>();

    public Spooler(Clock clock) {
        this(clock, false, 1000);
    }

    public Spooler(Clock clock, boolean keepSuccessFiles, int maxFailures) {
        this(defaultSpoolPath, 100, clock, keepSuccessFiles, maxFailures);
    }

    public Spooler(Path spoolPath, int maxEntriesPerFile, Clock clock, boolean keepSuccessFiles, int maxFailures) {
        this.spoolPath = spoolPath;
        this.maxEntriesPerFile = maxEntriesPerFile;
        this.clock = clock;
        this.fileNameBase.set(Spooler.newFileNameBase(clock));
        this.keepSuccessFiles = keepSuccessFiles;
        this.maxFailures = maxFailures;
        this.firstWriteTimestamp.set(Instant.EPOCH);
        this.createDirs(spoolPath);
    }

    void write(LoggerEntry entry) {
        this.writeEntry(entry);
    }

    public void processFiles(Function<LoggerEntry, Boolean> transport) throws IOException {
        List<Path> files = this.listFilesInPath(this.readyPath);
        if (files.size() == 0) {
            log.log(Level.FINEST, () -> "No files in ready path " + this.readyPath.toFile().getAbsolutePath());
            return;
        }
        log.log(Level.FINE, () -> "Files in ready path: " + files.size());
        List<File> fileList = this.getFiles(files);
        if (!fileList.isEmpty()) {
            this.processFiles(fileList, transport);
        }
    }

    List<Path> listFilesInPath(Path path) throws IOException {
        List<Path> files;
        try (Stream<Path> stream = Files.list(path);){
            files = stream.toList();
        }
        catch (NoSuchFileException e) {
            return List.of();
        }
        return files;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void processFiles(List<File> files, Function<LoggerEntry, Boolean> transport) {
        for (File f : files) {
            log.log(Level.FINE, "Processing file " + f);
            boolean success = false;
            try {
                List<String> lines = Files.readAllLines(f.toPath());
                for (String line : lines) {
                    LoggerEntry entry = LoggerEntry.deserialize(line);
                    log.log(Level.FINE, () -> "Read entry " + entry + " from " + f);
                    success = transport.apply(entry);
                    if (success) continue;
                    throw new RuntimeException("Unable to process file " + f + ": unsuccessful call to transport() for " + entry);
                }
                this.failures.remove(f);
            }
            catch (Exception e) {
                this.handleFailure(f);
            }
            finally {
                if (!success) continue;
                if (this.keepSuccessFiles) {
                    this.moveProcessedFile(f, this.successesPath);
                    continue;
                }
                try {
                    Files.delete(f.toPath());
                }
                catch (IOException e) {
                    log.log(Level.WARNING, "Unable to delete file " + f, e);
                }
            }
        }
    }

    private void handleFailure(File file) {
        this.failures.putIfAbsent(file, 0);
        Integer failCount = this.failures.compute(file, (f, count) -> count + 1);
        if (failCount > this.maxFailures) {
            log.log(Level.WARNING, "Unable to process file " + file + " after trying " + this.maxFailures + " times, moving it to " + this.failuresPath);
            this.moveProcessedFile(file, this.failuresPath);
        } else {
            log.log(Level.INFO, "Unable to process file " + file + " after trying " + this.maxFailures + " times, will retry");
        }
    }

    private void moveProcessedFile(File f, Path path) {
        Path file = f.toPath();
        Path target = this.spoolPath.resolve(path).resolve(f.toPath().relativize(file)).resolve(f.getName());
        try {
            Files.move(file, target, new CopyOption[0]);
        }
        catch (IOException e) {
            log.log(Level.SEVERE, "Unable to move processed file " + file + " to " + target, e);
        }
    }

    public Path processingPath() {
        return this.processingPath;
    }

    public Path readyPath() {
        return this.readyPath;
    }

    public Path successesPath() {
        return this.successesPath;
    }

    public Path failuresPath() {
        return this.failuresPath;
    }

    List<File> getFiles(List<Path> files) {
        Validation.requireAtLeast((Comparable)Integer.valueOf(50), (String)"count must be a positive number", (Comparable)Integer.valueOf(1));
        ArrayList<File> fileList = new ArrayList<File>();
        for (Path p : files) {
            File f = p.toFile();
            if (!f.isDirectory()) {
                fileList.add(f);
            }
            if (fileList.size() <= 50) continue;
            break;
        }
        fileList.sort(ordering);
        return fileList;
    }

    private void writeEntry(LoggerEntry entry) {
        String fileName = this.currentFileName();
        Path file = this.spoolPath.resolve(this.processingPath).resolve(fileName);
        try {
            log.log(Level.FINEST, () -> "Writing entry " + this.entryCounter.get() + " (" + entry.serialize() + ") to file " + fileName);
            Files.writeString(file, (CharSequence)(entry.serialize() + "\n"), StandardOpenOption.WRITE, StandardOpenOption.APPEND, StandardOpenOption.CREATE);
            this.firstWriteTimestamp.compareAndExchange(Instant.EPOCH, this.clock.instant());
            this.entryCounter.incrementAndGet();
            this.switchFileIfNeeded(file, fileName);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    void switchFileIfNeeded() throws IOException {
        String fileName = this.currentFileName();
        Path file = this.spoolPath.resolve(this.processingPath).resolve(fileName);
        this.switchFileIfNeeded(file, fileName);
    }

    Map<File, Integer> failures() {
        return this.failures;
    }

    private synchronized void switchFileIfNeeded(Path file, String fileName) throws IOException {
        if (file.toFile().exists() && (this.entryCounter.get() >= this.maxEntriesPerFile || this.firstWriteTimestamp.get().plus(maxDelayAfterFirstWrite).isBefore(this.clock.instant()))) {
            Path target = this.spoolPath.resolve(this.readyPath).resolve(file.relativize(file)).resolve(fileName);
            log.log(Level.INFO, "Finished writing file " + file + " with " + this.entryCounter.get() + " entries, moving it to " + target);
            Files.move(file, target, new CopyOption[0]);
            this.entryCounter.set(1);
            this.fileCounter.incrementAndGet();
            this.fileNameBase.set(Spooler.newFileNameBase(this.clock));
            this.firstWriteTimestamp.set(Instant.EPOCH);
        }
    }

    synchronized String currentFileName() {
        return this.fileNameBase.get() + "-" + this.fileCounter;
    }

    private static long newFileNameBase(Clock clock) {
        return clock.instant().getEpochSecond();
    }

    private void createDirs(Path spoolerPath) {
        this.processingPath = Spooler.createDir(spoolerPath.resolve("processing"));
        this.readyPath = Spooler.createDir(spoolerPath.resolve("ready"));
        this.failuresPath = Spooler.createDir(spoolerPath.resolve("failures"));
        this.successesPath = Spooler.createDir(spoolerPath.resolve("successes"));
    }

    private static Path createDir(Path path) {
        File file = path.toFile();
        if (file.exists() && file.canRead() && file.canWrite()) {
            log.log(Level.INFO, "Directory " + path + " already exists");
        } else if (file.mkdirs()) {
            log.log(Level.FINE, () -> "Created " + path);
        } else {
            log.log(Level.WARNING, "Could not create " + path + ", please check permissions");
        }
        return path;
    }

    private static class TimestampCompare
    implements Comparator<File> {
        private TimestampCompare() {
        }

        @Override
        public int compare(File a, File b) {
            return (int)(a.lastModified() - b.lastModified());
        }
    }
}

