/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.api.core.uuid;

import com.datastax.oss.driver.Assertions;
import com.datastax.oss.driver.api.core.DefaultProtocolVersion;
import com.datastax.oss.driver.api.core.ProtocolVersion;
import com.datastax.oss.driver.api.core.type.codec.TypeCodecs;
import com.datastax.oss.driver.api.core.uuid.Uuids;
import com.tngtech.java.junit.dataprovider.DataProvider;
import com.tngtech.java.junit.dataprovider.DataProviderRunner;
import com.tngtech.java.junit.dataprovider.UseDataProvider;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import java.util.SplittableRandom;
import java.util.UUID;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(value=DataProviderRunner.class)
public class UuidsTest {
    @Test
    public void should_generate_unique_random_uuids_Random() {
        Set<UUID> generated = this.serialGeneration(1000000, Uuids::random);
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_random_uuids_shared_Random2() {
        Random random = new Random();
        Set<UUID> generated = this.serialGeneration(1000000, () -> Uuids.random((Random)random));
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_random_uuids_across_threads_shared_Random() throws Exception {
        Random random = new Random();
        Set<UUID> generated = this.parallelGeneration(10, 10000, () -> () -> Uuids.random((Random)random));
        Assertions.assertThat(generated).hasSize(100000);
    }

    @Test
    public void should_generate_unique_random_uuids_shared_SecureRandom() {
        SecureRandom random = new SecureRandom();
        Set<UUID> generated = this.serialGeneration(1000000, () -> Uuids.random((Random)random));
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_random_uuids_across_threads_shared_SecureRandom() throws Exception {
        SecureRandom random = new SecureRandom();
        Set<UUID> generated = this.parallelGeneration(10, 10000, () -> () -> Uuids.random((Random)random));
        Assertions.assertThat(generated).hasSize(100000);
    }

    @Test
    public void should_generate_unique_random_uuids_ThreadLocalRandom() {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        Set<UUID> generated = this.serialGeneration(1000000, () -> Uuids.random((Random)random));
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_random_uuids_across_threads_ThreadLocalRandom() throws Exception {
        Set<UUID> generated = this.parallelGeneration(10, 10000, () -> {
            ThreadLocalRandom random = ThreadLocalRandom.current();
            return () -> Uuids.random((Random)random);
        });
        Assertions.assertThat(generated).hasSize(100000);
    }

    @Test
    public void should_generate_unique_random_uuids_Netty_ThreadLocalRandom() {
        io.netty.util.internal.ThreadLocalRandom random = io.netty.util.internal.ThreadLocalRandom.current();
        Set<UUID> generated = this.serialGeneration(1000000, () -> Uuids.random((Random)random));
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_random_uuids_across_threads_Netty_ThreadLocalRandom() throws Exception {
        Set<UUID> generated = this.parallelGeneration(10, 10000, () -> {
            io.netty.util.internal.ThreadLocalRandom random = io.netty.util.internal.ThreadLocalRandom.current();
            return () -> Uuids.random((Random)random);
        });
        Assertions.assertThat(generated).hasSize(100000);
    }

    @Test
    public void should_generate_unique_random_uuids_SplittableRandom() {
        SplittableRandom random = new SplittableRandom();
        Set<UUID> generated = this.serialGeneration(1000000, () -> Uuids.random((SplittableRandom)random));
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_random_uuids_across_threads_SplittableRandom() throws Exception {
        Set<UUID> generated = this.parallelGeneration(10, 10000, () -> {
            SplittableRandom random = new SplittableRandom();
            return () -> Uuids.random((SplittableRandom)random);
        });
        Assertions.assertThat(generated).hasSize(100000);
    }

    @Test
    @UseDataProvider(value="byteArrayNames")
    public void should_generate_name_based_uuid_from_namespace_and_byte_array(UUID namespace, byte[] name) throws NoSuchAlgorithmException {
        UUID actual = Uuids.nameBased((UUID)namespace, (byte[])name);
        Assertions.assertThat((Comparable)actual).isNotNull();
        Assertions.assertThat((int)actual.version()).isEqualTo(3);
        UuidsTest.assertUuid(namespace, name, 3, actual);
    }

    @DataProvider
    public static Object[][] byteArrayNames() {
        return new Object[][]{{Uuids.NAMESPACE_DNS, new byte[0]}, {Uuids.NAMESPACE_URL, new byte[]{1, 2, 3, 4}}};
    }

    @Test
    @UseDataProvider(value="byteArrayNamesWithVersions")
    public void should_generate_name_based_uuid_from_namespace_byte_array_and_version(UUID namespace, byte[] name, int version) throws NoSuchAlgorithmException {
        UUID actual = Uuids.nameBased((UUID)namespace, (byte[])name, (int)version);
        Assertions.assertThat((Comparable)actual).isNotNull();
        Assertions.assertThat((int)actual.version()).isEqualTo(version);
        UuidsTest.assertUuid(namespace, name, version, actual);
    }

    @DataProvider
    public static Object[][] byteArrayNamesWithVersions() {
        return new Object[][]{{Uuids.NAMESPACE_DNS, new byte[0], 3}, {Uuids.NAMESPACE_URL, new byte[]{1, 2, 3, 4}, 3}, {Uuids.NAMESPACE_OID, new byte[0], 5}, {Uuids.NAMESPACE_X500, new byte[]{1, 2, 3, 4}, 5}};
    }

    @Test
    @UseDataProvider(value="stringNames")
    public void should_generate_name_based_uuid_from_namespace_and_string(UUID namespace, String name) throws NoSuchAlgorithmException {
        UUID actual = Uuids.nameBased((UUID)namespace, (String)name);
        Assertions.assertThat((Comparable)actual).isNotNull();
        Assertions.assertThat((int)actual.version()).isEqualTo(3);
        UuidsTest.assertUuid(namespace, name, 3, actual);
    }

    @DataProvider
    public static Object[][] stringNames() {
        return new Object[][]{{Uuids.NAMESPACE_DNS, ""}, {Uuids.NAMESPACE_URL, "Hello world!"}, {Uuids.NAMESPACE_OID, "\u4f60\u597d"}};
    }

    @Test
    @UseDataProvider(value="stringNamesWithVersions")
    public void should_generate_name_based_uuid_from_namespace_string_and_version(UUID namespace, String name, int version) throws NoSuchAlgorithmException {
        UUID actual = Uuids.nameBased((UUID)namespace, (String)name, (int)version);
        Assertions.assertThat((Comparable)actual).isNotNull();
        Assertions.assertThat((int)actual.version()).isEqualTo(version);
        UuidsTest.assertUuid(namespace, name, version, actual);
    }

    @DataProvider
    public static Object[][] stringNamesWithVersions() {
        return new Object[][]{{Uuids.NAMESPACE_DNS, "", 3}, {Uuids.NAMESPACE_URL, "Hello world!", 3}, {Uuids.NAMESPACE_OID, "\u4f60\u597d", 3}, {Uuids.NAMESPACE_DNS, "", 5}, {Uuids.NAMESPACE_URL, "Hello world!", 5}, {Uuids.NAMESPACE_OID, "\u4f60\u597d", 5}};
    }

    @Test
    @UseDataProvider(value="concatenatedData")
    public void should_generate_name_based_uuid_from_concatenated_data(byte[] namespaceAndName) throws NoSuchAlgorithmException {
        UUID actual = Uuids.nameBased((byte[])namespaceAndName);
        Assertions.assertThat((Comparable)actual).isNotNull();
        Assertions.assertThat((int)actual.version()).isEqualTo(3);
        UuidsTest.assertUuid(namespaceAndName, 3, actual);
    }

    @DataProvider
    public static Object[][] concatenatedData() {
        return new Object[][]{{UuidsTest.concat(Uuids.NAMESPACE_DNS, new byte[0])}, {UuidsTest.concat(Uuids.NAMESPACE_URL, new byte[]{1, 2, 3, 4})}};
    }

    @Test
    @UseDataProvider(value="concatenatedDataWithVersions")
    public void should_generate_name_based_uuid_from_concatenated_data_and_version(byte[] namespaceAndName, int version) throws NoSuchAlgorithmException {
        UUID actual = Uuids.nameBased((byte[])namespaceAndName, (int)version);
        Assertions.assertThat((Comparable)actual).isNotNull();
        Assertions.assertThat((int)actual.version()).isEqualTo(version);
        UuidsTest.assertUuid(namespaceAndName, version, actual);
    }

    @DataProvider
    public static Object[][] concatenatedDataWithVersions() {
        return new Object[][]{{UuidsTest.concat(Uuids.NAMESPACE_DNS, new byte[0]), 3}, {UuidsTest.concat(Uuids.NAMESPACE_URL, new byte[]{1, 2, 3, 4}), 3}, {UuidsTest.concat(Uuids.NAMESPACE_DNS, new byte[0]), 5}, {UuidsTest.concat(Uuids.NAMESPACE_URL, new byte[]{1, 2, 3, 4}), 5}};
    }

    @Test
    public void should_throw_when_invalid_version() {
        Throwable error = org.assertj.core.api.Assertions.catchThrowable(() -> Uuids.nameBased((UUID)Uuids.NAMESPACE_URL, (String)"irrelevant", (int)1));
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)error).isInstanceOf(IllegalArgumentException.class)).hasMessage("Invalid name-based UUID version, expecting 3 or 5, got: 1");
    }

    @Test
    public void should_throw_when_invalid_data() {
        Throwable error = org.assertj.core.api.Assertions.catchThrowable(() -> Uuids.nameBased((byte[])new byte[]{1}, (int)3));
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)error).isInstanceOf(IllegalArgumentException.class)).hasMessage("namespaceAndName must be at least 16 bytes long");
    }

    @Test
    public void should_generate_timestamp_within_10_ms() {
        Uuids.timeBased();
        long start = System.currentTimeMillis();
        UUID uuid = Uuids.timeBased();
        Assertions.assertThat((int)uuid.version()).isEqualTo(1);
        Assertions.assertThat((int)uuid.variant()).isEqualTo(2);
        long timestamp = Uuids.unixTimestamp((UUID)uuid);
        ((AbstractLongAssert)Assertions.assertThat((long)timestamp).as("Generated timestamp should be within 10 ms", new Object[0])).isBetween(Long.valueOf(start), Long.valueOf(start + 10L));
    }

    @Test
    public void should_generate_unique_time_based_uuids() {
        Set<UUID> generated = this.serialGeneration(1000000, Uuids::timeBased);
        Assertions.assertThat(generated).hasSize(1000000);
    }

    @Test
    public void should_generate_unique_time_based_uuids_across_threads() throws Exception {
        Set<UUID> generated = this.parallelGeneration(10, 10000, () -> Uuids::timeBased);
        Assertions.assertThat(generated).hasSize(100000);
    }

    @Test
    public void should_generate_ever_increasing_timestamps() {
        int count = 1000000;
        long previous = 0L;
        for (int i = 0; i < count; ++i) {
            long current = Uuids.timeBased().timestamp();
            Assertions.assertThat((long)current).isGreaterThan(previous);
            previous = current;
        }
    }

    @Test
    public void should_generate_within_bounds_for_given_timestamp() {
        Random random = new Random(System.currentTimeMillis());
        int timestampsCount = 10;
        int uuidsPerTimestamp = 10;
        for (int i = 0; i < timestampsCount; ++i) {
            long timestamp = random.nextInt();
            for (int j = 0; j < uuidsPerTimestamp; ++j) {
                UUID uuid = new UUID(Uuids.makeMsb((long)Uuids.fromUnixTimestamp((long)timestamp)), random.nextLong());
                UuidsTest.assertBetween(uuid, Uuids.startOf((long)timestamp), Uuids.endOf((long)timestamp));
            }
        }
    }

    private static void assertBetween(UUID uuid, UUID lowerBound, UUID upperBound) {
        ByteBuffer uuidBytes = TypeCodecs.UUID.encode((Object)uuid, (ProtocolVersion)DefaultProtocolVersion.V3);
        ByteBuffer lb = TypeCodecs.UUID.encode((Object)lowerBound, (ProtocolVersion)DefaultProtocolVersion.V3);
        ByteBuffer ub = TypeCodecs.UUID.encode((Object)upperBound, (ProtocolVersion)DefaultProtocolVersion.V3);
        Assertions.assertThat((Comparable)uuidBytes).isNotNull();
        Assertions.assertThat((Comparable)lb).isNotNull();
        Assertions.assertThat((Comparable)ub).isNotNull();
        Assertions.assertThat((int)UuidsTest.compareTimestampBytes(lb, uuidBytes)).isLessThanOrEqualTo(0);
        Assertions.assertThat((int)UuidsTest.compareTimestampBytes(ub, uuidBytes)).isGreaterThanOrEqualTo(0);
    }

    private static int compareTimestampBytes(ByteBuffer o1, ByteBuffer o2) {
        int o1Pos = o1.position();
        int o2Pos = o2.position();
        int d = (o1.get(o1Pos + 6) & 0xF) - (o2.get(o2Pos + 6) & 0xF);
        if (d != 0) {
            return d;
        }
        d = (o1.get(o1Pos + 7) & 0xFF) - (o2.get(o2Pos + 7) & 0xFF);
        if (d != 0) {
            return d;
        }
        d = (o1.get(o1Pos + 4) & 0xFF) - (o2.get(o2Pos + 4) & 0xFF);
        if (d != 0) {
            return d;
        }
        d = (o1.get(o1Pos + 5) & 0xFF) - (o2.get(o2Pos + 5) & 0xFF);
        if (d != 0) {
            return d;
        }
        d = (o1.get(o1Pos) & 0xFF) - (o2.get(o2Pos) & 0xFF);
        if (d != 0) {
            return d;
        }
        d = (o1.get(o1Pos + 1) & 0xFF) - (o2.get(o2Pos + 1) & 0xFF);
        if (d != 0) {
            return d;
        }
        d = (o1.get(o1Pos + 2) & 0xFF) - (o2.get(o2Pos + 2) & 0xFF);
        if (d != 0) {
            return d;
        }
        return (o1.get(o1Pos + 3) & 0xFF) - (o2.get(o2Pos + 3) & 0xFF);
    }

    private static void assertUuid(UUID namespace, String name, int version, UUID actual) throws NoSuchAlgorithmException {
        UuidsTest.assertUuid(namespace, name.getBytes(StandardCharsets.UTF_8), version, actual);
    }

    private static void assertUuid(UUID namespace, byte[] name, int version, UUID actual) throws NoSuchAlgorithmException {
        byte[] data = UuidsTest.digest(namespace, name, version);
        Assertions.assertThat((byte[])UuidsTest.longToBytes(actual.getMostSignificantBits())).isEqualTo((Object)Arrays.copyOfRange(data, 0, 8));
        Assertions.assertThat((byte[])UuidsTest.longToBytes(actual.getLeastSignificantBits())).isEqualTo((Object)Arrays.copyOfRange(data, 8, 16));
    }

    private static void assertUuid(byte[] namespaceAndName, int version, UUID actual) throws NoSuchAlgorithmException {
        byte[] data = UuidsTest.digest(namespaceAndName, version);
        Assertions.assertThat((byte[])UuidsTest.longToBytes(actual.getMostSignificantBits())).isEqualTo((Object)Arrays.copyOfRange(data, 0, 8));
        Assertions.assertThat((byte[])UuidsTest.longToBytes(actual.getLeastSignificantBits())).isEqualTo((Object)Arrays.copyOfRange(data, 8, 16));
    }

    private static byte[] digest(UUID namespace, byte[] name, int version) throws NoSuchAlgorithmException {
        byte[] namespaceAndName = UuidsTest.concat(namespace, name);
        return UuidsTest.digest(namespaceAndName, version);
    }

    private static byte[] digest(byte[] namespaceAndName, int version) throws NoSuchAlgorithmException {
        String algorithm = version == 3 ? "MD5" : "SHA-1";
        MessageDigest result = MessageDigest.getInstance(algorithm);
        byte[] digest = result.digest(namespaceAndName);
        digest[6] = (byte)(digest[6] & 0xF);
        digest[6] = (byte)(digest[6] | (byte)(version << 4));
        digest[8] = (byte)(digest[8] & 0x3F);
        digest[8] = (byte)(digest[8] | 0xFFFFFF80);
        return digest;
    }

    private static byte[] concat(UUID namespace, byte[] name) {
        return ByteBuffer.allocate(16 + name.length).putLong(namespace.getMostSignificantBits()).putLong(namespace.getLeastSignificantBits()).put(name).array();
    }

    private static byte[] longToBytes(long x) {
        return ByteBuffer.allocate(8).putLong(x).array();
    }

    private Set<UUID> serialGeneration(int count, Supplier<UUID> uuidSupplier) {
        HashSet<UUID> generated = new HashSet<UUID>(count);
        for (int i = 0; i < count; ++i) {
            generated.add(uuidSupplier.get());
        }
        return generated;
    }

    public Set<UUID> parallelGeneration(int threadCount, int uuidsPerThread, Supplier<Supplier<UUID>> uuidSupplier) throws InterruptedException {
        int i;
        ConcurrentSkipListSet<UUID> generated = new ConcurrentSkipListSet<UUID>();
        UuidGenerator[] generators = new UuidGenerator[threadCount];
        for (i = 0; i < threadCount; ++i) {
            generators[i] = new UuidGenerator(uuidsPerThread, uuidSupplier, generated);
        }
        for (i = 0; i < threadCount; ++i) {
            generators[i].start();
        }
        for (i = 0; i < threadCount; ++i) {
            generators[i].join();
        }
        return generated;
    }

    private static class UuidGenerator
    extends Thread {
        private final int toGenerate;
        private final Set<UUID> generated;
        private final Supplier<Supplier<UUID>> uuidSupplier;

        UuidGenerator(int toGenerate, Supplier<Supplier<UUID>> uuidSupplier, Set<UUID> generated) {
            this.toGenerate = toGenerate;
            this.generated = generated;
            this.uuidSupplier = uuidSupplier;
        }

        @Override
        public void run() {
            Supplier<UUID> uuidSupplier = this.uuidSupplier.get();
            for (int i = 0; i < this.toGenerate; ++i) {
                this.generated.add(uuidSupplier.get());
            }
        }
    }
}

