/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.pricer.common;

import java.io.Serializable;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutablePreBuild;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.opengamma.strata.basics.date.Tenor;
import com.opengamma.strata.collect.tuple.Pair;
import com.opengamma.strata.market.option.Strike;
import com.opengamma.strata.market.param.ParameterMetadata;

/**
 * Surface node metadata for a generic volatility surface node with a specific time to expiry and strike.
 */
@BeanDefinition(builderScope = "private")
public final class GenericVolatilitySurfaceYearFractionParameterMetadata
    implements ParameterMetadata, ImmutableBean, Serializable {

  /**
   * The year fraction of the surface node.
   * <p>
   * This is the time to expiry that the node on the surface is defined as.
   * There is not necessarily a direct relationship with a date from an underlying instrument.
   */
  @PropertyDefinition
  private final double yearFraction;
  /**
   * The tenor associated with the year fraction.
   */
  @PropertyDefinition(get = "optional")
  private final Tenor yearFractionTenor;
  /**
   * The strike of the surface node.
   * <p>
   * This is the strike that the node on the surface is defined as.
   */
  @PropertyDefinition(validate = "notNull")
  private final Strike strike;
  /**
   * The label that describes the node.
   */
  @PropertyDefinition(validate = "notEmpty", overrideGet = true)
  private final String label;

  //-------------------------------------------------------------------------
  /**
   * Creates node metadata using year fraction and strike.
   * 
   * @param yearFraction  the year fraction
   * @param strike  the strike
   * @return node metadata 
   */
  public static GenericVolatilitySurfaceYearFractionParameterMetadata of(
      double yearFraction,
      Strike strike) {

    String label = Pair.of(yearFraction, strike.getLabel()).toString();
    return new GenericVolatilitySurfaceYearFractionParameterMetadata(yearFraction, null, strike, label);
  }

  /**
   * Creates node metadata using year fraction, associated tenor and strike.
   * 
   * @param yearFraction  the year fraction
   * @param yearFractionTenor  the tenor associated with year fraction
   * @param strike  the strike
   * @return node metadata 
   */
  public static GenericVolatilitySurfaceYearFractionParameterMetadata of(
      double yearFraction,
      Tenor yearFractionTenor,
      Strike strike) {

    String label = Pair.of(yearFraction, strike.getLabel()).toString();
    return new GenericVolatilitySurfaceYearFractionParameterMetadata(yearFraction, yearFractionTenor, strike, label);
  }

  /**
   * Creates node using year fraction, strike and label.
   * 
   * @param yearFraction  the year fraction
   * @param strike  the strike
   * @param label  the label to use
   * @return the metadata
   */
  public static GenericVolatilitySurfaceYearFractionParameterMetadata of(
      double yearFraction,
      Strike strike,
      String label) {

    return new GenericVolatilitySurfaceYearFractionParameterMetadata(yearFraction, null, strike, label);
  }

  /**
   * Creates node using year fraction, associated tenor, strike and label.
   * 
   * @param yearFraction  the year fraction
   * @param yearFractionTenor  the tenor associated with year fraction
   * @param strike  the strike
   * @param label  the label to use
   * @return the metadata
   */
  public static GenericVolatilitySurfaceYearFractionParameterMetadata of(
      double yearFraction,
      Tenor yearFractionTenor,
      Strike strike,
      String label) {

    return new GenericVolatilitySurfaceYearFractionParameterMetadata(yearFraction, yearFractionTenor, strike, label);
  }

  @ImmutablePreBuild
  private static void preBuild(Builder builder) {
    if (builder.label == null && builder.strike != null) {
      builder.label = Pair.of(builder.yearFraction, builder.strike.getLabel()).toString();
    }
  }

  @Override
  public Pair<Double, Strike> getIdentifier() {
    return Pair.of(yearFraction, strike);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code GenericVolatilitySurfaceYearFractionParameterMetadata}.
   * @return the meta-bean, not null
   */
  public static GenericVolatilitySurfaceYearFractionParameterMetadata.Meta meta() {
    return GenericVolatilitySurfaceYearFractionParameterMetadata.Meta.INSTANCE;
  }

  static {
    MetaBean.register(GenericVolatilitySurfaceYearFractionParameterMetadata.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private GenericVolatilitySurfaceYearFractionParameterMetadata(
      double yearFraction,
      Tenor yearFractionTenor,
      Strike strike,
      String label) {
    JodaBeanUtils.notNull(strike, "strike");
    JodaBeanUtils.notEmpty(label, "label");
    this.yearFraction = yearFraction;
    this.yearFractionTenor = yearFractionTenor;
    this.strike = strike;
    this.label = label;
  }

  @Override
  public GenericVolatilitySurfaceYearFractionParameterMetadata.Meta metaBean() {
    return GenericVolatilitySurfaceYearFractionParameterMetadata.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the year fraction of the surface node.
   * <p>
   * This is the time to expiry that the node on the surface is defined as.
   * There is not necessarily a direct relationship with a date from an underlying instrument.
   * @return the value of the property
   */
  public double getYearFraction() {
    return yearFraction;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the tenor associated with the year fraction.
   * @return the optional value of the property, not null
   */
  public Optional<Tenor> getYearFractionTenor() {
    return Optional.ofNullable(yearFractionTenor);
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the strike of the surface node.
   * <p>
   * This is the strike that the node on the surface is defined as.
   * @return the value of the property, not null
   */
  public Strike getStrike() {
    return strike;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the label that describes the node.
   * @return the value of the property, not empty
   */
  @Override
  public String getLabel() {
    return label;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      GenericVolatilitySurfaceYearFractionParameterMetadata other = (GenericVolatilitySurfaceYearFractionParameterMetadata) obj;
      return JodaBeanUtils.equal(yearFraction, other.yearFraction) &&
          JodaBeanUtils.equal(yearFractionTenor, other.yearFractionTenor) &&
          JodaBeanUtils.equal(strike, other.strike) &&
          JodaBeanUtils.equal(label, other.label);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(yearFraction);
    hash = hash * 31 + JodaBeanUtils.hashCode(yearFractionTenor);
    hash = hash * 31 + JodaBeanUtils.hashCode(strike);
    hash = hash * 31 + JodaBeanUtils.hashCode(label);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(160);
    buf.append("GenericVolatilitySurfaceYearFractionParameterMetadata{");
    buf.append("yearFraction").append('=').append(JodaBeanUtils.toString(yearFraction)).append(',').append(' ');
    buf.append("yearFractionTenor").append('=').append(JodaBeanUtils.toString(yearFractionTenor)).append(',').append(' ');
    buf.append("strike").append('=').append(JodaBeanUtils.toString(strike)).append(',').append(' ');
    buf.append("label").append('=').append(JodaBeanUtils.toString(label));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code GenericVolatilitySurfaceYearFractionParameterMetadata}.
   */
  public static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code yearFraction} property.
     */
    private final MetaProperty<Double> yearFraction = DirectMetaProperty.ofImmutable(
        this, "yearFraction", GenericVolatilitySurfaceYearFractionParameterMetadata.class, Double.TYPE);
    /**
     * The meta-property for the {@code yearFractionTenor} property.
     */
    private final MetaProperty<Tenor> yearFractionTenor = DirectMetaProperty.ofImmutable(
        this, "yearFractionTenor", GenericVolatilitySurfaceYearFractionParameterMetadata.class, Tenor.class);
    /**
     * The meta-property for the {@code strike} property.
     */
    private final MetaProperty<Strike> strike = DirectMetaProperty.ofImmutable(
        this, "strike", GenericVolatilitySurfaceYearFractionParameterMetadata.class, Strike.class);
    /**
     * The meta-property for the {@code label} property.
     */
    private final MetaProperty<String> label = DirectMetaProperty.ofImmutable(
        this, "label", GenericVolatilitySurfaceYearFractionParameterMetadata.class, String.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "yearFraction",
        "yearFractionTenor",
        "strike",
        "label");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          return yearFraction;
        case -1032770399:  // yearFractionTenor
          return yearFractionTenor;
        case -891985998:  // strike
          return strike;
        case 102727412:  // label
          return label;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends GenericVolatilitySurfaceYearFractionParameterMetadata> builder() {
      return new GenericVolatilitySurfaceYearFractionParameterMetadata.Builder();
    }

    @Override
    public Class<? extends GenericVolatilitySurfaceYearFractionParameterMetadata> beanType() {
      return GenericVolatilitySurfaceYearFractionParameterMetadata.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code yearFraction} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Double> yearFraction() {
      return yearFraction;
    }

    /**
     * The meta-property for the {@code yearFractionTenor} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Tenor> yearFractionTenor() {
      return yearFractionTenor;
    }

    /**
     * The meta-property for the {@code strike} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Strike> strike() {
      return strike;
    }

    /**
     * The meta-property for the {@code label} property.
     * @return the meta-property, not null
     */
    public MetaProperty<String> label() {
      return label;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          return ((GenericVolatilitySurfaceYearFractionParameterMetadata) bean).getYearFraction();
        case -1032770399:  // yearFractionTenor
          return ((GenericVolatilitySurfaceYearFractionParameterMetadata) bean).yearFractionTenor;
        case -891985998:  // strike
          return ((GenericVolatilitySurfaceYearFractionParameterMetadata) bean).getStrike();
        case 102727412:  // label
          return ((GenericVolatilitySurfaceYearFractionParameterMetadata) bean).getLabel();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code GenericVolatilitySurfaceYearFractionParameterMetadata}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<GenericVolatilitySurfaceYearFractionParameterMetadata> {

    private double yearFraction;
    private Tenor yearFractionTenor;
    private Strike strike;
    private String label;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          return yearFraction;
        case -1032770399:  // yearFractionTenor
          return yearFractionTenor;
        case -891985998:  // strike
          return strike;
        case 102727412:  // label
          return label;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1731780257:  // yearFraction
          this.yearFraction = (Double) newValue;
          break;
        case -1032770399:  // yearFractionTenor
          this.yearFractionTenor = (Tenor) newValue;
          break;
        case -891985998:  // strike
          this.strike = (Strike) newValue;
          break;
        case 102727412:  // label
          this.label = (String) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public GenericVolatilitySurfaceYearFractionParameterMetadata build() {
      preBuild(this);
      return new GenericVolatilitySurfaceYearFractionParameterMetadata(
          yearFraction,
          yearFractionTenor,
          strike,
          label);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(160);
      buf.append("GenericVolatilitySurfaceYearFractionParameterMetadata.Builder{");
      buf.append("yearFraction").append('=').append(JodaBeanUtils.toString(yearFraction)).append(',').append(' ');
      buf.append("yearFractionTenor").append('=').append(JodaBeanUtils.toString(yearFractionTenor)).append(',').append(' ');
      buf.append("strike").append('=').append(JodaBeanUtils.toString(strike)).append(',').append(' ');
      buf.append("label").append('=').append(JodaBeanUtils.toString(label));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
