/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.basics;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

/**
 * A set of reference data which combines the data from two other {@link ReferenceData} instances.
 * <p>
 * When an item of data is requested the underlying sets of reference data are checked in order.
 * If the item is present in the first set of data it is returned. If the item is not found
 * it is looked up in the second set of data.
 */
@BeanDefinition(style = "light", constructorScope = "package")
final class CombinedReferenceData
    implements ReferenceData, ImmutableBean, Serializable {

  /**
   * The first set of reference data.
   */
  @PropertyDefinition(validate = "notNull")
  private final ReferenceData refData1;
  /**
   * The second set of reference data.
   */
  @PropertyDefinition(validate = "notNull")
  private final ReferenceData refData2;

  //-------------------------------------------------------------------------
  @Override
  public boolean containsValue(ReferenceDataId<?> id) {
    return refData1.containsValue(id) || refData2.containsValue(id);
  }

  @Override
  public <T> T queryValueOrNull(ReferenceDataId<T> id) {
    T value1 = refData1.queryValueOrNull(id);
    return value1 != null ? value1 : refData2.queryValueOrNull(id);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code CombinedReferenceData}.
   */
  private static final TypedMetaBean<CombinedReferenceData> META_BEAN =
      LightMetaBean.of(
          CombinedReferenceData.class,
          MethodHandles.lookup(),
          new String[] {
              "refData1",
              "refData2"},
          new Object[0]);

  /**
   * The meta-bean for {@code CombinedReferenceData}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<CombinedReferenceData> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Creates an instance.
   * @param refData1  the value of the property, not null
   * @param refData2  the value of the property, not null
   */
  CombinedReferenceData(
      ReferenceData refData1,
      ReferenceData refData2) {
    JodaBeanUtils.notNull(refData1, "refData1");
    JodaBeanUtils.notNull(refData2, "refData2");
    this.refData1 = refData1;
    this.refData2 = refData2;
  }

  @Override
  public TypedMetaBean<CombinedReferenceData> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the first set of reference data.
   * @return the value of the property, not null
   */
  public ReferenceData getRefData1() {
    return refData1;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the second set of reference data.
   * @return the value of the property, not null
   */
  public ReferenceData getRefData2() {
    return refData2;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      CombinedReferenceData other = (CombinedReferenceData) obj;
      return JodaBeanUtils.equal(refData1, other.refData1) &&
          JodaBeanUtils.equal(refData2, other.refData2);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(refData1);
    hash = hash * 31 + JodaBeanUtils.hashCode(refData2);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("CombinedReferenceData{");
    buf.append("refData1").append('=').append(JodaBeanUtils.toString(refData1)).append(',').append(' ');
    buf.append("refData2").append('=').append(JodaBeanUtils.toString(refData2));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
