/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.basics;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.util.List;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.google.common.collect.ImmutableList;

/**
 * A list of calculation targets.
 * <p>
 * {@link CalculationTarget} is a marker interface that all financial instruments implement,
 * such as trades and positions. This allows them to be the target of calculations in the system.
 */
@BeanDefinition(style = "light")
public final class CalculationTargetList implements ImmutableBean, Serializable {

  /**
   * The targets.
   */
  @PropertyDefinition(validate = "notNull")
  private final ImmutableList<CalculationTarget> targets;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from a list of targets.
   * 
   * @param targets  the list of targets
   * @return the list of targets
   */
  public static CalculationTargetList of(CalculationTarget... targets) {
    return new CalculationTargetList(ImmutableList.copyOf(targets));
  }

  /**
   * Obtains an instance from a list of targets.
   * 
   * @param targets  the list of targets
   * @return the list of targets
   */
  public static CalculationTargetList of(List<? extends CalculationTarget> targets) {
    return new CalculationTargetList(ImmutableList.copyOf(targets));
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code CalculationTargetList}.
   */
  private static final TypedMetaBean<CalculationTargetList> META_BEAN =
      LightMetaBean.of(
          CalculationTargetList.class,
          MethodHandles.lookup(),
          new String[] {
              "targets"},
          ImmutableList.of());

  /**
   * The meta-bean for {@code CalculationTargetList}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<CalculationTargetList> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private CalculationTargetList(
      List<CalculationTarget> targets) {
    JodaBeanUtils.notNull(targets, "targets");
    this.targets = ImmutableList.copyOf(targets);
  }

  @Override
  public TypedMetaBean<CalculationTargetList> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the targets.
   * @return the value of the property, not null
   */
  public ImmutableList<CalculationTarget> getTargets() {
    return targets;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      CalculationTargetList other = (CalculationTargetList) obj;
      return JodaBeanUtils.equal(targets, other.targets);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(targets);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(64);
    buf.append("CalculationTargetList{");
    buf.append("targets").append('=').append(JodaBeanUtils.toString(targets));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
