package com.moengage.locationlibrary;

import android.Manifest;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofenceStatusCodes;
import com.google.android.gms.location.GeofencingEvent;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;
import com.moe.pushlibrary.GeofenceIntentService;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.internal.MoEConstants;
import com.moe.pushlibrary.internal.MoEController;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.GeoManager;
import com.moengage.config.ConfigurationProvider;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import static com.moe.pushlibrary.MoEHelper.TAG;

public class LocationHandlerImpl
    implements GeoManager.LocationHandler, GoogleApiClient.ConnectionCallbacks,
    GoogleApiClient.OnConnectionFailedListener{

  private static boolean DEBUG = MoEHelperUtils.isDebugEnabled();
  Context context;
  private String geoResponse;
  GoogleApiClient mGoogleApiClient;

  private final static int TRACK_LOCATION = 1;
  private final static int SET_GEOFENCE = 2;

  private int mode = -1;

  private final static Object lock = new Object();

  private void triggerApiClientConnection() {
    synchronized (lock) {
      mGoogleApiClient = new GoogleApiClient.Builder(context).addConnectionCallbacks(this)
          .addOnConnectionFailedListener(this)
          .addApi(LocationServices.API)
          .build();
      mGoogleApiClient.connect();
    }
  }

  @Override public void onConnected(Bundle bundle) {
    try {
      if (mode == TRACK_LOCATION) {
        trackLocation();
      } else if (mode == SET_GEOFENCE) {
        setGeofence();
      }
      new Thread(disconnect).start();
    } catch (Exception e) {
      if (DEBUG) Log.e(MoEHelper.TAG, "Location: disconnect: ", e);
    }
  }

  @Override public void onConnectionSuspended(int i) {

  }

  private void setGeofence() {
    ArrayList<Geofence> tFences = parseGeofences();
    if (null != tFences && !tFences.isEmpty()) {
      //get existing fences list
      boolean removeFences = true;
      String geoList = ConfigurationProvider.getInstance().getGeoIDList(context);
      if (DEBUG) Log.d(TAG, "Location: getGeoFences: Existing fences: " + geoList);
      List<String> oldGeoFenceList = null;
      if (TextUtils.isEmpty(geoList)) {
        removeFences = false;
      } else {
        if (geoList.contains(MoEHelperConstants.EVENT_SEPERATOR)) {
          String geoIds[] = geoList.split(MoEHelperConstants.EVENT_SEPERATOR);
          oldGeoFenceList = Arrays.asList(geoIds);
        } else {
          //only one geo fence found
          oldGeoFenceList = new ArrayList<String>();
          oldGeoFenceList.add(geoList);
        }
      }
      boolean noFencesFound = false;
      if (tFences.isEmpty()) {
        if (DEBUG) Log.d(TAG, "Location: getGeoFences: No new geo fences found");
        noFencesFound = true;
      }
      //if nothing needs to be done just go home
      if (!removeFences && noFencesFound) {
        if (DEBUG) {
          Log.d(TAG, "Location: getGeoFences: Nothing needs to be done no existing or new "
                  + "geo fences found");
        }
        return;
      }

      if (removeFences) {
        if (DEBUG) Log.d(TAG, "Location: getGeoFences: Removing all existing geo fences");
        if (mGoogleApiClient.isConnected()) {
          try {
            LocationServices.GeofencingApi.removeGeofences(mGoogleApiClient, oldGeoFenceList);
          } catch (SecurityException e) {
            Log.e(TAG, "Location: getGeoFences: setGeoFencesAroundLatLng", e);
          }
        } else if (DEBUG) {
          Log.d(TAG, "Location: getGeoFences: Google api client not connected, could not"
                  + " remove geo fence");
        }
      }

      if (noFencesFound) {
        return;
      }
      //store the geo ids on the client side
      StringBuilder geoIDBuilder = new StringBuilder();
      int size = tFences.size();
      for (int index = 0; index < size; index++) {
        Geofence fence = tFences.get(index);
        geoIDBuilder.append(fence.getRequestId());
        if (index < size - 2) {
          geoIDBuilder.append(MoEHelperConstants.EVENT_SEPERATOR);
        }
      }
      //save the geo fence ids so that they can be removed later
      ConfigurationProvider.getInstance().saveGeoIDList(context, geoIDBuilder.toString());

      Intent intent = new Intent(context, GeofenceIntentService.class);
      PendingIntent geoPending =
          PendingIntent.getService(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);

      GeofencingRequest.Builder builder = new GeofencingRequest.Builder();
      builder.addGeofences(tFences);

      try {
        for (int index = 0; index<tFences.size(); index+=5){

          LocationServices.GeofencingApi.addGeofences(mGoogleApiClient, builder.build(),
              geoPending);
        }
        if (DEBUG) Log.d(TAG, "Location: getGeoFences: Successfully added geo fences");
      } catch (SecurityException e) {
        Log.e(TAG, "Location: getGeoFences: setGeoFencesAroundLatLng", e);
      }
    } else if (DEBUG) {
      Log.d(TAG, "Location: setFences: no permission so skipped");
    }
  }

  private void trackLocation() {
    Log.d(TAG, "Location:onConnected Google APi Client connected");
    Location lastLocation = LocationServices.FusedLocationApi.getLastLocation(mGoogleApiClient);
    if (null != lastLocation && !ConfigurationProvider.getInstance()
        .haveSentLocationForToday(context)) {
      GeoLocation location =
          new GeoLocation(lastLocation.getLatitude(), lastLocation.getLongitude());
      GeoLocation storedLocation =
          ConfigurationProvider.getInstance().getLastKnownUserLocation(context);
      if (!location.equals(storedLocation)) {
        ConfigurationProvider.getInstance().storeLastKnownLocation(context, location);
        MoEHelper.getInstance(context)
            .setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LOCATION, location);
      }
    }
  }

  Runnable disconnect = new Runnable() {
    @Override public void run() {
      try {
        if (mGoogleApiClient != null && mGoogleApiClient.isConnected()) {
          mGoogleApiClient.disconnect();
        }
      } catch (Exception e) {
        if (DEBUG) Log.e(MoEHelper.TAG, "Location: disconnect: ", e);
      }
    }
  };

  @Override public void setGeoFences(Context context, String response) {
    this.context = context;
    if (MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)) {
      mode = SET_GEOFENCE;
      geoResponse = response;
      triggerApiClientConnection();
    }
  }

  @Override public void trackLocation(Context context) {
    this.context = context;
    if (MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)
        || MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION)) {
      mode = TRACK_LOCATION;
      triggerApiClientConnection();
    }
  }

  @Override public void onGeofenceHit(Context context, Intent intent) {
    this.context = context;
    GeofencingEvent geofencingEvent = GeofencingEvent.fromIntent(intent);
    if (null == geofencingEvent) {
      Log.e(TAG, "Location : Null geo fence transition event");
      return;
    }
    if (geofencingEvent.hasError()) {
      Log.e(TAG, "Location : Received geo fence transition intent with error"
              + GeofenceStatusCodes.getStatusCodeString(geofencingEvent.getErrorCode()));
      return;
    }
    if (DEBUG) Log.d(TAG, "Location : Received geo fence transition intent");
    int geofenceTransition = geofencingEvent.getGeofenceTransition();

    if (geofenceTransition == Geofence.GEOFENCE_TRANSITION_ENTER
        || geofenceTransition == Geofence.GEOFENCE_TRANSITION_EXIT
        || geofenceTransition == Geofence.GEOFENCE_TRANSITION_DWELL) {
      List<Geofence> triggeringGeofences = geofencingEvent.getTriggeringGeofences();
      if (null != triggeringGeofences && !triggeringGeofences.isEmpty()) {
        if (DEBUG) {
          Log.d(TAG,
              "Location : Received geo fences count: " + triggeringGeofences.size());
        }
        for (Geofence fence : triggeringGeofences) {
          if (DEBUG) {
            Log.d(TAG, "Location : Registering geo fencing hit for GeoId: "
                + fence.getRequestId());
          }
          if (!MoEController.registerGeofenceHit(fence.getRequestId(), geofenceTransition,
              context)) {
            Log.e(TAG, "Location : Failed response");
            break;
          }
        }
      }
    } else {
      Log.e(TAG,
          "Location : Transition type was not in our interest: " + geofenceTransition);
    }
    Log.d(TAG, "onGeofenceHit: ");
  }

  private ArrayList<Geofence> parseGeofences() {

    JSONObject jsonResponse;
    try {
      jsonResponse = new JSONObject(geoResponse);
      JSONArray jsonPlaces = jsonResponse.getJSONArray(MoEConstants.RESPONSE_ATTR_FENCES_INFO);
      int placesLength = jsonPlaces.length();
      ArrayList<Geofence> newFences = new ArrayList<Geofence>(placesLength);

      for (int i = 0; i < placesLength; i++) {
        try {
          JSONObject jsonFence = jsonPlaces.getJSONObject(i);
          String reqTransition = jsonFence.getString(MoEConstants.RESPONSE_ATTR_TRANSITION_TYPE);
          if (TextUtils.isEmpty(reqTransition)) {
            continue;
          }
          int transitionType = Geofence.GEOFENCE_TRANSITION_ENTER;
          if (reqTransition.equals(MoEConstants.TRANSITION_TYPE_EXIT)) {
            transitionType = Geofence.GEOFENCE_TRANSITION_EXIT;
          } else if (reqTransition.equals(MoEConstants.TRANSITION_TYPE_DWELL)) {
            transitionType = Geofence.GEOFENCE_TRANSITION_DWELL;
          }
          Geofence.Builder builder = new Geofence.Builder().setRequestId(
              jsonFence.getString(MoEConstants.RESPONSE_ATTR_GEOID))
              .setCircularRegion(jsonFence.getDouble(MoEConstants.PARAM_LAT),
                  jsonFence.getDouble(MoEConstants.PARAM_LNG),
                  (float) jsonFence.getDouble(MoEConstants.RESPONSE_ATTR_DISTANCE))
              .setExpirationDuration(Geofence.NEVER_EXPIRE)
              .setTransitionTypes(transitionType);

          if (jsonFence.has(MoEConstants.RESPONSE_ATTR_FENCE_LDELAY)) {
            builder.setLoiteringDelay(
                Integer.parseInt(jsonFence.getString(MoEConstants.RESPONSE_ATTR_FENCE_LDELAY)));
          }
          if (jsonFence.has(MoEConstants.RESPONSE_ATTR_FENCE_EXPIRY)) {
            builder.setExpirationDuration(
                Long.parseLong(jsonFence.getString(MoEConstants.RESPONSE_ATTR_FENCE_EXPIRY)));
          }
          if (jsonFence.has(MoEConstants.RESPONSE_ATTR_FENCE_RESPONSOVENESS)) {
            builder.setNotificationResponsiveness(
                Integer.parseInt(MoEConstants.RESPONSE_ATTR_FENCE_RESPONSOVENESS));
          }
          newFences.add(builder.build());
        } catch (Exception e) {
          if (DEBUG) Log.e(TAG, "Location: parseFencesInfo - INNER", e);
        }
      }
      return newFences;
    } catch (JSONException e) {
      if (DEBUG) Log.e(TAG, "Location: parseFencesInfo", e);
    }
    return null;
  }

  @Override public void onConnectionFailed(ConnectionResult connectionResult) {

  }

  static boolean checkPlayServices(Context context) {
    return null != context
        && GooglePlayServicesUtil.isGooglePlayServicesAvailable(context)
        != ConnectionResult.SERVICE_MISSING;
  }
}