// This javascript builds a tree chart and pushes it into an array named raveTreeFunctionArray which must be predefined
// The uppercase Strings that start with TREE_ are placeholders for the real values which must be replaced after loading this javascript.
raveTreeFunctionArray.push(function() {
	var treeIndent = TREE_INDENT;
	try {
		var canvas = document.createElement("canvas");
		var context = canvas.getContext("2d");
		context.font = "14px Arial, Helvetica, sans-serif";
		var metrics = context.measureText("TREE_ROOT_NAME");
		treeIndent = Math.ceil(metrics.width) + 10;
	} catch (e) {
		console.warn("Could not calculate rave tree root indent; using default");
	}
	
	var m = [ 5, 200, 5, treeIndent ], w = TREE_WIDTH - m[1] - m[3], h = TREE_HEIGHT - m[0] - m[2], i = 0;

	var tree = rave.layout.tree().size([ h, w ]);

	var diagonal = rave.svg.diagonal().projection(function(d) {
		return [ d.y, d.x ];
	});

	//Draw a white box as the background for the rave tree for consistency in windows high contrast mode
	//The preferred solution is @media queries in css using forced-colors-adjust:auto or prefers-color-scheme
	//but firefox currently does not support this parameter - https://bugzilla.mozilla.org/show_bug.cgi?id=1591210
	var vis = rave.select("#TREE_ID").append("svg:svg").attr("font-family","Arial, Helvetica, sans-serif").attr("width", w + m[1] + m[3]).attr("height", h + m[0] + m[2]).append("svg:rect").attr("width", "100%").attr("height", "100%").attr("fill", "white");
	vis = rave.select("#TREE_ID svg").append("svg:g").attr("transform", "translate(" + m[3] + "," + m[0] + ")");
	
	rave.select(self.frameElement).style("height", h + m[0] + m[2] + "px");
	rave.select(self.frameElement).style("width", w + "px");

	var root =
{
TREE_CONTENT
};

	root.x0 = h / 2;
	root.y0 = 0;

	update(root);

	//Functions used by Rave Tree
	function toggleAll(d) {
		if (d.children) {
			d.children.forEach(toggleAll);
			toggle(d);
		}
	}

	function update(source) {
		var duration = rave.event && rave.event.altKey ? 5000 : 500;

		// Compute the new tree layout.
		var nodes = tree.nodes(root).reverse();

		// Normalize for fixed-depth.
		// The d.y = d.depth * 300 affects the spacing between nodes horizontally
		nodes.forEach(function(d) {
			d.y = d.depth * 300;
		});

		// Update the nodes…
		var node = vis.selectAll("g.node").data(nodes, function(d) {
			return d.id || (d.id = ++i);
		});

		// Enter any new nodes at the parent's previous position.
		var nodeEnter = node.enter().append("svg:g").attr("class", "node").attr("transform", function(d) {
			return "translate(" + source.y0 + "," + source.x0 + ")";
		}).on("click", function(d) {
			toggle(d);
			update(d);
		});

		nodeEnter.append("svg:circle").attr("r", 1e-6).style("fill", function(d) {
			return d._children ? "lightsteelblue" : "#fff";
		});

		var text = nodeEnter.append("svg:text");
		text.attr("x", function(d) {
			return d.children || d._children ? -10 : 10;
		}).attr("dy", ".35em").attr("text-anchor", function(d) {
			return d.children || d._children ? "end" : "start";
		}).text(function(d) {
			return d.name + " ";
		});
		
		var colors = ["#FF0000", "#478DE4", "#FFA500", "#9855D4", "#FF00FF", "#008000"];
		colors.forEach(function(color, i) {
			text.append("svg:tspan").style("fill", color).text(function(d) {
				var issueNumber = (i+1).toString();
				if(d.issues.indexOf(issueNumber) != -1) {
					return "("+issueNumber+") ";
				} else {
					return "";
				}
			});
		});

		// Transition nodes to their new position.
		var nodeUpdate = node.transition().duration(duration).attr("transform", function(d) {
			return "translate(" + d.y + "," + d.x + ")";
		});

		nodeUpdate.select("circle").attr("r", 4.5).style("fill", function(d) {
			return d._children ? "lightsteelblue" : "#fff";
		});

		nodeUpdate.select("text").style("fill-opacity", 1);

		// Transition exiting nodes to the parent's new position.
		var nodeExit = node.exit().transition().duration(duration).attr("transform", function(d) {
			return "translate(" + source.y + "," + source.x + ")";
		}).remove();

		nodeExit.select("circle").attr("r", 1e-6);

		nodeExit.select("text").style("fill-opacity", 1e-6);

		// Update the links…
		var link = vis.selectAll("path.link").data(tree.links(nodes), function(d) {
			return d.target.id;
		});

		// Enter any new links at the parent's previous position.
		link.enter().insert("svg:path", "g").attr("class", "link").attr("d", function(d) {
			var o = {
				x : source.x0,
				y : source.y0
			};
			return diagonal({
				source : o,
				target : o
			});
		}).transition().duration(duration).attr("d", diagonal);

		// Transition links to their new position.
		link.transition().duration(duration).attr("d", diagonal);

		// Transition exiting nodes to the parent's new position.
		link.exit().transition().duration(duration).attr("d", function(d) {
			var o = {
				x : source.x,
				y : source.y
			};
			return diagonal({
				source : o,
				target : o
			});
		}).remove();

		// Stash the old positions for transition.
		nodes.forEach(function(d) {
			d.x0 = d.x;
			d.y0 = d.y;
		});
	}

	// Toggle children.
	function toggle(d) {
		if (d.children) {
			d._children = d.children;
			d.children = null;
		} else {
			d.children = d._children;
			d._children = null;
		}
	}
});