/*
 * Decompiled with CFR 0.152.
 */
package com.arcadedb.server.ha;

import com.arcadedb.database.Binary;
import com.arcadedb.engine.WALFile;
import com.arcadedb.log.LogManager;
import com.arcadedb.server.ha.ReplicationLogException;
import com.arcadedb.server.ha.ReplicationMessage;
import com.arcadedb.utility.FileUtils;
import com.arcadedb.utility.LockContext;
import com.arcadedb.utility.Pair;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.logging.Level;

public class ReplicationLogFile
extends LockContext {
    private final String filePath;
    private FileChannel lastChunkChannel;
    private FileChannel searchChannel = null;
    private long searchChannelChunkId = -1L;
    private static final int BUFFER_HEADER_SIZE = 12;
    private final ByteBuffer bufferHeader = ByteBuffer.allocate(12);
    private static final int BUFFER_FOOTER_SIZE = 12;
    private final ByteBuffer bufferFooter = ByteBuffer.allocate(12);
    private static final long MAGIC_NUMBER = 93719829258702L;
    private long lastMessageNumber = -1L;
    private static final long CHUNK_SIZE = 0x4000000L;
    private long chunkNumber = 0L;
    private WALFile.FlushType flushPolicy = WALFile.FlushType.NO;
    private ReplicationLogArchiveCallback archiveChunkCallback = null;
    private long totalArchivedChunks = 0L;
    private long maxArchivedChunks = 200L;
    private static final Comparator<File> LOG_COMPARATOR = (file1, file2) -> {
        int seq1 = Integer.parseInt(file1.getName().substring(file1.getName().lastIndexOf(".") + 1));
        int seq2 = Integer.parseInt(file2.getName().substring(file2.getName().lastIndexOf(".") + 1));
        return seq1 - seq2;
    };

    public ReplicationLogFile(String filePath) throws FileNotFoundException {
        this.filePath = filePath;
        this.openLastFile();
    }

    public void close() {
        this.executeInLock(() -> {
            this.lastChunkChannel.force(true);
            this.lastChunkChannel.close();
            this.lastChunkChannel = null;
            if (this.searchChannel != null) {
                this.searchChannel.close();
                this.searchChannel = null;
            }
            return null;
        });
    }

    public void drop() {
        this.close();
        new File(this.filePath).delete();
    }

    public long getLastMessageNumber() {
        return this.lastMessageNumber;
    }

    public boolean appendMessage(ReplicationMessage message) {
        return (Boolean)this.executeInLock(() -> {
            try {
                if (!this.checkMessageOrder(message)) {
                    return false;
                }
                if (this.lastChunkChannel == null) {
                    return false;
                }
                this.lastMessageNumber = message.messageNumber;
                byte[] content = message.payload.toByteArray();
                int entrySize = 12 + content.length + 12;
                if ((long)entrySize > 0x4000000L) {
                    throw new IllegalArgumentException("Cannot store in replication file messages bigger than " + FileUtils.getSizeAsString((long)0x4000000L));
                }
                if (this.lastChunkChannel.size() + (long)entrySize > 0x4000000L) {
                    this.archiveChunk();
                }
                this.bufferHeader.clear();
                this.bufferHeader.putLong(message.messageNumber);
                this.bufferHeader.putInt(content.length);
                this.bufferHeader.rewind();
                this.lastChunkChannel.write(this.bufferHeader, this.lastChunkChannel.size());
                this.lastChunkChannel.write(ByteBuffer.wrap(content), this.lastChunkChannel.size());
                this.bufferFooter.clear();
                this.bufferFooter.putInt(entrySize);
                this.bufferFooter.putLong(93719829258702L);
                this.bufferFooter.rewind();
                this.lastChunkChannel.write(this.bufferFooter, this.lastChunkChannel.size());
                switch (this.flushPolicy) {
                    case YES_FULL: {
                        this.lastChunkChannel.force(true);
                        break;
                    }
                    case YES_NOMETADATA: {
                        this.lastChunkChannel.force(false);
                    }
                }
                return true;
            }
            catch (Exception e) {
                throw new ReplicationLogException("Error on writing message " + message.messageNumber + " to the replication log", e);
            }
        });
    }

    public long findMessagePosition(long messageNumberToFind) {
        return (Long)this.executeInLock(() -> {
            int contentLength;
            long chunkId;
            for (chunkId = this.chunkNumber; chunkId > -1L; --chunkId) {
                if (!this.openChunk(chunkId)) {
                    return -1L;
                }
                this.bufferHeader.clear();
                this.searchChannel.read(this.bufferHeader, 0L);
                this.bufferHeader.rewind();
                long chunkBeginMessageNumber = this.bufferHeader.getLong();
                if (messageNumberToFind == chunkBeginMessageNumber) {
                    return chunkId * 0x4000000L;
                }
                if (messageNumberToFind > chunkBeginMessageNumber) break;
            }
            long fileSize = this.searchChannel.size();
            for (long pos = 0L; pos < fileSize; pos += (long)(12 + contentLength + 12)) {
                this.bufferHeader.clear();
                this.searchChannel.read(this.bufferHeader, pos);
                this.bufferHeader.rewind();
                long messageNumber = this.bufferHeader.getLong();
                if (messageNumber == messageNumberToFind) {
                    return pos + chunkId * 0x4000000L;
                }
                if (messageNumber > messageNumberToFind) {
                    return -1L;
                }
                contentLength = this.bufferHeader.getInt();
            }
            return -1L;
        });
    }

    public void setLastMessageNumber(long lastMessageNumber) {
        this.lastMessageNumber = lastMessageNumber;
    }

    public Pair<ReplicationMessage, Long> getMessage(long positionInFile) {
        return (Pair)this.executeInLock(() -> {
            if (positionInFile < 0L) {
                throw new ReplicationLogException("Invalid position (" + positionInFile + ") in replication log file of size " + this.getSize());
            }
            if (positionInFile > this.searchChannel.size() - 12L - 12L + this.chunkNumber * 0x4000000L) {
                throw new ReplicationLogException("Invalid position (" + positionInFile + ") in replication log file of size " + this.getSize());
            }
            int chunkId = (int)(positionInFile / 0x4000000L);
            if (!this.openChunk(chunkId)) {
                throw new ReplicationLogException("Cannot find replication log file with chunk id " + chunkId);
            }
            long posInChunk = positionInFile % 0x4000000L;
            this.bufferHeader.clear();
            this.searchChannel.read(this.bufferHeader, posInChunk);
            this.bufferHeader.rewind();
            long messageNumber = this.bufferHeader.getLong();
            int contentLength = this.bufferHeader.getInt();
            ByteBuffer bufferPayload = ByteBuffer.allocate(contentLength);
            this.searchChannel.read(bufferPayload, posInChunk + 12L);
            this.bufferFooter.clear();
            this.searchChannel.read(this.bufferFooter, posInChunk + 12L + (long)contentLength);
            this.bufferFooter.rewind();
            this.bufferFooter.getInt();
            long magicNumber = this.bufferFooter.getLong();
            if (magicNumber != 93719829258702L) {
                throw new ReplicationLogException("Corrupted replication log file at position " + positionInFile);
            }
            long nextPos = posInChunk + 12L + (long)contentLength + 12L >= this.searchChannel.size() ? ((long)chunkId + 1L) * 0x4000000L : positionInFile + 12L + (long)contentLength + 12L;
            return new Pair((Object)new ReplicationMessage(messageNumber, new Binary(bufferPayload.rewind())), (Object)nextPos);
        });
    }

    public boolean checkMessageOrder(ReplicationMessage message) {
        if (this.lastMessageNumber > -1L) {
            if (message.messageNumber < this.lastMessageNumber) {
                LogManager.instance().log((Object)this, Level.WARNING, "Wrong sequence in message numbers. Last was %d and now receiving %d. Skip saving this entry (threadId=%d)", (Object)this.lastMessageNumber, (Object)message.messageNumber, (Object)Thread.currentThread().threadId());
                return false;
            }
            if (message.messageNumber != this.lastMessageNumber + 1L) {
                LogManager.instance().log((Object)this, Level.WARNING, "Found a jump (%d) in message numbers. Last was %d and now receiving %d. Skip saving this entry (threadId=%d)", (Object)(message.messageNumber - this.lastMessageNumber), (Object)this.lastMessageNumber, (Object)message.messageNumber, (Object)Thread.currentThread().threadId());
                return false;
            }
        }
        return true;
    }

    public ReplicationMessage getLastMessage() {
        return (ReplicationMessage)this.executeInLock(() -> {
            if (this.lastChunkChannel == null) {
                return null;
            }
            long pos = this.lastChunkChannel.size();
            if (pos == 0L) {
                return null;
            }
            if (pos < 24L) {
                throw new ReplicationLogException("Invalid position (" + pos + ") in replication log file of size " + this.lastChunkChannel.size());
            }
            this.bufferFooter.clear();
            this.lastChunkChannel.read(this.bufferFooter, pos - 12L);
            this.bufferFooter.rewind();
            int entrySize = this.bufferFooter.getInt();
            long magicNumber = this.bufferFooter.getLong();
            if (magicNumber != 93719829258702L) {
                throw new ReplicationLogException("Corrupted replication log file");
            }
            this.bufferHeader.clear();
            this.lastChunkChannel.read(this.bufferHeader, pos - (long)entrySize);
            this.bufferHeader.rewind();
            long messageNumber = this.bufferHeader.getLong();
            int contentLength = this.bufferHeader.getInt();
            ByteBuffer bufferPayload = ByteBuffer.allocate(contentLength);
            this.lastChunkChannel.read(bufferPayload, pos - (long)entrySize + 12L);
            return new ReplicationMessage(messageNumber, new Binary(bufferPayload.rewind()));
        });
    }

    public long getSize() {
        return (Long)this.executeInLock(new Callable<Object>(){

            @Override
            public Object call() {
                try {
                    return ReplicationLogFile.this.lastChunkChannel.size() + ReplicationLogFile.this.chunkNumber * 0x4000000L;
                }
                catch (IOException e) {
                    LogManager.instance().log((Object)this, Level.SEVERE, "Error on computing file size for last chunk (%d) in replication log '%s', reopening file...", (Throwable)e, (Object)ReplicationLogFile.this.chunkNumber, (Object)ReplicationLogFile.this.filePath);
                    try {
                        ReplicationLogFile.this.lastChunkChannel.close();
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                    try {
                        ReplicationLogFile.this.openLastFile();
                        return ReplicationLogFile.this.lastChunkChannel.size() + ReplicationLogFile.this.chunkNumber * 0x4000000L;
                    }
                    catch (IOException ex) {
                        LogManager.instance().log((Object)this, Level.SEVERE, "Error on computing file size for last chunk (%d) in replication log '%s'", (Throwable)e, (Object)ReplicationLogFile.this.chunkNumber, (Object)ReplicationLogFile.this.filePath);
                        throw new ReplicationLogException("Error on computing file size for last chunk in replication log", e);
                    }
                }
            }
        });
    }

    public WALFile.FlushType getFlushPolicy() {
        return this.flushPolicy;
    }

    public void setFlushPolicy(WALFile.FlushType flushPolicy) {
        this.flushPolicy = flushPolicy;
    }

    public void setArchiveChunkCallback(ReplicationLogArchiveCallback archiveChunkCallback) {
        this.archiveChunkCallback = archiveChunkCallback;
    }

    public int getMaxArchivedChunks() {
        return (int)this.maxArchivedChunks;
    }

    public void setMaxArchivedChunks(int maxArchivedChunks) {
        this.maxArchivedChunks = maxArchivedChunks;
    }

    public String toString() {
        return this.filePath;
    }

    protected RuntimeException manageExceptionInLock(Throwable e) {
        if (e instanceof ReplicationLogException) {
            ReplicationLogException exception = (ReplicationLogException)e;
            throw exception;
        }
        return new ReplicationLogException("Error in replication log", e);
    }

    private void openLastFileChunk(File logFile) throws FileNotFoundException {
        String prefix = logFile.getName() + ".";
        List<File> fileChunks = Arrays.asList(logFile.getParentFile().listFiles(f -> f.getName().startsWith(prefix)));
        fileChunks.sort(LOG_COMPARATOR);
        this.totalArchivedChunks = fileChunks.isEmpty() ? 0L : (long)(fileChunks.size() - 1);
        File lastFile = fileChunks.isEmpty() ? new File(logFile.getAbsolutePath() + ".0") : fileChunks.getLast();
        this.lastChunkChannel = new RandomAccessFile(lastFile, "rw").getChannel();
        this.chunkNumber = Long.parseLong(lastFile.getName().substring(lastFile.getName().lastIndexOf(".") + 1));
    }

    private void archiveChunk() throws IOException {
        this.lastChunkChannel.force(true);
        this.lastChunkChannel.close();
        this.lastChunkChannel = null;
        if (this.archiveChunkCallback != null) {
            File archivedFile = new File(this.filePath + "." + this.chunkNumber);
            try {
                this.archiveChunkCallback.archiveChunk(archivedFile, (int)this.chunkNumber);
            }
            catch (Exception e) {
                LogManager.instance().log((Object)this, Level.WARNING, "Error in replication log archive callback invoked on file '%s'", (Throwable)e, (Object)archivedFile);
            }
        }
        if (this.maxArchivedChunks > 0L && ++this.totalArchivedChunks > this.maxArchivedChunks) {
            File file2Remove = new File(this.filePath + "." + (this.chunkNumber - this.maxArchivedChunks));
            if (file2Remove.exists()) {
                file2Remove.delete();
            }
            --this.totalArchivedChunks;
        }
        File f = new File(this.filePath + "." + (this.chunkNumber + 1L));
        this.lastChunkChannel = new RandomAccessFile(f, "rw").getChannel();
        ++this.chunkNumber;
    }

    private boolean openChunk(long chunkId) throws IOException {
        if (chunkId != this.searchChannelChunkId) {
            File chunkFile;
            if (this.searchChannel != null) {
                this.searchChannel.close();
            }
            if (!(chunkFile = new File(this.filePath + "." + chunkId)).exists()) {
                this.searchChannel = null;
                this.searchChannelChunkId = -1L;
                LogManager.instance().log((Object)this, Level.WARNING, "Replication log chunk file %d was not found", null, (Object)chunkId);
                return false;
            }
            this.searchChannel = new RandomAccessFile(chunkFile, "rw").getChannel();
            this.searchChannelChunkId = chunkId;
        }
        return true;
    }

    private void openLastFile() throws FileNotFoundException {
        File f = new File(this.filePath);
        if (!f.exists()) {
            f.getParentFile().mkdirs();
        }
        this.openLastFileChunk(f);
        ReplicationMessage lastMessage = this.getLastMessage();
        if (lastMessage != null) {
            this.lastMessageNumber = lastMessage.messageNumber;
        }
    }

    public static interface ReplicationLogArchiveCallback {
        public void archiveChunk(File var1, int var2);
    }

    public static class Entry {
        public final long messageNumber;
        public final Binary payload;
        public final int length;

        public Entry(long messageNumber, Binary payload, int length) {
            this.messageNumber = messageNumber;
            this.payload = payload;
            this.length = length;
        }
    }
}

