/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.calcite.avatica.remote;

import com.aliyun.calcite.avatica.org.apache.http.conn.ConnectTimeoutException;
import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.SocketException;
import java.net.UnknownHostException;

public class FailoverRetryPolicy {
    private static final Logger LOG = LoggerFactory.getLogger(FailoverRetryPolicy.class);
    private int maxRetries;
    private long intervalMillis;
    private long maxDelayBase;
    private int retries = 0;

    public FailoverRetryPolicy(int maxRetries, long intervalMillis, long maxDelayBase) {
        this.maxRetries = maxRetries;
        this.intervalMillis = intervalMillis;
        this.maxDelayBase = maxDelayBase;
    }

    private long getRetrySleepTime(int times) {
        return Math.min(intervalMillis * (1L << times), maxDelayBase);
    }

    public int getRetries() {
        return retries;
    }

    public boolean awaitUntilNextRetry(Throwable e) {
        if (retries  > maxRetries) {
            LOG.warn("retries (" + retries + ") exceeded maximum allowed (" + maxRetries + ")");
            throw new IllegalStateException("retries ("
                    + retries + ") exceeded maximum allowed (" + maxRetries + ")");
        }

        if (e instanceof ConnectException ||
                e instanceof NoRouteToHostException ||
                e instanceof UnknownHostException ||
                e instanceof ConnectTimeoutException ||
                e instanceof SocketException) {
            long sleepTime = getRetrySleepTime(retries);
            Preconditions.checkArgument(sleepTime > 0,
                    "retry interval times must be greater than 0");
            try {
                Thread.sleep(sleepTime);
            } catch (InterruptedException e1) {
                LOG.warn("retries (" + retries + ") was Interrupted", e1);
            }
            retries++;
            return true;
        }
        return false;
    }
}