/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.executiongraph.failover;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import org.apache.flink.runtime.executiongraph.failover.ExponentialDelayRestartBackoffTimeStrategy;
import org.apache.flink.runtime.executiongraph.failover.RestartBackoffTimeStrategy;
import org.apache.flink.util.clock.Clock;
import org.apache.flink.util.clock.ManualClock;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

class ExponentialDelayRestartBackoffTimeStrategyTest {
    private final Exception failure = new Exception();

    ExponentialDelayRestartBackoffTimeStrategyTest() {
    }

    @Test
    void testMaxAttempts() {
        int maxAttempts = 13;
        ManualClock clock = new ManualClock();
        long maxBackoffMS = 3L;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)clock, 1L, maxBackoffMS, 1.2, 10L, 0.25, maxAttempts);
        for (int i = 0; i <= maxAttempts; ++i) {
            Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
            Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
            clock.advanceTime(Duration.ofMillis(maxBackoffMS + 1L));
        }
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isFalse();
    }

    @Test
    void testNotCallNotifyFailure() {
        long initialBackoffMS = 42L;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)new ManualClock(), initialBackoffMS, 45L, 2.0, 8L, 0.0, 10);
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> ((ExponentialDelayRestartBackoffTimeStrategy)restartStrategy).getBackoffTime()).isInstanceOf(IllegalStateException.class)).hasMessage("Please call notifyFailure first.");
    }

    @Test
    void testInitialBackoff() {
        long initialBackoffMS = 42L;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)new ManualClock(), initialBackoffMS, 45L, 2.0, 8L, 0.0, Integer.MAX_VALUE);
        restartStrategy.notifyFailure((Throwable)this.failure);
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(initialBackoffMS);
    }

    @Test
    void testMaxBackoff() {
        long maxBackoffMS = 6L;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)new ManualClock(), 1L, 6L, 2.0, 8L, 0.25, Integer.MAX_VALUE);
        for (int i = 0; i < 10; ++i) {
            restartStrategy.notifyFailure((Throwable)this.failure);
            Assertions.assertThat((long)restartStrategy.getBackoffTime()).isLessThanOrEqualTo(6L);
        }
    }

    @Test
    void testResetBackoff() {
        long initialBackoffMS = 1L;
        long resetBackoffThresholdMS = 8L;
        ManualClock clock = new ManualClock();
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)clock, 1L, 5L, 2.0, 8L, 0.25, Integer.MAX_VALUE);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        clock.advanceTime(8L + restartStrategy.getBackoffTime() - 1L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        ((AbstractLongAssert)Assertions.assertThat((long)restartStrategy.getBackoffTime()).as("Backoff should be increased", new Object[0])).isEqualTo(2L);
        clock.advanceTime(8L + restartStrategy.getBackoffTime(), TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        ((AbstractLongAssert)Assertions.assertThat((long)restartStrategy.getBackoffTime()).as("Backoff should be reset", new Object[0])).isEqualTo(1L);
    }

    @Test
    void testBackoffMultiplier() {
        long initialBackoffMS = 4L;
        double jitterFactor = 0.0;
        double backoffMultiplier = 2.3;
        long maxBackoffMS = 300L;
        ManualClock clock = new ManualClock();
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)clock, initialBackoffMS, maxBackoffMS, backoffMultiplier, Integer.MAX_VALUE, jitterFactor, 10);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(4L);
        clock.advanceTime(Duration.ofMillis(maxBackoffMS + 1L));
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(9L);
        clock.advanceTime(Duration.ofMillis(maxBackoffMS + 1L));
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(21L);
        clock.advanceTime(Duration.ofMillis(maxBackoffMS + 1L));
    }

    @Test
    void testJitter() throws Exception {
        long initialBackoffMS = 2L;
        long maxBackoffMS = 7L;
        ManualClock clock = new ManualClock();
        ExponentialDelayRestartBackoffTimeStrategy.ExponentialDelayRestartBackoffTimeStrategyFactory restartStrategyFactory = new ExponentialDelayRestartBackoffTimeStrategy.ExponentialDelayRestartBackoffTimeStrategyFactory((Clock)clock, 2L, 7L, 2.0, Integer.MAX_VALUE, 0.25, Integer.MAX_VALUE);
        this.assertCorrectRandomRangeWithFailureCount(restartStrategyFactory, clock, 8L, 2, 3L, 4L, 5L);
        this.assertCorrectRandomRangeWithFailureCount(restartStrategyFactory, clock, 8L, 3, 6L, 7L);
        this.assertCorrectRandomRangeWithFailureCount(restartStrategyFactory, clock, 8L, 4, 7L);
    }

    @Test
    void testJitterNoHigherThanMax() throws Exception {
        double jitterFactor = 1.0;
        long maxBackoffMS = 7L;
        ManualClock clock = new ManualClock();
        ExponentialDelayRestartBackoffTimeStrategy.ExponentialDelayRestartBackoffTimeStrategyFactory restartStrategyFactory = new ExponentialDelayRestartBackoffTimeStrategy.ExponentialDelayRestartBackoffTimeStrategyFactory((Clock)clock, 1L, maxBackoffMS, 2.0, Integer.MAX_VALUE, jitterFactor, Integer.MAX_VALUE);
        this.assertCorrectRandomRangeWithFailureCount(restartStrategyFactory, clock, maxBackoffMS + 1L, 1, 1L, 2L);
        this.assertCorrectRandomRangeWithFailureCount(restartStrategyFactory, clock, maxBackoffMS + 1L, 2, 1L, 2L, 3L, 4L);
        this.assertCorrectRandomRangeWithFailureCount(restartStrategyFactory, clock, maxBackoffMS + 1L, 3, 1L, 2L, 3L, 4L, 5L, 6L, 7L);
    }

    private void assertCorrectRandomRangeWithFailureCount(ExponentialDelayRestartBackoffTimeStrategy.ExponentialDelayRestartBackoffTimeStrategyFactory factory, ManualClock clock, long advanceMsEachFailure, int failureCount, Long ... expectedNumbers) throws Exception {
        this.assertCorrectRandomRange(() -> {
            RestartBackoffTimeStrategy restartStrategy = factory.create();
            for (int i = 0; i < failureCount; ++i) {
                clock.advanceTime(Duration.ofMillis(advanceMsEachFailure));
                Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
            }
            return restartStrategy.getBackoffTime();
        }, expectedNumbers);
    }

    @Test
    void testMultipleSettings() {
        ManualClock clock = new ManualClock();
        long initialBackoffMS = 1L;
        long maxBackoffMS = 9L;
        double backoffMultiplier = 2.0;
        long resetBackoffThresholdMS = 80L;
        double jitterFactor = 0.25;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)clock, 1L, 9L, backoffMultiplier, 80L, jitterFactor, Integer.MAX_VALUE);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(1L);
        clock.advanceTime(81L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(1L);
        clock.advanceTime(4L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(2L);
        clock.advanceTime(90L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isOne();
        clock.advanceTime(Duration.ofMillis(10L));
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
        Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(2L);
    }

    @Test
    void testMergeMultipleExceptionsIntoOneAttempt() {
        ManualClock clock = new ManualClock();
        long initialBackoffMS = 2L;
        double backoffMultiplier = 2.0;
        long maxBackoffMS = 6L;
        long resetBackoffThresholdMS = 80L;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)clock, initialBackoffMS, 6L, backoffMultiplier, 80L, 0.0, 3);
        long currentBackOffMs = initialBackoffMS;
        this.checkMultipleExceptionsAreMerged(clock, currentBackOffMs, restartStrategy);
        clock.advanceTime(1L, TimeUnit.MILLISECONDS);
        currentBackOffMs = (long)((double)currentBackOffMs * backoffMultiplier);
        this.checkMultipleExceptionsAreMerged(clock, currentBackOffMs, restartStrategy);
        clock.advanceTime(1L, TimeUnit.MILLISECONDS);
        currentBackOffMs = 6L;
        this.checkMultipleExceptionsAreMerged(clock, currentBackOffMs, restartStrategy);
        clock.advanceTime(1L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isTrue();
        Assertions.assertThat((boolean)restartStrategy.canRestart()).isFalse();
    }

    @Test
    void testMergingExceptionsWorksWithResetting() {
        ManualClock clock = new ManualClock();
        long initialBackoffMS = 2L;
        double backoffMultiplier = 2.0;
        long maxBackoffMS = 6L;
        long resetBackoffThresholdMS = 80L;
        ExponentialDelayRestartBackoffTimeStrategy restartStrategy = new ExponentialDelayRestartBackoffTimeStrategy((Clock)clock, initialBackoffMS, 6L, backoffMultiplier, 80L, 0.0, 3);
        for (int i = 0; i < 10; ++i) {
            long currentBackOffMs = initialBackoffMS;
            this.checkMultipleExceptionsAreMerged(clock, currentBackOffMs, restartStrategy);
            clock.advanceTime(1L, TimeUnit.MILLISECONDS);
            currentBackOffMs = (long)((double)currentBackOffMs * backoffMultiplier);
            this.checkMultipleExceptionsAreMerged(clock, currentBackOffMs, restartStrategy);
            clock.advanceTime(1L, TimeUnit.MILLISECONDS);
            currentBackOffMs = 6L;
            this.checkMultipleExceptionsAreMerged(clock, currentBackOffMs, restartStrategy);
            clock.advanceTime(80L, TimeUnit.MILLISECONDS);
        }
    }

    private void checkMultipleExceptionsAreMerged(ManualClock clock, long expectedBackoffMS, ExponentialDelayRestartBackoffTimeStrategy restartStrategy) {
        boolean expectedNewAttempt = true;
        int advanceMs = 0;
        while ((long)advanceMs < expectedBackoffMS) {
            for (int i = 0; i < 10; ++i) {
                Assertions.assertThat((boolean)restartStrategy.notifyFailure((Throwable)this.failure)).isEqualTo(expectedNewAttempt);
                if (expectedNewAttempt) {
                    expectedNewAttempt = false;
                }
                Assertions.assertThat((boolean)restartStrategy.canRestart()).isTrue();
                Assertions.assertThat((long)restartStrategy.getBackoffTime()).isEqualTo(expectedBackoffMS - (long)advanceMs);
            }
            clock.advanceTime(1L, TimeUnit.MILLISECONDS);
            ++advanceMs;
        }
    }

    private void assertCorrectRandomRange(Callable<Long> numberGenerator, Long ... expectedNumbers) throws Exception {
        HashSet<Long> generatedNumbers = new HashSet<Long>();
        for (int i = 0; i < 1000; ++i) {
            long generatedNumber = numberGenerator.call();
            generatedNumbers.add(generatedNumber);
        }
        Assertions.assertThat(generatedNumbers).isEqualTo(new HashSet<Long>(Arrays.asList(expectedNumbers)));
    }
}

