/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.decider;

import java.util.Map;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.InternalClusterInfoService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDecider;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.node.settings.NodeSettingsService;

public class DiskThresholdDecider
extends AllocationDecider {
    public static final String NAME = "disk_threshold";
    private volatile Double freeDiskThresholdLow;
    private volatile Double freeDiskThresholdHigh;
    private volatile ByteSizeValue freeBytesThresholdLow;
    private volatile ByteSizeValue freeBytesThresholdHigh;
    private volatile boolean enabled;
    public static final String CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED = "cluster.routing.allocation.disk.threshold_enabled";
    public static final String CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK = "cluster.routing.allocation.disk.watermark.low";
    public static final String CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK = "cluster.routing.allocation.disk.watermark.high";

    public DiskThresholdDecider(Settings settings) {
        this(settings, new NodeSettingsService(settings));
    }

    @Inject
    public DiskThresholdDecider(Settings settings, NodeSettingsService nodeSettingsService) {
        super(settings);
        String lowWatermark = settings.get(CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, "0.7");
        String highWatermark = settings.get(CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, "0.85");
        if (!this.validWatermarkSetting(lowWatermark)) {
            throw new ElasticsearchParseException("Unable to parse low watermark: [" + lowWatermark + "]");
        }
        if (!this.validWatermarkSetting(highWatermark)) {
            throw new ElasticsearchParseException("Unable to parse high watermark: [" + highWatermark + "]");
        }
        this.freeDiskThresholdLow = 100.0 - this.thresholdPercentageFromWatermark(lowWatermark);
        this.freeDiskThresholdHigh = 100.0 - this.thresholdPercentageFromWatermark(highWatermark);
        this.freeBytesThresholdLow = this.thresholdBytesFromWatermark(lowWatermark);
        this.freeBytesThresholdHigh = this.thresholdBytesFromWatermark(highWatermark);
        this.enabled = settings.getAsBoolean(CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, (Boolean)false);
        nodeSettingsService.addListener(new ApplySettings());
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!this.enabled) {
            return allocation.decision(Decision.YES, NAME, "disk threshold decider disabled", new Object[0]);
        }
        if (allocation.nodes().size() <= 1) {
            return allocation.decision(Decision.YES, NAME, "only a single node is present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Cluster info unavailable for disk threshold decider, allowing allocation.", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "cluster info unavailable", new Object[0]);
        }
        Map<String, DiskUsage> usages = clusterInfo.getNodeDiskUsages();
        Map<String, Long> shardSizes = clusterInfo.getShardSizes();
        if (usages.isEmpty()) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Unable to determine disk usages for disk-aware allocation, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "disk usages unavailable", new Object[0]);
        }
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Unable to determine disk usage for [{}], defaulting to average across nodes [{} total] [{} free] [{}% free]", node.nodeId(), usage.getTotalBytes(), usage.getFreeBytes(), usage.getFreeDiskAsPercentage());
            }
        }
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Node [{}] has {}% free disk", node.nodeId(), freeDiskPercentage);
        }
        if (freeBytes < this.freeBytesThresholdLow.bytes()) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, preventing allocation", this.freeBytesThresholdLow, freeBytes, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "less than required [%s] free on node, free: [%s]", this.freeBytesThresholdLow, new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.freeDiskThresholdLow) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {}% free disk threshold ({}% free) on node [{}], preventing allocation", this.freeDiskThresholdLow, freeDiskPercentage, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdLow, this.freeDiskThresholdLow);
        }
        Long shardSize = shardSizes.get(InternalClusterInfoService.shardIdentifierFromRouting(shardRouting));
        shardSize = shardSize == null ? 0L : shardSize;
        double freeSpaceAfterShard = this.freeDiskPercentageAfterShardAssigned(usage, shardSize);
        long freeBytesAfterShard = freeBytes - shardSize;
        if (freeBytesAfterShard < this.freeBytesThresholdHigh.bytes()) {
            this.logger.warn("After allocating, node [{}] would have less than the required {} free bytes threshold ({} bytes free), preventing allocation", node.nodeId(), this.freeBytesThresholdHigh, freeBytesAfterShard);
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s] free on node, free: [%s]", this.freeBytesThresholdLow, new ByteSizeValue(freeBytesAfterShard));
        }
        if (freeSpaceAfterShard < this.freeDiskThresholdHigh) {
            this.logger.warn("After allocating, node [{}] would have less than the required {}% free disk threshold ({}% free), preventing allocation", node.nodeId(), this.freeDiskThresholdHigh, freeSpaceAfterShard);
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdLow, freeSpaceAfterShard);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard on node, free: [%s]", new ByteSizeValue(freeBytes));
    }

    @Override
    public Decision canRemain(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!this.enabled) {
            return allocation.decision(Decision.YES, NAME, "disk threshold decider disabled", new Object[0]);
        }
        if (allocation.nodes().size() <= 1) {
            return allocation.decision(Decision.YES, NAME, "only a single node is present", new Object[0]);
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        if (clusterInfo == null) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Cluster info unavailable for disk threshold decider, allowing allocation.", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "cluster info unavailable", new Object[0]);
        }
        Map<String, DiskUsage> usages = clusterInfo.getNodeDiskUsages();
        if (usages.isEmpty()) {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Unable to determine disk usages for disk-aware allocation, allowing allocation", new Object[0]);
            }
            return allocation.decision(Decision.YES, NAME, "disk usages unavailable", new Object[0]);
        }
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = this.averageUsage(node, usages);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Unable to determine disk usage for {}, defaulting to average across nodes [{} total] [{} free] [{}% free]", node.nodeId(), usage.getTotalBytes(), usage.getFreeBytes(), usage.getFreeDiskAsPercentage());
            }
        }
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("Node [{}] has {}% free disk ({} bytes)", node.nodeId(), freeDiskPercentage, freeBytes);
        }
        if (freeBytes < this.freeBytesThresholdHigh.bytes()) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {} free bytes threshold ({} bytes free) on node {}, shard cannot remain", this.freeBytesThresholdHigh, freeBytes, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s] free on node, free: [%s]", this.freeBytesThresholdHigh, new ByteSizeValue(freeBytes));
        }
        if (freeDiskPercentage < this.freeDiskThresholdHigh) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Less than the required {}% free disk threshold ({}% free) on node {}, shard cannot remain", this.freeDiskThresholdHigh, freeDiskPercentage, node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "after allocation less than required [%s%%] free disk on node, free: [%s%%]", this.freeDiskThresholdHigh, freeDiskPercentage);
        }
        return allocation.decision(Decision.YES, NAME, "enough disk for shard to remain on node, free: [%s]", new ByteSizeValue(freeBytes));
    }

    public DiskUsage averageUsage(RoutingNode node, Map<String, DiskUsage> usages) {
        long totalBytes = 0L;
        long freeBytes = 0L;
        for (DiskUsage du : usages.values()) {
            totalBytes += du.getTotalBytes();
            freeBytes += du.getFreeBytes();
        }
        return new DiskUsage(node.nodeId(), totalBytes / (long)usages.size(), freeBytes / (long)usages.size());
    }

    public double freeDiskPercentageAfterShardAssigned(DiskUsage usage, Long shardSize) {
        shardSize = shardSize == null ? 0L : shardSize;
        return 100.0 - (double)(usage.getUsedBytes() + shardSize) / (double)usage.getTotalBytes() * 100.0;
    }

    public double thresholdPercentageFromWatermark(String watermark) {
        try {
            return 100.0 * Double.parseDouble(watermark);
        }
        catch (NumberFormatException ex) {
            return 100.0;
        }
    }

    public ByteSizeValue thresholdBytesFromWatermark(String watermark) {
        try {
            return ByteSizeValue.parseBytesSizeValue(watermark);
        }
        catch (ElasticsearchParseException ex) {
            return ByteSizeValue.parseBytesSizeValue("0b");
        }
    }

    public boolean validWatermarkSetting(String watermark) {
        try {
            double w = Double.parseDouble(watermark);
            return !(w < 0.0) && !(w > 1.0);
        }
        catch (NumberFormatException e) {
            try {
                ByteSizeValue.parseBytesSizeValue(watermark);
                return true;
            }
            catch (ElasticsearchParseException ex) {
                return false;
            }
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            String newLowWatermark = settings.get(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, null);
            String newHighWatermark = settings.get(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, null);
            Boolean newEnableSetting = settings.getAsBoolean(DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, null);
            if (newEnableSetting != null) {
                DiskThresholdDecider.this.logger.info("updating [{}] from [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_DISK_THRESHOLD_ENABLED, DiskThresholdDecider.this.enabled, newEnableSetting);
                DiskThresholdDecider.this.enabled = newEnableSetting;
            }
            if (newLowWatermark != null) {
                if (!DiskThresholdDecider.this.validWatermarkSetting(newLowWatermark)) {
                    throw new ElasticsearchParseException("Unable to parse low watermark: [" + newLowWatermark + "]");
                }
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_LOW_DISK_WATERMARK, newLowWatermark);
                DiskThresholdDecider.this.freeDiskThresholdLow = 100.0 - DiskThresholdDecider.this.thresholdPercentageFromWatermark(newLowWatermark);
                DiskThresholdDecider.this.freeBytesThresholdLow = DiskThresholdDecider.this.thresholdBytesFromWatermark(newLowWatermark);
            }
            if (newHighWatermark != null) {
                if (!DiskThresholdDecider.this.validWatermarkSetting(newHighWatermark)) {
                    throw new ElasticsearchParseException("Unable to parse high watermark: [" + newHighWatermark + "]");
                }
                DiskThresholdDecider.this.logger.info("updating [{}] to [{}]", DiskThresholdDecider.CLUSTER_ROUTING_ALLOCATION_HIGH_DISK_WATERMARK, newHighWatermark);
                DiskThresholdDecider.this.freeDiskThresholdHigh = 100.0 - DiskThresholdDecider.this.thresholdPercentageFromWatermark(newHighWatermark);
                DiskThresholdDecider.this.freeBytesThresholdHigh = DiskThresholdDecider.this.thresholdBytesFromWatermark(newHighWatermark);
            }
        }
    }
}

