/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.jms;

import java.io.IOException;
import java.util.Map;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.InitialContext;

import org.jboss.logging.Logger;

/**
 * 
 * @author <mailto:clebert.suconic@jboss.org">Clebert Suconic</a>
 * 
 */
public class HornetQTestAdmin extends JMSTestAdmin
{
   private final MBeanServerConnection jmx;

   /** The static log */
   private static final Logger staticLog = Logger.getLogger(HornetQTestAdmin.class);

   protected final ObjectName hornetQJMSServerName = new ObjectName("org.hornetq:module=JMS,type=Server");

   protected final ObjectName hornetQCoreServerName = new ObjectName("org.hornetq:module=Core,type=Server");

   public HornetQTestAdmin() throws Exception
   {
      InitialContext ctx = new InitialContext();
      String adaptorName = System.getProperty("jbosstest.server.name", "jmx/invoker/RMIAdaptor");
      jmx = (MBeanServerConnection) ctx.lookup(adaptorName);
      ctx.close();
   }

   @Override
   public boolean isSupported()
   {
      try
      {
         createQueue("deleteMe");
         destroyCreatedDestinations();
         staticLog.info("HornetQ is supported");
         return true;
      } catch (Exception e)
      {
         staticLog.debug("HornetQTestAdmin is not support because an exception happened", e);
         return false;
      }
   }

   public void createQueue(String name, TestRole... securityConfig) throws Exception
   {
      createQueue(name, new String[]
      { "/queue/" + name, name }, securityConfig);
   }

   private void addRoles(String address, TestRole[] securityConf) throws Exception
   {

      if (securityConf == null || securityConf.length == 0)
      {
         securityConf = new TestRole[]
         { new TestRole("guest", true, true, true) };
      }

      String sendRoles = "";
      String consumeRoles = "";
      String createDurableQueueRoles = "";
      String deleteDurableQueueRoles = "";
      String createTempQueueRoles = "";
      String deleteTempQueueRoles = "";
      String manageRoles = "";

      for (TestRole role : securityConf)
      {
         if (role.isSend())
         {
            sendRoles += isEmpty(sendRoles) ? role.getName() : "," + role.getName();
         }

         if (role.isConsume())
         {
            consumeRoles += isEmpty(consumeRoles) ? role.getName() : "," + role.getName();
         }

         if (role.isCreateDurableQueue())
         {
            createDurableQueueRoles += isEmpty(createDurableQueueRoles) ? role.getName() : "," + role.getName();
         }

         if (role.isDeleteDurableQueue())
         {
            deleteDurableQueueRoles += isEmpty(deleteDurableQueueRoles) ? role.getName() : "," + role.getName();
         }

         if (role.isCreateNonDurableQueue())
         {
            createTempQueueRoles += isEmpty(createTempQueueRoles) ? role.getName() : "," + role.getName();
         }

         if (role.isDeleteNonDurableQueue())
         {
            deleteTempQueueRoles += isEmpty(deleteTempQueueRoles) ? role.getName() : "," + role.getName();
         }

         if (role.isManage())
         {
            manageRoles += isEmpty(manageRoles) ? role.getName() : "," + role.getName();
         }
      }

      jmx.invoke(hornetQCoreServerName, "addSecuritySettings", new Object[]
      { address, sendRoles, consumeRoles, createDurableQueueRoles, deleteDurableQueueRoles, createTempQueueRoles, deleteTempQueueRoles, manageRoles }, new String[]
      { String.class.getName(), String.class.getName(), String.class.getName(), String.class.getName(), String.class.getName(), String.class.getName(), String.class.getName(), String.class.getName() });
   }

   private boolean isEmpty(String str)
   {
      return (str.length() == 0);
   }

   /**
    * @param isTopic
    * @param address
    * @return
    */
   private String getCoreAddress(boolean isTopic, String address)
   {
      if (isTopic)
      {
         address = "jms.topic." + address;
      } else
      {
         address = "jms.queue." + address;
      }
      return address;
   }

   private void resetRoles(boolean isTopic, String address) throws Exception
   {
      jmx.invoke(hornetQCoreServerName, "removeSecuritySettings", new Object[]
      { address }, new String[]
      { String.class.getName() });

   }

   public void createQueue(String name, String[] jndiArray, TestRole... securityConfig) throws Exception
   {
      for (String jndi : jndiArray)
      {
         jmx.invoke(hornetQJMSServerName, "createQueue", new Object[]
         { name, jndi }, new String[]
         { "java.lang.String", "java.lang.String" });
      }

      addRoles(getCoreAddress(false, name), securityConfig);

      deployedQueues.put(name, securityConfig);
   }

   public void createTopic(String name, TestRole... securityConfig) throws Exception
   {
      createTopic(name, new String[]
      { "/topic/" + name, name }, securityConfig);
   }

   public void createTopic(String name, String[] jndiArray, TestRole... securityConfig) throws Exception
   {
      for (String jndi : jndiArray)
      {
         jmx.invoke(hornetQJMSServerName, "createTopic", new Object[]
         { name, jndi }, new String[]
         { "java.lang.String", "java.lang.String" });
      }

      addRoles(getCoreAddress(true, name), securityConfig);

      deployedTopics.put(name, securityConfig);
   }

   public void deleteQueue(String name) throws Exception
   {
      try
      {
         deployedQueues.remove(name);
         resetRoles(false, name);
         invokeDeleteQueue(name);
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         throw e;
      }
   }

   public void deleteTopic(String name) throws Exception
   {
      try
      {
         deployedTopics.remove(name);
         resetRoles(true, name);
         invokeDeleteTopic(name);
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         throw e;
      }
   }

   public void destroyEveryDestination() throws Exception
   {
      String[] queues = (String[]) jmx.getAttribute(hornetQJMSServerName, "QueueNames");

      for (String queue : queues)
      {
         if (!queue.equals("DLQ") && !queue.equals("ExpiryQueue"))
         {
            deleteQueue(queue);
         }
      }

      String[] topics = (String[]) jmx.getAttribute(hornetQJMSServerName, "TopicNames");

      for (String topic : topics)
      {
         deleteTopic(topic);
      }
   }

   @Override
   public void destroyCreatedDestinations() throws Exception
   {
      for (Map.Entry<String, TestRole[]> queue : deployedQueues.entrySet())
      {
         resetRoles(false, queue.getKey());
         try
         {
            invokeDeleteQueue(queue.getKey());
         } catch (Exception e)
         {
            staticLog.warn("Error deleting queue " + queue.getKey(), e);
         }
      }

      for (Map.Entry<String, TestRole[]> topic : deployedTopics.entrySet())
      {
         resetRoles(true, topic.getKey());
         try
         {
            invokeDeleteTopic(topic.getKey());
         } catch (Exception e)
         {
            staticLog.warn("Error deleting topic " + topic.getKey(), e);
         }
      }

      deployedQueues.clear();
      deployedTopics.clear();
   }

   private void invokeDeleteQueue(String name) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException
   {
      jmx.invoke(hornetQJMSServerName, "destroyQueue", new Object[]
      { name }, new String[]
      { "java.lang.String" });
   }

   private void invokeDeleteTopic(String name) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException
   {
      jmx.invoke(hornetQJMSServerName, "destroyTopic", new Object[]
      { name }, new String[]
      { "java.lang.String" });
   }

   @Override
   public ObjectName createQueueJMXName(String queueName)
   {
      try
      {
         return new ObjectName("org.hornetq:module=JMS,name=\"" + queueName + "\",type=Queue");
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         e.printStackTrace(); // >> junit reports
         return null;
      }
   }

   @Override
   public ObjectName createTopicJMXName(String queueName)
   {
      try
      {
         return new ObjectName("org.hornetq:module=JMS,name=\"" + queueName + "\",type=Topic");
      } catch (Exception e)
      {
         staticLog.warn(e.getMessage(), e);
         e.printStackTrace(); // >> junit reports
         return null;
      }
   }

}
