/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.metadata.spi.retrieval.cummulative;

import java.lang.annotation.Annotation;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jboss.metadata.spi.context.MetaDataContext;
import org.jboss.metadata.spi.retrieval.AnnotationItem;
import org.jboss.metadata.spi.retrieval.AnnotationsItem;
import org.jboss.metadata.spi.retrieval.MetaDataRetrieval;
import org.jboss.metadata.spi.retrieval.simple.SimpleAnnotationsItem;

/**
 * CummulativeAnnotationsItem.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 69888 $
 */
public class CummulativeAnnotationsItem extends SimpleAnnotationsItem
{
   /** The context */
   private MetaDataContext context;
   
   /** Whether to include the parent */
   private boolean includeParent;
   
   /** The valid time */
   private long validTime;
   
   /**
    * Create a new CummulativeAnnotationsItem.
    * 
    * @param context the context
    * @param includeParent whether to include the parent
    */
   public CummulativeAnnotationsItem(MetaDataContext context, boolean includeParent)
   {
      if (context == null)
         throw new IllegalArgumentException("Null context");

      this.context = context;
      this.includeParent = includeParent;
      init(context.getValidTime().getValidTime());
   }

   public Annotation[] getValue()
   {
      checkValid();
      return super.getValue();
   }
   
   public AnnotationItem<? extends Annotation>[] getAnnotations()
   {
      checkValid();
      return super.getAnnotations();
   }

   public boolean isCachable()
   {
      return true;
   }

   public boolean isValid()
   {
      return true;
   }

   /**
    * Check whether we are valid
    */
   protected void checkValid()
   {
      AnnotationItem<? extends Annotation>[] items = super.getAnnotations();
      boolean valid = (items != null);
      
      long newValidTime = context.getValidTime().getValidTime();
      if (validTime < newValidTime)
         valid = false;
      
      if (valid && items != null)
      {
         for (AnnotationItem<? extends Annotation> item : items)
         {
            if (item.isValid() == false)
               valid = false;
         }
      }
      
      if (valid == false)
         init(newValidTime);
   }

   /**
    * Initialise
    * 
    * @param validTime the valid time
    */
   @SuppressWarnings("unchecked")
   protected void init(long validTime)
   {
      Set<AnnotationItem<? extends Annotation>> temp = null;

      List<MetaDataRetrieval> retrievals;
      
      if (includeParent)
         retrievals = context.getRetrievals();
      else
         retrievals = context.getLocalRetrievals();
      
      for (MetaDataRetrieval retrieval : retrievals)
      {
         AnnotationsItem item = retrieval.retrieveAnnotations();
         if (item != null)
         {
            AnnotationItem<? extends Annotation>[] items = item.getAnnotations();
            for (AnnotationItem<? extends Annotation> it : items)
            {
               if (temp == null)
                  temp = new HashSet<AnnotationItem<? extends Annotation>>();
               temp.add(it);
            }
         }
      }
      
      AnnotationItem<? extends Annotation>[] items = NO_ANNOTATION_ITEMS;
      if (temp != null)
         items = temp.toArray(new AnnotationItem[temp.size()]);
      setAnnotationItems(items);
      this.validTime = validTime;
   }
}
