/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import com.amazonaws.AmazonClientException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.DefaultAWSCredentialsProviderChain;
import com.amazonaws.auth.PropertiesFileCredentialsProvider;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.SSEAwsKeyManagementParams;
import com.amazonaws.services.securitytoken.AWSSecurityTokenService;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClient;
import com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder;
import java.io.File;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.AutoAbortingGenericInputStream;
import kafka.tier.store.S3TierObjectStoreConfig;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import org.apache.kafka.common.utils.ByteBufferInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3TierObjectStore
implements TierObjectStore {
    private static final Logger log = LoggerFactory.getLogger(S3TierObjectStore.class);
    private final Optional<String> clusterIdOpt;
    private final Optional<Integer> brokerIdOpt;
    private final AmazonS3 client;
    private final String bucket;
    private final String prefix;
    private final String sseAlgorithm;
    private final String sseCustomerEncryptionKey;
    private final int autoAbortThresholdBytes;

    public S3TierObjectStore(S3TierObjectStoreConfig config) {
        this(S3TierObjectStore.client(config), config);
    }

    S3TierObjectStore(AmazonS3 client, S3TierObjectStoreConfig config) {
        this.clusterIdOpt = config.clusterIdOpt;
        this.brokerIdOpt = config.brokerIdOpt;
        this.client = client;
        this.bucket = config.s3Bucket;
        this.prefix = config.s3Prefix;
        this.sseAlgorithm = config.s3SseAlgorithm;
        this.sseCustomerEncryptionKey = config.s3SseCustomerEncryptionKey;
        this.autoAbortThresholdBytes = config.s3AutoAbortThresholdBytes;
        this.expectBucket(this.bucket, config.s3Region);
    }

    @Override
    public TierObjectStore.Backend getBackend() {
        return TierObjectStore.Backend.S3;
    }

    @Override
    public TierObjectStoreResponse getObject(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType, Integer byteOffsetStart, Integer byteOffsetEnd) {
        S3Object object;
        String key = this.keyPath(objectMetadata, fileType);
        GetObjectRequest request = new GetObjectRequest(this.bucket, key);
        if (byteOffsetStart != null && byteOffsetEnd != null) {
            request.setRange((long)byteOffsetStart.intValue(), (long)byteOffsetEnd.intValue());
        } else if (byteOffsetStart != null) {
            request.setRange((long)byteOffsetStart.intValue());
        } else if (byteOffsetEnd != null) {
            throw new IllegalStateException("Cannot specify a byteOffsetEnd without specifying a byteOffsetStart");
        }
        log.debug("Fetching object from s3://{}/{}, with range start {}", new Object[]{this.bucket, key, byteOffsetStart});
        try {
            object = this.client.getObject(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to fetch object, metadata: %s type: %s range %s-%s", new Object[]{objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Unknown exception when fetching object, metadata: %s type: %s range %s-%s", new Object[]{objectMetadata, fileType, byteOffsetStart, byteOffsetEnd}), e);
        }
        S3ObjectInputStream inputStream = object.getObjectContent();
        return new S3TierObjectStoreResponse(inputStream, this.autoAbortThresholdBytes, object.getObjectMetadata().getContentLength());
    }

    @Override
    public void putObject(TierObjectStore.ObjectStoreMetadata objectMetadata, File file, TierObjectStore.FileType fileType) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, fileType), metadata, file);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException(String.format("Failed to upload object %s, file %s, type %s", new Object[]{objectMetadata, file, fileType}), e);
        }
    }

    @Override
    public void putSegment(TierObjectStore.ObjectMetadata objectMetadata, File segmentData, File offsetIndexData, File timestampIndexData, Optional<File> producerStateSnapshotData, Optional<ByteBuffer> transactionIndexData, Optional<ByteBuffer> epochState) {
        Map<String, String> metadata = objectMetadata.objectMetadata(this.clusterIdOpt, this.brokerIdOpt);
        try {
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.SEGMENT), metadata, segmentData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.OFFSET_INDEX), metadata, offsetIndexData);
            this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.TIMESTAMP_INDEX), metadata, timestampIndexData);
            producerStateSnapshotData.ifPresent(file -> this.putFile(this.keyPath(objectMetadata, TierObjectStore.FileType.PRODUCER_STATE), metadata, (File)file));
            transactionIndexData.ifPresent(abortedTxnsBuf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.TRANSACTION_INDEX), metadata, (ByteBuffer)abortedTxnsBuf));
            epochState.ifPresent(buf -> this.putBuf(this.keyPath(objectMetadata, TierObjectStore.FileType.EPOCH_STATE), metadata, (ByteBuffer)buf));
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to upload segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when uploading segment: " + objectMetadata, e);
        }
    }

    @Override
    public void deleteSegment(TierObjectStore.ObjectMetadata objectMetadata) {
        ArrayList<DeleteObjectsRequest.KeyVersion> keys = new ArrayList<DeleteObjectsRequest.KeyVersion>();
        for (TierObjectStore.FileType type : TierObjectStore.FileType.values()) {
            String keyPath = this.keyPath(objectMetadata, type);
            log.debug("Deleting object s3://{}/{}", (Object)this.bucket, (Object)keyPath);
            keys.add(new DeleteObjectsRequest.KeyVersion(keyPath));
        }
        DeleteObjectsRequest request = new DeleteObjectsRequest(this.bucket).withKeys(keys);
        try {
            this.client.deleteObjects(request);
        }
        catch (AmazonClientException e) {
            throw new TierObjectStoreRetriableException("Failed to delete segment: " + objectMetadata, e);
        }
        catch (Exception e) {
            throw new TierObjectStoreFatalException("Unknown exception when deleting segment: " + objectMetadata, e);
        }
    }

    @Override
    public void close() {
        this.client.shutdown();
    }

    private String keyPath(TierObjectStore.ObjectStoreMetadata objectMetadata, TierObjectStore.FileType fileType) {
        return objectMetadata.toPath(this.prefix, fileType);
    }

    private ObjectMetadata putObjectMetadata(Map<String, String> userMetadata) {
        ObjectMetadata metadata = new ObjectMetadata();
        if (this.sseAlgorithm != null) {
            metadata.setSSEAlgorithm(this.sseAlgorithm);
        }
        if (userMetadata != null) {
            metadata.setUserMetadata(userMetadata);
        }
        return metadata;
    }

    private void putFile(String key, Map<String, String> metadata, File file) {
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, file).withMetadata(this.putObjectMetadata(metadata));
        if (this.sseCustomerEncryptionKey != null && this.sseAlgorithm.equals("aws:kms")) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
        log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        this.client.putObject(request);
    }

    private void putBuf(String key, Map<String, String> metadata, ByteBuffer buf) {
        ObjectMetadata s3metadata = this.putObjectMetadata(metadata);
        s3metadata.setContentLength((long)(buf.limit() - buf.position()));
        PutObjectRequest request = new PutObjectRequest(this.bucket, key, (InputStream)new ByteBufferInputStream(buf.duplicate()), s3metadata);
        if (this.sseCustomerEncryptionKey != null && !this.sseCustomerEncryptionKey.isEmpty() && this.sseAlgorithm.equals("aws:kms")) {
            request.setSSEAwsKeyManagementParams(new SSEAwsKeyManagementParams(this.sseCustomerEncryptionKey));
        }
        log.debug("Uploading object to s3://{}/{}", (Object)this.bucket, (Object)key);
        this.client.putObject(request);
    }

    public static AmazonS3 client(S3TierObjectStoreConfig config) {
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        clientConfiguration.setUserAgentPrefix(config.s3UserAgentPrefix);
        AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard();
        builder.setClientConfiguration(clientConfiguration);
        if (config.s3ForcePathStyleAccess.booleanValue()) {
            builder.setPathStyleAccessEnabled(Boolean.valueOf(true));
        }
        if (config.s3SignerOverride.isPresent() && !config.s3SignerOverride.get().isEmpty()) {
            clientConfiguration.setSignerOverride(config.s3SignerOverride.get());
        }
        if (config.s3EndpointOverride.isPresent() && !config.s3EndpointOverride.get().isEmpty()) {
            builder.setEndpointConfiguration(new AwsClientBuilder.EndpointConfiguration(config.s3EndpointOverride.get(), Regions.fromName((String)config.s3Region).getName()));
        } else if (config.s3Region != null && !config.s3Region.isEmpty()) {
            builder.setRegion(config.s3Region);
        }
        AWSCredentialsProvider provider = config.s3CredFilePath.map(PropertiesFileCredentialsProvider::new).orElse((AWSCredentialsProvider)new DefaultAWSCredentialsProviderChain());
        if (config.assumeRoleArn.isPresent()) {
            AWSSecurityTokenService stsClient = (AWSSecurityTokenService)((AWSSecurityTokenServiceClientBuilder)AWSSecurityTokenServiceClient.builder().withCredentials(provider)).build();
            provider = new STSAssumeRoleSessionCredentialsProvider.Builder(config.assumeRoleArn.get(), "tiered-storage").withStsClient(stsClient).build();
        }
        builder.setCredentials(provider);
        return (AmazonS3)builder.build();
    }

    private void expectBucket(String bucket, String expectedRegion) throws TierObjectStoreFatalException {
        try {
            String actualRegion = this.client.getBucketLocation(bucket);
            if (actualRegion.equals("US") && expectedRegion.equals("us-east-1")) {
                return;
            }
            if (!expectedRegion.equals(actualRegion)) {
                log.warn("Bucket region {} does not match expected region {}", (Object)actualRegion, (Object)expectedRegion);
            }
        }
        catch (AmazonClientException ex) {
            throw new TierObjectStoreFatalException("Failed to access bucket " + bucket, ex);
        }
    }

    private static class S3TierObjectStoreResponse
    implements TierObjectStoreResponse {
        private final AutoAbortingGenericInputStream inputStream;

        S3TierObjectStoreResponse(S3ObjectInputStream inputStream, long autoAbortSize, long streamSize) {
            this.inputStream = new AutoAbortingGenericInputStream((InputStream)inputStream, autoAbortSize, streamSize);
        }

        @Override
        public void close() {
            this.inputStream.close();
        }

        @Override
        public InputStream getInputStream() {
            return this.inputStream;
        }
    }
}

