/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.util;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayDeque;
import java.util.Iterator;
import java.util.Locale;
import java.util.Objects;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileUtil {
    public static final int BUFFER_SIZE = 131072;
    private static final Logger LOG = LoggerFactory.getLogger(FileUtil.class);
    private static final int RETRY_SLEEP_MILLIS = 10;
    private static final String USER_DIR_KEY = "user.dir";
    private static final File USER_DIR = new File(System.getProperty("user.dir"));
    private static final boolean IS_WINDOWS = FileUtil.initWindowsOs();

    private FileUtil() {
    }

    private static boolean initWindowsOs() {
        String osName = System.getProperty("os.name").toLowerCase(Locale.ENGLISH);
        return osName.contains("windows");
    }

    public static File getUserDir() {
        return USER_DIR;
    }

    public static String normalizePath(String path) {
        if (path == null) {
            return null;
        }
        if (FileUtil.isWindows()) {
            return path.replace('/', '\\');
        }
        return path.replace('\\', '/');
    }

    public static boolean isWindows() {
        return IS_WINDOWS;
    }

    public static File createTempFile(String prefix, String suffix, File parentDir) throws IOException {
        Objects.requireNonNull(parentDir);
        if (suffix == null) {
            suffix = ".tmp";
        }
        if (prefix == null) {
            prefix = "camel";
        } else if (((String)prefix).length() < 3) {
            prefix = (String)prefix + "camel";
        }
        parentDir.mkdirs();
        return Files.createTempFile(parentDir.toPath(), (String)prefix, suffix, new FileAttribute[0]).toFile();
    }

    public static String stripLeadingSeparator(String name) {
        if (name == null) {
            return null;
        }
        while (name.startsWith("/") || name.startsWith(File.separator)) {
            name = name.substring(1);
        }
        return name;
    }

    public static boolean hasLeadingSeparator(String name) {
        if (name == null) {
            return false;
        }
        return name.startsWith("/") || name.startsWith(File.separator);
    }

    public static String stripFirstLeadingSeparator(String name) {
        if (name == null) {
            return null;
        }
        if (name.startsWith("/") || name.startsWith(File.separator)) {
            name = name.substring(1);
        }
        return name;
    }

    public static String stripTrailingSeparator(String name) {
        if (ObjectHelper.isEmpty(name)) {
            return name;
        }
        String s = name;
        while (s.endsWith("/") || s.endsWith(File.separator)) {
            s = s.substring(0, s.length() - 1);
        }
        if (ObjectHelper.isEmpty(s)) {
            return name;
        }
        return s;
    }

    public static String stripPath(String name) {
        int posWin;
        if (name == null) {
            return null;
        }
        int posUnix = name.lastIndexOf(47);
        int pos = Math.max(posUnix, posWin = name.lastIndexOf(92));
        if (pos != -1) {
            return name.substring(pos + 1);
        }
        return name;
    }

    public static String stripExt(String name) {
        return FileUtil.stripExt(name, false);
    }

    public static String stripExt(String name, boolean singleMode) {
        int posWin;
        if (name == null) {
            return null;
        }
        int posUnix = name.lastIndexOf(47);
        int pos = Math.max(posUnix, posWin = name.lastIndexOf(92));
        if (pos > 0) {
            int pos2;
            String onlyName = name.substring(pos + 1);
            int n = pos2 = singleMode ? onlyName.lastIndexOf(46) : onlyName.indexOf(46);
            if (pos2 > 0) {
                return name.substring(0, pos + pos2 + 1);
            }
        } else {
            int pos2;
            int n = pos2 = singleMode ? name.lastIndexOf(46) : name.indexOf(46);
            if (pos2 > 0) {
                return name.substring(0, pos2);
            }
        }
        return name;
    }

    public static String onlyExt(String name) {
        return FileUtil.onlyExt(name, false);
    }

    public static String onlyExt(String name, boolean singleMode) {
        int pos;
        if (name == null) {
            return null;
        }
        name = FileUtil.stripPath(name);
        int n = pos = singleMode ? name.lastIndexOf(46) : name.indexOf(46);
        if (pos != -1) {
            return name.substring(pos + 1);
        }
        return null;
    }

    public static String onlyPath(String name) {
        int posWin;
        if (name == null) {
            return null;
        }
        int posUnix = name.lastIndexOf(47);
        int pos = Math.max(posUnix, posWin = name.lastIndexOf(92));
        if (pos > 0) {
            return name.substring(0, pos);
        }
        if (pos == 0) {
            return name.substring(0, 1);
        }
        return null;
    }

    public static String onlyName(String name) {
        return FileUtil.onlyName(name, false);
    }

    public static String onlyName(String name, boolean singleMode) {
        name = FileUtil.stripPath(name);
        name = FileUtil.stripExt(name, singleMode);
        return name;
    }

    public static String compactPath(String path) {
        return FileUtil.compactPath(path, String.valueOf(File.separatorChar));
    }

    public static String compactPath(String path, char separator) {
        return FileUtil.compactPath(path, String.valueOf(separator));
    }

    public static String compactPath(String path, String separator) {
        String[] parts;
        if (path == null) {
            return null;
        }
        if (path.startsWith("http:") || path.startsWith("https:")) {
            return path;
        }
        if (path.indexOf(47) == -1 && path.indexOf(92) == -1) {
            return path;
        }
        path = FileUtil.normalizePath(path);
        String scheme = null;
        if (FileUtil.hasScheme(path)) {
            int pos = path.indexOf(58);
            scheme = path.substring(0, pos);
            path = path.substring(pos + 1);
        }
        boolean endsWithSlash = path.endsWith("/") || path.endsWith("\\");
        int cntSlashsAtStart = 0;
        if (path.startsWith("/") || path.startsWith("\\")) {
            ++cntSlashsAtStart;
            if (FileUtil.isWindows() && path.length() > 1 && (path.charAt(1) == '/' || path.charAt(1) == '\\')) {
                ++cntSlashsAtStart;
            }
        }
        ArrayDeque<String> stack = new ArrayDeque<String>();
        String separatorRegex = "[\\\\/]";
        for (String part : parts = path.split(separatorRegex)) {
            if (part.equals("..") && !stack.isEmpty() && !"..".equals(stack.peek())) {
                stack.pop();
                continue;
            }
            if (part.equals(".") || part.isEmpty()) continue;
            stack.push(part);
        }
        StringBuilder sb = new StringBuilder(256);
        if (scheme != null) {
            sb.append(scheme);
            sb.append(":");
        }
        sb.append(String.valueOf(separator).repeat(cntSlashsAtStart));
        Iterator it = stack.descendingIterator();
        while (it.hasNext()) {
            sb.append((String)it.next());
            if (!it.hasNext()) continue;
            sb.append(separator);
        }
        if (endsWithSlash && !stack.isEmpty()) {
            sb.append(separator);
        }
        return sb.toString();
    }

    public static void removeDir(File d) {
        String[] list = d.list();
        if (list == null) {
            list = new String[]{};
        }
        for (String s : list) {
            File f = new File(d, s);
            if (f.isDirectory()) {
                FileUtil.removeDir(f);
                continue;
            }
            FileUtil.delete(f);
        }
        FileUtil.delete(d);
    }

    private static void delete(File f) {
        try {
            Files.delete(f.toPath());
        }
        catch (IOException e) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException ex) {
                LOG.info("Interrupted while trying to delete file {}", (Object)f, (Object)e);
                Thread.currentThread().interrupt();
            }
            try {
                Files.delete(f.toPath());
            }
            catch (IOException ex) {
                f.deleteOnExit();
            }
        }
    }

    public static boolean renameFile(File from, File to, boolean copyAndDeleteOnRenameFail) throws IOException {
        return FileUtil.renameFile(from.toPath(), to.toPath(), copyAndDeleteOnRenameFail);
    }

    public static boolean renameFile(Path from, Path to, boolean copyAndDeleteOnRenameFail) throws IOException {
        int count;
        if (!Files.exists(from, new LinkOption[0])) {
            return false;
        }
        boolean renamed = false;
        for (count = 0; !renamed && count < 3; ++count) {
            if (LOG.isDebugEnabled() && count > 0) {
                LOG.debug("Retrying attempt {} to rename file from: {} to: {}", new Object[]{count, from, to});
            }
            try {
                Files.move(from, to, StandardCopyOption.ATOMIC_MOVE);
                renamed = true;
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (renamed || count <= 0) continue;
            try {
                Thread.sleep(1000L);
                continue;
            }
            catch (InterruptedException e) {
                LOG.info("Interrupted while trying to rename file from {} to {}", new Object[]{from, to, e});
                Thread.currentThread().interrupt();
            }
        }
        if (!renamed && copyAndDeleteOnRenameFail) {
            LOG.debug("Cannot rename file from: {} to: {}, will now use a copy/delete approach instead", (Object)from, (Object)to);
            renamed = FileUtil.renameFileUsingCopy(from, to);
        }
        if (LOG.isDebugEnabled() && count > 0) {
            LOG.debug("Tried {} to rename file: {} to: {} with result: {}", new Object[]{count, from, to, renamed});
        }
        return renamed;
    }

    public static boolean renameFileUsingCopy(File from, File to) throws IOException {
        return FileUtil.renameFileUsingCopy(from.toPath(), to.toPath());
    }

    public static boolean renameFileUsingCopy(Path from, Path to) throws IOException {
        if (!Files.exists(from, new LinkOption[0])) {
            return false;
        }
        LOG.debug("Rename file '{}' to '{}' using copy/delete strategy.", (Object)from, (Object)to);
        FileUtil.copyFile(from, to);
        if (!FileUtil.deleteFile(from)) {
            throw new IOException("Renaming file from '" + from + "' to '" + to + "' failed: Cannot delete file '" + from + "' after copy succeeded");
        }
        return true;
    }

    public static void copyFile(File from, File to) throws IOException {
        FileUtil.copyFile(from.toPath(), to.toPath());
    }

    public static void copyFile(Path from, Path to) throws IOException {
        Files.copy(from, to, StandardCopyOption.REPLACE_EXISTING);
    }

    public static boolean deleteFile(File file) {
        return FileUtil.deleteFile(file.toPath());
    }

    public static boolean deleteFile(Path file) {
        int count;
        if (!Files.exists(file, new LinkOption[0])) {
            return false;
        }
        boolean deleted = false;
        for (count = 0; !deleted && count < 3; ++count) {
            LOG.debug("Retrying attempt {} to delete file: {}", (Object)count, (Object)file);
            try {
                Files.delete(file);
                deleted = true;
                continue;
            }
            catch (IOException e) {
                if (count <= 0) continue;
                try {
                    Thread.sleep(1000L);
                    continue;
                }
                catch (InterruptedException ie) {
                    LOG.info("Interrupted while trying to delete file {}", (Object)file, (Object)e);
                    Thread.currentThread().interrupt();
                }
            }
        }
        if (LOG.isDebugEnabled() && count > 0) {
            LOG.debug("Tried {} to delete file: {} with result: {}", new Object[]{count, file, deleted});
        }
        return deleted;
    }

    public static boolean isAbsolute(File file) {
        String path;
        if (FileUtil.isWindows() && (path = file.getPath()).startsWith(File.separator)) {
            return true;
        }
        return file.isAbsolute();
    }

    public static boolean createNewFile(File file) throws IOException {
        if (file.exists()) {
            return false;
        }
        try {
            return file.createNewFile();
        }
        catch (IOException e) {
            if (file.exists()) {
                return true;
            }
            throw e;
        }
    }

    public static boolean setPosixFilePermissions(Path path, String permissions) throws IOException {
        try {
            Files.setPosixFilePermissions(path, PosixFilePermissions.fromString(permissions));
            return true;
        }
        catch (UnsupportedOperationException unsupportedOperationException) {
            return false;
        }
    }

    private static boolean hasScheme(String uri) {
        if (uri == null) {
            return false;
        }
        return uri.startsWith("file:") || uri.startsWith("classpath:") || uri.startsWith("http:");
    }
}

