/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.transcribe.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about a custom language model, including:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The base model name
 * </p>
 * </li>
 * <li>
 * <p>
 * When the model was created
 * </p>
 * </li>
 * <li>
 * <p>
 * The location of the files used to train the model
 * </p>
 * </li>
 * <li>
 * <p>
 * When the model was last modified
 * </p>
 * </li>
 * <li>
 * <p>
 * The name you chose for the model
 * </p>
 * </li>
 * <li>
 * <p>
 * The model's language
 * </p>
 * </li>
 * <li>
 * <p>
 * The model's processing state
 * </p>
 * </li>
 * <li>
 * <p>
 * Any available upgrades for the base model
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LanguageModel implements SdkPojo, Serializable, ToCopyableBuilder<LanguageModel.Builder, LanguageModel> {
    private static final SdkField<String> MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelName").getter(getter(LanguageModel::modelName)).setter(setter(Builder::modelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelName").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime").getter(getter(LanguageModel::createTime)).setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(LanguageModel::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(LanguageModel::languageCodeAsString)).setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<String> BASE_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaseModelName").getter(getter(LanguageModel::baseModelNameAsString))
            .setter(setter(Builder::baseModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseModelName").build()).build();

    private static final SdkField<String> MODEL_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelStatus").getter(getter(LanguageModel::modelStatusAsString)).setter(setter(Builder::modelStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelStatus").build()).build();

    private static final SdkField<Boolean> UPGRADE_AVAILABILITY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("UpgradeAvailability").getter(getter(LanguageModel::upgradeAvailability))
            .setter(setter(Builder::upgradeAvailability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeAvailability").build())
            .build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(LanguageModel::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<InputDataConfig> INPUT_DATA_CONFIG_FIELD = SdkField
            .<InputDataConfig> builder(MarshallingType.SDK_POJO).memberName("InputDataConfig")
            .getter(getter(LanguageModel::inputDataConfig)).setter(setter(Builder::inputDataConfig))
            .constructor(InputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MODEL_NAME_FIELD,
            CREATE_TIME_FIELD, LAST_MODIFIED_TIME_FIELD, LANGUAGE_CODE_FIELD, BASE_MODEL_NAME_FIELD, MODEL_STATUS_FIELD,
            UPGRADE_AVAILABILITY_FIELD, FAILURE_REASON_FIELD, INPUT_DATA_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String modelName;

    private final Instant createTime;

    private final Instant lastModifiedTime;

    private final String languageCode;

    private final String baseModelName;

    private final String modelStatus;

    private final Boolean upgradeAvailability;

    private final String failureReason;

    private final InputDataConfig inputDataConfig;

    private LanguageModel(BuilderImpl builder) {
        this.modelName = builder.modelName;
        this.createTime = builder.createTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.languageCode = builder.languageCode;
        this.baseModelName = builder.baseModelName;
        this.modelStatus = builder.modelStatus;
        this.upgradeAvailability = builder.upgradeAvailability;
        this.failureReason = builder.failureReason;
        this.inputDataConfig = builder.inputDataConfig;
    }

    /**
     * <p>
     * A unique name, chosen by you, for your custom language model.
     * </p>
     * <p>
     * This name is case sensitive, cannot contain spaces, and must be unique within an Amazon Web Services account.
     * </p>
     * 
     * @return A unique name, chosen by you, for your custom language model.</p>
     *         <p>
     *         This name is case sensitive, cannot contain spaces, and must be unique within an Amazon Web Services
     *         account.
     */
    public final String modelName() {
        return modelName;
    }

    /**
     * <p>
     * The date and time the specified custom language model was created.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified custom language model was created.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The date and time the specified custom language model was last modified.
     * </p>
     * <p>
     * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     * <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
     * </p>
     * 
     * @return The date and time the specified custom language model was last modified.</p>
     *         <p>
     *         Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
     *         <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The language code used to create your custom language model. Each custom language model must contain terms in
     * only one language, and the language you select for your custom language model must match the language of your
     * training and tuning data.
     * </p>
     * <p>
     * For a list of supported languages and their associated language codes, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * Note that U.S. English (<code>en-US</code>) is the only language supported with Amazon Transcribe Medical.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link CLMLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your custom language model. Each custom language model must contain
     *         terms in only one language, and the language you select for your custom language model must match the
     *         language of your training and tuning data.</p>
     *         <p>
     *         For a list of supported languages and their associated language codes, refer to the <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
     *         table. Note that U.S. English (<code>en-US</code>) is the only language supported with Amazon Transcribe
     *         Medical.
     * @see CLMLanguageCode
     */
    public final CLMLanguageCode languageCode() {
        return CLMLanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code used to create your custom language model. Each custom language model must contain terms in
     * only one language, and the language you select for your custom language model must match the language of your
     * training and tuning data.
     * </p>
     * <p>
     * For a list of supported languages and their associated language codes, refer to the <a
     * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a> table.
     * Note that U.S. English (<code>en-US</code>) is the only language supported with Amazon Transcribe Medical.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link CLMLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code used to create your custom language model. Each custom language model must contain
     *         terms in only one language, and the language you select for your custom language model must match the
     *         language of your training and tuning data.</p>
     *         <p>
     *         For a list of supported languages and their associated language codes, refer to the <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
     *         table. Note that U.S. English (<code>en-US</code>) is the only language supported with Amazon Transcribe
     *         Medical.
     * @see CLMLanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The Amazon Transcribe standard language model, or base model, used to create your custom language model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #baseModelName}
     * will return {@link BaseModelName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #baseModelNameAsString}.
     * </p>
     * 
     * @return The Amazon Transcribe standard language model, or base model, used to create your custom language model.
     * @see BaseModelName
     */
    public final BaseModelName baseModelName() {
        return BaseModelName.fromValue(baseModelName);
    }

    /**
     * <p>
     * The Amazon Transcribe standard language model, or base model, used to create your custom language model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #baseModelName}
     * will return {@link BaseModelName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #baseModelNameAsString}.
     * </p>
     * 
     * @return The Amazon Transcribe standard language model, or base model, used to create your custom language model.
     * @see BaseModelName
     */
    public final String baseModelNameAsString() {
        return baseModelName;
    }

    /**
     * <p>
     * The status of the specified custom language model. When the status displays as <code>COMPLETED</code> the model
     * is ready for use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #modelStatus} will
     * return {@link ModelStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modelStatusAsString}.
     * </p>
     * 
     * @return The status of the specified custom language model. When the status displays as <code>COMPLETED</code> the
     *         model is ready for use.
     * @see ModelStatus
     */
    public final ModelStatus modelStatus() {
        return ModelStatus.fromValue(modelStatus);
    }

    /**
     * <p>
     * The status of the specified custom language model. When the status displays as <code>COMPLETED</code> the model
     * is ready for use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #modelStatus} will
     * return {@link ModelStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modelStatusAsString}.
     * </p>
     * 
     * @return The status of the specified custom language model. When the status displays as <code>COMPLETED</code> the
     *         model is ready for use.
     * @see ModelStatus
     */
    public final String modelStatusAsString() {
        return modelStatus;
    }

    /**
     * <p>
     * Shows if a more current base model is available for use with the specified custom language model.
     * </p>
     * <p>
     * If <code>false</code>, your custom language model is using the most up-to-date base model.
     * </p>
     * <p>
     * If <code>true</code>, there is a newer base model available than the one your language model is using.
     * </p>
     * <p>
     * Note that to update a base model, you must recreate the custom language model using the new base model. Base
     * model upgrades for existing custom language models are not supported.
     * </p>
     * 
     * @return Shows if a more current base model is available for use with the specified custom language model.</p>
     *         <p>
     *         If <code>false</code>, your custom language model is using the most up-to-date base model.
     *         </p>
     *         <p>
     *         If <code>true</code>, there is a newer base model available than the one your language model is using.
     *         </p>
     *         <p>
     *         Note that to update a base model, you must recreate the custom language model using the new base model.
     *         Base model upgrades for existing custom language models are not supported.
     */
    public final Boolean upgradeAvailability() {
        return upgradeAvailability;
    }

    /**
     * <p>
     * If <code>ModelStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information about why the
     * custom language model request failed. See also: <a
     * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
     * </p>
     * 
     * @return If <code>ModelStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information about
     *         why the custom language model request failed. See also: <a
     *         href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The Amazon S3 location of the input files used to train and tune your custom language model, in addition to the
     * data access role ARN (Amazon Resource Name) that has permissions to access these data.
     * </p>
     * 
     * @return The Amazon S3 location of the input files used to train and tune your custom language model, in addition
     *         to the data access role ARN (Amazon Resource Name) that has permissions to access these data.
     */
    public final InputDataConfig inputDataConfig() {
        return inputDataConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(modelName());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(baseModelNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modelStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(upgradeAvailability());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LanguageModel)) {
            return false;
        }
        LanguageModel other = (LanguageModel) obj;
        return Objects.equals(modelName(), other.modelName()) && Objects.equals(createTime(), other.createTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(baseModelNameAsString(), other.baseModelNameAsString())
                && Objects.equals(modelStatusAsString(), other.modelStatusAsString())
                && Objects.equals(upgradeAvailability(), other.upgradeAvailability())
                && Objects.equals(failureReason(), other.failureReason())
                && Objects.equals(inputDataConfig(), other.inputDataConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LanguageModel").add("ModelName", modelName()).add("CreateTime", createTime())
                .add("LastModifiedTime", lastModifiedTime()).add("LanguageCode", languageCodeAsString())
                .add("BaseModelName", baseModelNameAsString()).add("ModelStatus", modelStatusAsString())
                .add("UpgradeAvailability", upgradeAvailability()).add("FailureReason", failureReason())
                .add("InputDataConfig", inputDataConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ModelName":
            return Optional.ofNullable(clazz.cast(modelName()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "BaseModelName":
            return Optional.ofNullable(clazz.cast(baseModelNameAsString()));
        case "ModelStatus":
            return Optional.ofNullable(clazz.cast(modelStatusAsString()));
        case "UpgradeAvailability":
            return Optional.ofNullable(clazz.cast(upgradeAvailability()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "InputDataConfig":
            return Optional.ofNullable(clazz.cast(inputDataConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LanguageModel, T> g) {
        return obj -> g.apply((LanguageModel) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LanguageModel> {
        /**
         * <p>
         * A unique name, chosen by you, for your custom language model.
         * </p>
         * <p>
         * This name is case sensitive, cannot contain spaces, and must be unique within an Amazon Web Services account.
         * </p>
         * 
         * @param modelName
         *        A unique name, chosen by you, for your custom language model.</p>
         *        <p>
         *        This name is case sensitive, cannot contain spaces, and must be unique within an Amazon Web Services
         *        account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelName(String modelName);

        /**
         * <p>
         * The date and time the specified custom language model was created.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param createTime
         *        The date and time the specified custom language model was created.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The date and time the specified custom language model was last modified.
         * </p>
         * <p>
         * Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         * <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time the specified custom language model was last modified.</p>
         *        <p>
         *        Timestamps are in the format <code>YYYY-MM-DD'T'HH:MM:SS.SSSSSS-UTC</code>. For example,
         *        <code>2022-05-04T12:32:58.761000-07:00</code> represents 12:32 PM UTC-7 on May 4, 2022.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The language code used to create your custom language model. Each custom language model must contain terms in
         * only one language, and the language you select for your custom language model must match the language of your
         * training and tuning data.
         * </p>
         * <p>
         * For a list of supported languages and their associated language codes, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table. Note that U.S. English (<code>en-US</code>) is the only language supported with Amazon Transcribe
         * Medical.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your custom language model. Each custom language model must contain
         *        terms in only one language, and the language you select for your custom language model must match the
         *        language of your training and tuning data.</p>
         *        <p>
         *        For a list of supported languages and their associated language codes, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table. Note that U.S. English (<code>en-US</code>) is the only language supported with
         *        Amazon Transcribe Medical.
         * @see CLMLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CLMLanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code used to create your custom language model. Each custom language model must contain terms in
         * only one language, and the language you select for your custom language model must match the language of your
         * training and tuning data.
         * </p>
         * <p>
         * For a list of supported languages and their associated language codes, refer to the <a
         * href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported languages</a>
         * table. Note that U.S. English (<code>en-US</code>) is the only language supported with Amazon Transcribe
         * Medical.
         * </p>
         * 
         * @param languageCode
         *        The language code used to create your custom language model. Each custom language model must contain
         *        terms in only one language, and the language you select for your custom language model must match the
         *        language of your training and tuning data.</p>
         *        <p>
         *        For a list of supported languages and their associated language codes, refer to the <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/dg/supported-languages.html">Supported
         *        languages</a> table. Note that U.S. English (<code>en-US</code>) is the only language supported with
         *        Amazon Transcribe Medical.
         * @see CLMLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CLMLanguageCode
         */
        Builder languageCode(CLMLanguageCode languageCode);

        /**
         * <p>
         * The Amazon Transcribe standard language model, or base model, used to create your custom language model.
         * </p>
         * 
         * @param baseModelName
         *        The Amazon Transcribe standard language model, or base model, used to create your custom language
         *        model.
         * @see BaseModelName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BaseModelName
         */
        Builder baseModelName(String baseModelName);

        /**
         * <p>
         * The Amazon Transcribe standard language model, or base model, used to create your custom language model.
         * </p>
         * 
         * @param baseModelName
         *        The Amazon Transcribe standard language model, or base model, used to create your custom language
         *        model.
         * @see BaseModelName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BaseModelName
         */
        Builder baseModelName(BaseModelName baseModelName);

        /**
         * <p>
         * The status of the specified custom language model. When the status displays as <code>COMPLETED</code> the
         * model is ready for use.
         * </p>
         * 
         * @param modelStatus
         *        The status of the specified custom language model. When the status displays as <code>COMPLETED</code>
         *        the model is ready for use.
         * @see ModelStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelStatus
         */
        Builder modelStatus(String modelStatus);

        /**
         * <p>
         * The status of the specified custom language model. When the status displays as <code>COMPLETED</code> the
         * model is ready for use.
         * </p>
         * 
         * @param modelStatus
         *        The status of the specified custom language model. When the status displays as <code>COMPLETED</code>
         *        the model is ready for use.
         * @see ModelStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelStatus
         */
        Builder modelStatus(ModelStatus modelStatus);

        /**
         * <p>
         * Shows if a more current base model is available for use with the specified custom language model.
         * </p>
         * <p>
         * If <code>false</code>, your custom language model is using the most up-to-date base model.
         * </p>
         * <p>
         * If <code>true</code>, there is a newer base model available than the one your language model is using.
         * </p>
         * <p>
         * Note that to update a base model, you must recreate the custom language model using the new base model. Base
         * model upgrades for existing custom language models are not supported.
         * </p>
         * 
         * @param upgradeAvailability
         *        Shows if a more current base model is available for use with the specified custom language model.</p>
         *        <p>
         *        If <code>false</code>, your custom language model is using the most up-to-date base model.
         *        </p>
         *        <p>
         *        If <code>true</code>, there is a newer base model available than the one your language model is using.
         *        </p>
         *        <p>
         *        Note that to update a base model, you must recreate the custom language model using the new base
         *        model. Base model upgrades for existing custom language models are not supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upgradeAvailability(Boolean upgradeAvailability);

        /**
         * <p>
         * If <code>ModelStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information about why
         * the custom language model request failed. See also: <a
         * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
         * </p>
         * 
         * @param failureReason
         *        If <code>ModelStatus</code> is <code>FAILED</code>, <code>FailureReason</code> contains information
         *        about why the custom language model request failed. See also: <a
         *        href="https://docs.aws.amazon.com/transcribe/latest/APIReference/CommonErrors.html">Common Errors</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The Amazon S3 location of the input files used to train and tune your custom language model, in addition to
         * the data access role ARN (Amazon Resource Name) that has permissions to access these data.
         * </p>
         * 
         * @param inputDataConfig
         *        The Amazon S3 location of the input files used to train and tune your custom language model, in
         *        addition to the data access role ARN (Amazon Resource Name) that has permissions to access these data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(InputDataConfig inputDataConfig);

        /**
         * <p>
         * The Amazon S3 location of the input files used to train and tune your custom language model, in addition to
         * the data access role ARN (Amazon Resource Name) that has permissions to access these data.
         * </p>
         * This is a convenience method that creates an instance of the {@link InputDataConfig.Builder} avoiding the
         * need to create one manually via {@link InputDataConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InputDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #inputDataConfig(InputDataConfig)}.
         * 
         * @param inputDataConfig
         *        a consumer that will call methods on {@link InputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDataConfig(InputDataConfig)
         */
        default Builder inputDataConfig(Consumer<InputDataConfig.Builder> inputDataConfig) {
            return inputDataConfig(InputDataConfig.builder().applyMutation(inputDataConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String modelName;

        private Instant createTime;

        private Instant lastModifiedTime;

        private String languageCode;

        private String baseModelName;

        private String modelStatus;

        private Boolean upgradeAvailability;

        private String failureReason;

        private InputDataConfig inputDataConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(LanguageModel model) {
            modelName(model.modelName);
            createTime(model.createTime);
            lastModifiedTime(model.lastModifiedTime);
            languageCode(model.languageCode);
            baseModelName(model.baseModelName);
            modelStatus(model.modelStatus);
            upgradeAvailability(model.upgradeAvailability);
            failureReason(model.failureReason);
            inputDataConfig(model.inputDataConfig);
        }

        public final String getModelName() {
            return modelName;
        }

        public final void setModelName(String modelName) {
            this.modelName = modelName;
        }

        @Override
        public final Builder modelName(String modelName) {
            this.modelName = modelName;
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(CLMLanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final String getBaseModelName() {
            return baseModelName;
        }

        public final void setBaseModelName(String baseModelName) {
            this.baseModelName = baseModelName;
        }

        @Override
        public final Builder baseModelName(String baseModelName) {
            this.baseModelName = baseModelName;
            return this;
        }

        @Override
        public final Builder baseModelName(BaseModelName baseModelName) {
            this.baseModelName(baseModelName == null ? null : baseModelName.toString());
            return this;
        }

        public final String getModelStatus() {
            return modelStatus;
        }

        public final void setModelStatus(String modelStatus) {
            this.modelStatus = modelStatus;
        }

        @Override
        public final Builder modelStatus(String modelStatus) {
            this.modelStatus = modelStatus;
            return this;
        }

        @Override
        public final Builder modelStatus(ModelStatus modelStatus) {
            this.modelStatus(modelStatus == null ? null : modelStatus.toString());
            return this;
        }

        public final Boolean getUpgradeAvailability() {
            return upgradeAvailability;
        }

        public final void setUpgradeAvailability(Boolean upgradeAvailability) {
            this.upgradeAvailability = upgradeAvailability;
        }

        @Override
        public final Builder upgradeAvailability(Boolean upgradeAvailability) {
            this.upgradeAvailability = upgradeAvailability;
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final InputDataConfig.Builder getInputDataConfig() {
            return inputDataConfig != null ? inputDataConfig.toBuilder() : null;
        }

        public final void setInputDataConfig(InputDataConfig.BuilderImpl inputDataConfig) {
            this.inputDataConfig = inputDataConfig != null ? inputDataConfig.build() : null;
        }

        @Override
        public final Builder inputDataConfig(InputDataConfig inputDataConfig) {
            this.inputDataConfig = inputDataConfig;
            return this;
        }

        @Override
        public LanguageModel build() {
            return new LanguageModel(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
