/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.tnb;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.tnb.model.AccessDeniedException;
import software.amazon.awssdk.services.tnb.model.CancelSolNetworkOperationRequest;
import software.amazon.awssdk.services.tnb.model.CancelSolNetworkOperationResponse;
import software.amazon.awssdk.services.tnb.model.CreateSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.CreateSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.CreateSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.DeleteSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.DeleteSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.DeleteSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionInstanceRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionInstanceResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageDescriptorRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageDescriptorResponse;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.GetSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkOperationRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkOperationResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageDescriptorRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageDescriptorResponse;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.GetSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.InstantiateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.InstantiateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.InternalServerException;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesResponse;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesResponse;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesResponse;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsRequest;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsResponse;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesRequest;
import software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesResponse;
import software.amazon.awssdk.services.tnb.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.tnb.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.tnb.model.PutSolFunctionPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.PutSolFunctionPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.PutSolNetworkPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.PutSolNetworkPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.tnb.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.tnb.model.TagResourceRequest;
import software.amazon.awssdk.services.tnb.model.TagResourceResponse;
import software.amazon.awssdk.services.tnb.model.TerminateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.TerminateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.ThrottlingException;
import software.amazon.awssdk.services.tnb.model.TnbException;
import software.amazon.awssdk.services.tnb.model.TnbRequest;
import software.amazon.awssdk.services.tnb.model.UntagResourceRequest;
import software.amazon.awssdk.services.tnb.model.UntagResourceResponse;
import software.amazon.awssdk.services.tnb.model.UpdateSolFunctionPackageRequest;
import software.amazon.awssdk.services.tnb.model.UpdateSolFunctionPackageResponse;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkInstanceRequest;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkInstanceResponse;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkPackageRequest;
import software.amazon.awssdk.services.tnb.model.UpdateSolNetworkPackageResponse;
import software.amazon.awssdk.services.tnb.model.ValidateSolFunctionPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.ValidateSolFunctionPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.ValidateSolNetworkPackageContentRequest;
import software.amazon.awssdk.services.tnb.model.ValidateSolNetworkPackageContentResponse;
import software.amazon.awssdk.services.tnb.model.ValidationException;
import software.amazon.awssdk.services.tnb.paginators.ListSolFunctionInstancesIterable;
import software.amazon.awssdk.services.tnb.paginators.ListSolFunctionPackagesIterable;
import software.amazon.awssdk.services.tnb.paginators.ListSolNetworkInstancesIterable;
import software.amazon.awssdk.services.tnb.paginators.ListSolNetworkOperationsIterable;
import software.amazon.awssdk.services.tnb.paginators.ListSolNetworkPackagesIterable;
import software.amazon.awssdk.services.tnb.transform.CancelSolNetworkOperationRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.CreateSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.CreateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.CreateSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.DeleteSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.DeleteSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.DeleteSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionPackageDescriptorRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkOperationRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkPackageDescriptorRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.GetSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.InstantiateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolFunctionInstancesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolFunctionPackagesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolNetworkInstancesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolNetworkOperationsRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListSolNetworkPackagesRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.PutSolFunctionPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.PutSolNetworkPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.TerminateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UpdateSolFunctionPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UpdateSolNetworkInstanceRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.UpdateSolNetworkPackageRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ValidateSolFunctionPackageContentRequestMarshaller;
import software.amazon.awssdk.services.tnb.transform.ValidateSolNetworkPackageContentRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link TnbClient}.
 *
 * @see TnbClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultTnbClient implements TnbClient {
    private static final Logger log = Logger.loggerFor(DefaultTnbClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final TnbServiceClientConfiguration serviceClientConfiguration;

    protected DefaultTnbClient(TnbServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels a network operation.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     *
     * @param cancelSolNetworkOperationRequest
     * @return Result of the CancelSolNetworkOperation operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.CancelSolNetworkOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CancelSolNetworkOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelSolNetworkOperationResponse cancelSolNetworkOperation(
            CancelSolNetworkOperationRequest cancelSolNetworkOperationRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelSolNetworkOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelSolNetworkOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelSolNetworkOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelSolNetworkOperation");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelSolNetworkOperationRequest, CancelSolNetworkOperationResponse>()
                            .withOperationName("CancelSolNetworkOperation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(cancelSolNetworkOperationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelSolNetworkOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network. For more information, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/ug/function-packages.html">Function packages</a> in the <i>Amazon
     * Web Services Telco Network Builder User Guide</i>.
     * </p>
     * <p>
     * Creating a function package is the first step for creating a network in AWS TNB. This request creates an empty
     * container with an ID. The next step is to upload the actual CSAR zip file into that empty container. To upload
     * function package content, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolFunctionPackageContent.html"
     * >PutSolFunctionPackageContent</a>.
     * </p>
     *
     * @param createSolFunctionPackageRequest
     * @return Result of the CreateSolFunctionPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ServiceQuotaExceededException
     *         Service quotas have been exceeded.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.CreateSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CreateSolFunctionPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateSolFunctionPackageResponse createSolFunctionPackage(
            CreateSolFunctionPackageRequest createSolFunctionPackageRequest) throws InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSolFunctionPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSolFunctionPackage");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSolFunctionPackageRequest, CreateSolFunctionPackageResponse>()
                            .withOperationName("CreateSolFunctionPackage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createSolFunctionPackageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSolFunctionPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed. Creating a network instance is the
     * third step after creating a network package. For more information about network instances, <a
     * href="https://docs.aws.amazon.com/tnb/latest/ug/network-instances.html">Network instances</a> in the <i>Amazon
     * Web Services Telco Network Builder User Guide</i>.
     * </p>
     * <p>
     * Once you create a network instance, you can instantiate it. To instantiate a network, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_InstantiateSolNetworkInstance.html"
     * >InstantiateSolNetworkInstance</a>.
     * </p>
     *
     * @param createSolNetworkInstanceRequest
     * @return Result of the CreateSolNetworkInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ServiceQuotaExceededException
     *         Service quotas have been exceeded.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.CreateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CreateSolNetworkInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateSolNetworkInstanceResponse createSolNetworkInstance(
            CreateSolNetworkInstanceRequest createSolNetworkInstanceRequest) throws InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSolNetworkInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSolNetworkInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSolNetworkInstanceRequest, CreateSolNetworkInstanceResponse>()
                            .withOperationName("CreateSolNetworkInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createSolNetworkInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSolNetworkInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on. For more information, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/ug/network-instances.html">Network instances</a> in the <i>Amazon
     * Web Services Telco Network Builder User Guide</i>.
     * </p>
     * <p>
     * A network package consists of a network service descriptor (NSD) file (required) and any additional files
     * (optional), such as scripts specific to your needs. For example, if you have multiple function packages in your
     * network package, you can use the NSD to define which network functions should run in certain VPCs, subnets, or
     * EKS clusters.
     * </p>
     * <p>
     * This request creates an empty network package container with an ID. Once you create a network package, you can
     * upload the network package content using <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolNetworkPackageContent.html"
     * >PutSolNetworkPackageContent</a>.
     * </p>
     *
     * @param createSolNetworkPackageRequest
     * @return Result of the CreateSolNetworkPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ServiceQuotaExceededException
     *         Service quotas have been exceeded.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.CreateSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/CreateSolNetworkPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateSolNetworkPackageResponse createSolNetworkPackage(CreateSolNetworkPackageRequest createSolNetworkPackageRequest)
            throws InternalServerException, ServiceQuotaExceededException, ThrottlingException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSolNetworkPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSolNetworkPackage");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSolNetworkPackageRequest, CreateSolNetworkPackageResponse>()
                            .withOperationName("CreateSolNetworkPackage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createSolNetworkPackageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSolNetworkPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     * <p>
     * To delete a function package, the package must be in a disabled state. To disable a function package, see <a
     * href=
     * "https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolFunctionPackage.html">UpdateSolFunctionPackage
     * </a>.
     * </p>
     *
     * @param deleteSolFunctionPackageRequest
     * @return Result of the DeleteSolFunctionPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.DeleteSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/DeleteSolFunctionPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteSolFunctionPackageResponse deleteSolFunctionPackage(
            DeleteSolFunctionPackageRequest deleteSolFunctionPackageRequest) throws InternalServerException, ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSolFunctionPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSolFunctionPackage");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSolFunctionPackageRequest, DeleteSolFunctionPackageResponse>()
                            .withOperationName("DeleteSolFunctionPackage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteSolFunctionPackageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSolFunctionPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * To delete a network instance, the instance must be in a stopped or terminated state. To terminate a network
     * instance, see <a href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_TerminateSolNetworkInstance.html">
     * TerminateSolNetworkInstance</a>.
     * </p>
     *
     * @param deleteSolNetworkInstanceRequest
     * @return Result of the DeleteSolNetworkInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.DeleteSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/DeleteSolNetworkInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteSolNetworkInstanceResponse deleteSolNetworkInstance(
            DeleteSolNetworkInstanceRequest deleteSolNetworkInstanceRequest) throws InternalServerException, ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSolNetworkInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSolNetworkInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSolNetworkInstanceRequest, DeleteSolNetworkInstanceResponse>()
                            .withOperationName("DeleteSolNetworkInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteSolNetworkInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSolNetworkInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     * <p>
     * To delete a network package, the package must be in a disable state. To disable a network package, see <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolNetworkPackage.html"
     * >UpdateSolNetworkPackage</a>.
     * </p>
     *
     * @param deleteSolNetworkPackageRequest
     * @return Result of the DeleteSolNetworkPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.DeleteSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/DeleteSolNetworkPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteSolNetworkPackageResponse deleteSolNetworkPackage(DeleteSolNetworkPackageRequest deleteSolNetworkPackageRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSolNetworkPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSolNetworkPackage");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSolNetworkPackageRequest, DeleteSolNetworkPackageResponse>()
                            .withOperationName("DeleteSolNetworkPackage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteSolNetworkPackageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSolNetworkPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the details of a network function instance, including the instantation state and metadata from the function
     * package descriptor in the network function package.
     * </p>
     * <p>
     * A network function instance is a function in a function package .
     * </p>
     *
     * @param getSolFunctionInstanceRequest
     * @return Result of the GetSolFunctionInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolFunctionInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSolFunctionInstanceResponse getSolFunctionInstance(GetSolFunctionInstanceRequest getSolFunctionInstanceRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSolFunctionInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolFunctionInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolFunctionInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionInstanceRequest, GetSolFunctionInstanceResponse>()
                            .withOperationName("GetSolFunctionInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSolFunctionInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSolFunctionInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the details of an individual function package, such as the operational state and whether the package is in
     * use.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network..
     * </p>
     *
     * @param getSolFunctionPackageRequest
     * @return Result of the GetSolFunctionPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionPackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSolFunctionPackageResponse getSolFunctionPackage(GetSolFunctionPackageRequest getSolFunctionPackageRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolFunctionPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionPackage");

            return clientHandler.execute(new ClientExecutionParams<GetSolFunctionPackageRequest, GetSolFunctionPackageResponse>()
                    .withOperationName("GetSolFunctionPackage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSolFunctionPackageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSolFunctionPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the contents of a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param getSolFunctionPackageContentRequest
     * @return Result of the GetSolFunctionPackageContent operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolFunctionPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSolFunctionPackageContentResponse getSolFunctionPackageContent(
            GetSolFunctionPackageContentRequest getSolFunctionPackageContentRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetSolFunctionPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolFunctionPackageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolFunctionPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionPackageContent");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionPackageContentRequest, GetSolFunctionPackageContentResponse>()
                            .withOperationName("GetSolFunctionPackageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSolFunctionPackageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSolFunctionPackageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a function package descriptor in a function package.
     * </p>
     * <p>
     * A function package descriptor is a .yaml file in a function package that uses the TOSCA standard to describe how
     * the network function in the function package should run on your network.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param getSolFunctionPackageDescriptorRequest
     * @return Result of the GetSolFunctionPackageDescriptor operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolFunctionPackageDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolFunctionPackageDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSolFunctionPackageDescriptorResponse getSolFunctionPackageDescriptor(
            GetSolFunctionPackageDescriptorRequest getSolFunctionPackageDescriptorRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetSolFunctionPackageDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolFunctionPackageDescriptorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getSolFunctionPackageDescriptorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolFunctionPackageDescriptor");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSolFunctionPackageDescriptorRequest, GetSolFunctionPackageDescriptorResponse>()
                            .withOperationName("GetSolFunctionPackageDescriptor").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSolFunctionPackageDescriptorRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSolFunctionPackageDescriptorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the details of the network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     *
     * @param getSolNetworkInstanceRequest
     * @return Result of the GetSolNetworkInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkInstance" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSolNetworkInstanceResponse getSolNetworkInstance(GetSolNetworkInstanceRequest getSolNetworkInstanceRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolNetworkInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkInstance");

            return clientHandler.execute(new ClientExecutionParams<GetSolNetworkInstanceRequest, GetSolNetworkInstanceResponse>()
                    .withOperationName("GetSolNetworkInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSolNetworkInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSolNetworkInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the details of a network operation, including the tasks involved in the network operation and the status of
     * the tasks.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     *
     * @param getSolNetworkOperationRequest
     * @return Result of the GetSolNetworkOperation operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolNetworkOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkOperation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSolNetworkOperationResponse getSolNetworkOperation(GetSolNetworkOperationRequest getSolNetworkOperationRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSolNetworkOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolNetworkOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkOperation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkOperationRequest, GetSolNetworkOperationResponse>()
                            .withOperationName("GetSolNetworkOperation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSolNetworkOperationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSolNetworkOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the details of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param getSolNetworkPackageRequest
     * @return Result of the GetSolNetworkPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkPackage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSolNetworkPackageResponse getSolNetworkPackage(GetSolNetworkPackageRequest getSolNetworkPackageRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolNetworkPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkPackage");

            return clientHandler.execute(new ClientExecutionParams<GetSolNetworkPackageRequest, GetSolNetworkPackageResponse>()
                    .withOperationName("GetSolNetworkPackage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSolNetworkPackageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSolNetworkPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the contents of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param getSolNetworkPackageContentRequest
     * @return Result of the GetSolNetworkPackageContent operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolNetworkPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSolNetworkPackageContentResponse getSolNetworkPackageContent(
            GetSolNetworkPackageContentRequest getSolNetworkPackageContentRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetSolNetworkPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolNetworkPackageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSolNetworkPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkPackageContent");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkPackageContentRequest, GetSolNetworkPackageContentResponse>()
                            .withOperationName("GetSolNetworkPackageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSolNetworkPackageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSolNetworkPackageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the content of the network service descriptor.
     * </p>
     * <p>
     * A network service descriptor is a .yaml file in a network package that uses the TOSCA standard to describe the
     * network functions you want to deploy and the Amazon Web Services infrastructure you want to deploy the network
     * functions on.
     * </p>
     *
     * @param getSolNetworkPackageDescriptorRequest
     * @return Result of the GetSolNetworkPackageDescriptor operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.GetSolNetworkPackageDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/GetSolNetworkPackageDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSolNetworkPackageDescriptorResponse getSolNetworkPackageDescriptor(
            GetSolNetworkPackageDescriptorRequest getSolNetworkPackageDescriptorRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetSolNetworkPackageDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSolNetworkPackageDescriptorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getSolNetworkPackageDescriptorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSolNetworkPackageDescriptor");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSolNetworkPackageDescriptorRequest, GetSolNetworkPackageDescriptorResponse>()
                            .withOperationName("GetSolNetworkPackageDescriptor").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSolNetworkPackageDescriptorRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSolNetworkPackageDescriptorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Instantiates a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * Before you can instantiate a network instance, you have to create a network instance. For more information, see
     * <a href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_CreateSolNetworkInstance.html">
     * CreateSolNetworkInstance</a>.
     * </p>
     *
     * @param instantiateSolNetworkInstanceRequest
     * @return Result of the InstantiateSolNetworkInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ServiceQuotaExceededException
     *         Service quotas have been exceeded.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.InstantiateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/InstantiateSolNetworkInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public InstantiateSolNetworkInstanceResponse instantiateSolNetworkInstance(
            InstantiateSolNetworkInstanceRequest instantiateSolNetworkInstanceRequest) throws InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InstantiateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InstantiateSolNetworkInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                instantiateSolNetworkInstanceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InstantiateSolNetworkInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<InstantiateSolNetworkInstanceRequest, InstantiateSolNetworkInstanceResponse>()
                            .withOperationName("InstantiateSolNetworkInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(instantiateSolNetworkInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new InstantiateSolNetworkInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists network function instances.
     * </p>
     * <p>
     * A network function instance is a function in a function package .
     * </p>
     *
     * @param listSolFunctionInstancesRequest
     * @return Result of the ListSolFunctionInstances operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolFunctionInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolFunctionInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolFunctionInstancesResponse listSolFunctionInstances(
            ListSolFunctionInstancesRequest listSolFunctionInstancesRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSolFunctionInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSolFunctionInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolFunctionInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolFunctionInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSolFunctionInstancesRequest, ListSolFunctionInstancesResponse>()
                            .withOperationName("ListSolFunctionInstances").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listSolFunctionInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSolFunctionInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists network function instances.
     * </p>
     * <p>
     * A network function instance is a function in a function package .
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSolFunctionInstances(software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolFunctionInstancesIterable responses = client.listSolFunctionInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.tnb.paginators.ListSolFunctionInstancesIterable responses = client
     *             .listSolFunctionInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolFunctionInstancesIterable responses = client.listSolFunctionInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSolFunctionInstances(software.amazon.awssdk.services.tnb.model.ListSolFunctionInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSolFunctionInstancesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolFunctionInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolFunctionInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolFunctionInstancesIterable listSolFunctionInstancesPaginator(
            ListSolFunctionInstancesRequest listSolFunctionInstancesRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        return new ListSolFunctionInstancesIterable(this, applyPaginatorUserAgent(listSolFunctionInstancesRequest));
    }

    /**
     * <p>
     * Lists information about function packages.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param listSolFunctionPackagesRequest
     * @return Result of the ListSolFunctionPackages operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolFunctionPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolFunctionPackages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolFunctionPackagesResponse listSolFunctionPackages(ListSolFunctionPackagesRequest listSolFunctionPackagesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSolFunctionPackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSolFunctionPackagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolFunctionPackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolFunctionPackages");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSolFunctionPackagesRequest, ListSolFunctionPackagesResponse>()
                            .withOperationName("ListSolFunctionPackages").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listSolFunctionPackagesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSolFunctionPackagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about function packages.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSolFunctionPackages(software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolFunctionPackagesIterable responses = client.listSolFunctionPackagesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.tnb.paginators.ListSolFunctionPackagesIterable responses = client
     *             .listSolFunctionPackagesPaginator(request);
     *     for (software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolFunctionPackagesIterable responses = client.listSolFunctionPackagesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSolFunctionPackages(software.amazon.awssdk.services.tnb.model.ListSolFunctionPackagesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSolFunctionPackagesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolFunctionPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolFunctionPackages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolFunctionPackagesIterable listSolFunctionPackagesPaginator(
            ListSolFunctionPackagesRequest listSolFunctionPackagesRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        return new ListSolFunctionPackagesIterable(this, applyPaginatorUserAgent(listSolFunctionPackagesRequest));
    }

    /**
     * <p>
     * Lists your network instances.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     *
     * @param listSolNetworkInstancesRequest
     * @return Result of the ListSolNetworkInstances operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolNetworkInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolNetworkInstancesResponse listSolNetworkInstances(ListSolNetworkInstancesRequest listSolNetworkInstancesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSolNetworkInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSolNetworkInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolNetworkInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolNetworkInstances");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSolNetworkInstancesRequest, ListSolNetworkInstancesResponse>()
                            .withOperationName("ListSolNetworkInstances").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listSolNetworkInstancesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSolNetworkInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your network instances.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSolNetworkInstances(software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolNetworkInstancesIterable responses = client.listSolNetworkInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.tnb.paginators.ListSolNetworkInstancesIterable responses = client
     *             .listSolNetworkInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolNetworkInstancesIterable responses = client.listSolNetworkInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSolNetworkInstances(software.amazon.awssdk.services.tnb.model.ListSolNetworkInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSolNetworkInstancesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolNetworkInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkInstances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolNetworkInstancesIterable listSolNetworkInstancesPaginator(
            ListSolNetworkInstancesRequest listSolNetworkInstancesRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        return new ListSolNetworkInstancesIterable(this, applyPaginatorUserAgent(listSolNetworkInstancesRequest));
    }

    /**
     * <p>
     * Lists details for a network operation, including when the operation started and the status of the operation.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     *
     * @param listSolNetworkOperationsRequest
     * @return Result of the ListSolNetworkOperations operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolNetworkOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolNetworkOperationsResponse listSolNetworkOperations(
            ListSolNetworkOperationsRequest listSolNetworkOperationsRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSolNetworkOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSolNetworkOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolNetworkOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolNetworkOperations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSolNetworkOperationsRequest, ListSolNetworkOperationsResponse>()
                            .withOperationName("ListSolNetworkOperations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listSolNetworkOperationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSolNetworkOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists details for a network operation, including when the operation started and the status of the operation.
     * </p>
     * <p>
     * A network operation is any operation that is done to your network, such as network instance instantiation or
     * termination.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSolNetworkOperations(software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolNetworkOperationsIterable responses = client.listSolNetworkOperationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.tnb.paginators.ListSolNetworkOperationsIterable responses = client
     *             .listSolNetworkOperationsPaginator(request);
     *     for (software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolNetworkOperationsIterable responses = client.listSolNetworkOperationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSolNetworkOperations(software.amazon.awssdk.services.tnb.model.ListSolNetworkOperationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSolNetworkOperationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolNetworkOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolNetworkOperationsIterable listSolNetworkOperationsPaginator(
            ListSolNetworkOperationsRequest listSolNetworkOperationsRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        return new ListSolNetworkOperationsIterable(this, applyPaginatorUserAgent(listSolNetworkOperationsRequest));
    }

    /**
     * <p>
     * Lists network packages.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param listSolNetworkPackagesRequest
     * @return Result of the ListSolNetworkPackages operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolNetworkPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkPackages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolNetworkPackagesResponse listSolNetworkPackages(ListSolNetworkPackagesRequest listSolNetworkPackagesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSolNetworkPackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSolNetworkPackagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSolNetworkPackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSolNetworkPackages");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSolNetworkPackagesRequest, ListSolNetworkPackagesResponse>()
                            .withOperationName("ListSolNetworkPackages").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listSolNetworkPackagesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSolNetworkPackagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists network packages.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSolNetworkPackages(software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolNetworkPackagesIterable responses = client.listSolNetworkPackagesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.tnb.paginators.ListSolNetworkPackagesIterable responses = client
     *             .listSolNetworkPackagesPaginator(request);
     *     for (software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.tnb.paginators.ListSolNetworkPackagesIterable responses = client.listSolNetworkPackagesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSolNetworkPackages(software.amazon.awssdk.services.tnb.model.ListSolNetworkPackagesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSolNetworkPackagesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListSolNetworkPackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListSolNetworkPackages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSolNetworkPackagesIterable listSolNetworkPackagesPaginator(
            ListSolNetworkPackagesRequest listSolNetworkPackagesRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        return new ListSolNetworkPackagesIterable(this, applyPaginatorUserAgent(listSolNetworkPackagesRequest));
    }

    /**
     * <p>
     * Lists tags for AWS TNB resources.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads the contents of a function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param putSolFunctionPackageContentRequest
     * @return Result of the PutSolFunctionPackageContent operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.PutSolFunctionPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/PutSolFunctionPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutSolFunctionPackageContentResponse putSolFunctionPackageContent(
            PutSolFunctionPackageContentRequest putSolFunctionPackageContentRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutSolFunctionPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutSolFunctionPackageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSolFunctionPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSolFunctionPackageContent");

            return clientHandler
                    .execute(new ClientExecutionParams<PutSolFunctionPackageContentRequest, PutSolFunctionPackageContentResponse>()
                            .withOperationName("PutSolFunctionPackageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putSolFunctionPackageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutSolFunctionPackageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uploads the contents of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param putSolNetworkPackageContentRequest
     * @return Result of the PutSolNetworkPackageContent operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.PutSolNetworkPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/PutSolNetworkPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutSolNetworkPackageContentResponse putSolNetworkPackageContent(
            PutSolNetworkPackageContentRequest putSolNetworkPackageContentRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutSolNetworkPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutSolNetworkPackageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSolNetworkPackageContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSolNetworkPackageContent");

            return clientHandler
                    .execute(new ClientExecutionParams<PutSolNetworkPackageContentRequest, PutSolNetworkPackageContentResponse>()
                            .withOperationName("PutSolNetworkPackageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putSolNetworkPackageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutSolNetworkPackageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags an AWS TNB resource.
     * </p>
     * <p>
     * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an optional
     * value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Terminates a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     * <p>
     * You must terminate a network instance before you can delete it.
     * </p>
     *
     * @param terminateSolNetworkInstanceRequest
     * @return Result of the TerminateSolNetworkInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ServiceQuotaExceededException
     *         Service quotas have been exceeded.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.TerminateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/TerminateSolNetworkInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TerminateSolNetworkInstanceResponse terminateSolNetworkInstance(
            TerminateSolNetworkInstanceRequest terminateSolNetworkInstanceRequest) throws InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TerminateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TerminateSolNetworkInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateSolNetworkInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<TerminateSolNetworkInstanceRequest, TerminateSolNetworkInstanceResponse>()
                            .withOperationName("TerminateSolNetworkInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(terminateSolNetworkInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TerminateSolNetworkInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untags an AWS TNB resource.
     * </p>
     * <p>
     * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an optional
     * value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the operational state of function package.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param updateSolFunctionPackageRequest
     * @return Result of the UpdateSolFunctionPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.UpdateSolFunctionPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UpdateSolFunctionPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateSolFunctionPackageResponse updateSolFunctionPackage(
            UpdateSolFunctionPackageRequest updateSolFunctionPackageRequest) throws InternalServerException, ThrottlingException,
            ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSolFunctionPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSolFunctionPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSolFunctionPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSolFunctionPackage");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSolFunctionPackageRequest, UpdateSolFunctionPackageResponse>()
                            .withOperationName("UpdateSolFunctionPackage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateSolFunctionPackageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSolFunctionPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a network instance.
     * </p>
     * <p>
     * A network instance is a single network created in Amazon Web Services TNB that can be deployed and on which
     * life-cycle operations (like terminate, update, and delete) can be performed.
     * </p>
     *
     * @param updateSolNetworkInstanceRequest
     * @return Result of the UpdateSolNetworkInstance operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ServiceQuotaExceededException
     *         Service quotas have been exceeded.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.UpdateSolNetworkInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UpdateSolNetworkInstance" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateSolNetworkInstanceResponse updateSolNetworkInstance(
            UpdateSolNetworkInstanceRequest updateSolNetworkInstanceRequest) throws InternalServerException,
            ServiceQuotaExceededException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSolNetworkInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSolNetworkInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSolNetworkInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSolNetworkInstance");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSolNetworkInstanceRequest, UpdateSolNetworkInstanceResponse>()
                            .withOperationName("UpdateSolNetworkInstance").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateSolNetworkInstanceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSolNetworkInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the operational state of a network package.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     * <p>
     * A network service descriptor is a .yaml file in a network package that uses the TOSCA standard to describe the
     * network functions you want to deploy and the Amazon Web Services infrastructure you want to deploy the network
     * functions on.
     * </p>
     *
     * @param updateSolNetworkPackageRequest
     * @return Result of the UpdateSolNetworkPackage operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.UpdateSolNetworkPackage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/UpdateSolNetworkPackage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateSolNetworkPackageResponse updateSolNetworkPackage(UpdateSolNetworkPackageRequest updateSolNetworkPackageRequest)
            throws InternalServerException, ThrottlingException, ValidationException, ResourceNotFoundException,
            AccessDeniedException, AwsServiceException, SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSolNetworkPackageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSolNetworkPackageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSolNetworkPackageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSolNetworkPackage");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSolNetworkPackageRequest, UpdateSolNetworkPackageResponse>()
                            .withOperationName("UpdateSolNetworkPackage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateSolNetworkPackageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSolNetworkPackageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Validates function package content. This can be used as a dry run before uploading function package content with
     * <a href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolFunctionPackageContent.html">
     * PutSolFunctionPackageContent</a>.
     * </p>
     * <p>
     * A function package is a .zip file in CSAR (Cloud Service Archive) format that contains a network function (an
     * ETSI standard telecommunication application) and function package descriptor that uses the TOSCA standard to
     * describe how the network functions should run on your network.
     * </p>
     *
     * @param validateSolFunctionPackageContentRequest
     * @return Result of the ValidateSolFunctionPackageContent operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ValidateSolFunctionPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ValidateSolFunctionPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ValidateSolFunctionPackageContentResponse validateSolFunctionPackageContent(
            ValidateSolFunctionPackageContentRequest validateSolFunctionPackageContentRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidateSolFunctionPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ValidateSolFunctionPackageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                validateSolFunctionPackageContentRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateSolFunctionPackageContent");

            return clientHandler
                    .execute(new ClientExecutionParams<ValidateSolFunctionPackageContentRequest, ValidateSolFunctionPackageContentResponse>()
                            .withOperationName("ValidateSolFunctionPackageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(validateSolFunctionPackageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ValidateSolFunctionPackageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Validates network package content. This can be used as a dry run before uploading network package content with <a
     * href="https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolNetworkPackageContent.html">
     * PutSolNetworkPackageContent</a>.
     * </p>
     * <p>
     * A network package is a .zip file in CSAR (Cloud Service Archive) format defines the function packages you want to
     * deploy and the Amazon Web Services infrastructure you want to deploy them on.
     * </p>
     *
     * @param validateSolNetworkPackageContentRequest
     * @return Result of the ValidateSolNetworkPackageContent operation returned by the service.
     * @throws InternalServerException
     *         Unexpected error occurred. Problem on the server.
     * @throws ThrottlingException
     *         Exception caused by throttling.
     * @throws ValidationException
     *         Unable to process the request because the client provided input failed to satisfy request constraints.
     * @throws ResourceNotFoundException
     *         Request references a resource that doesn't exist.
     * @throws AccessDeniedException
     *         Insufficient permissions to make request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws TnbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample TnbClient.ValidateSolNetworkPackageContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/tnb-2008-10-21/ValidateSolNetworkPackageContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ValidateSolNetworkPackageContentResponse validateSolNetworkPackageContent(
            ValidateSolNetworkPackageContentRequest validateSolNetworkPackageContentRequest) throws InternalServerException,
            ThrottlingException, ValidationException, ResourceNotFoundException, AccessDeniedException, AwsServiceException,
            SdkClientException, TnbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidateSolNetworkPackageContentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ValidateSolNetworkPackageContentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                validateSolNetworkPackageContentRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "tnb");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateSolNetworkPackageContent");

            return clientHandler
                    .execute(new ClientExecutionParams<ValidateSolNetworkPackageContentRequest, ValidateSolNetworkPackageContentResponse>()
                            .withOperationName("ValidateSolNetworkPackageContent").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(validateSolNetworkPackageContentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ValidateSolNetworkPackageContentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends TnbRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(TnbException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final TnbServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
