/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.textract.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <code>Block</code> represents items that are recognized in a document within a group of pixels close to each other.
 * The information returned in a <code>Block</code> object depends on the type of operation. In text detection for
 * documents (for example <a>DetectDocumentText</a>), you get information about the detected words and lines of text. In
 * text analysis (for example <a>AnalyzeDocument</a>), you can also get information about the fields, tables, and
 * selection elements that are detected in the document.
 * </p>
 * <p>
 * An array of <code>Block</code> objects is returned by both synchronous and asynchronous operations. In synchronous
 * operations, such as <a>DetectDocumentText</a>, the array of <code>Block</code> objects is the entire set of results.
 * In asynchronous operations, such as <a>GetDocumentAnalysis</a>, the array is returned over one or more responses.
 * </p>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/textract/latest/dg/how-it-works.html">How Amazon
 * Textract Works</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Block implements SdkPojo, Serializable, ToCopyableBuilder<Block.Builder, Block> {
    private static final SdkField<String> BLOCK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BlockType").getter(getter(Block::blockTypeAsString)).setter(setter(Builder::blockType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockType").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("Confidence").getter(getter(Block::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<String> TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Text")
            .getter(getter(Block::text)).setter(setter(Builder::text))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Text").build()).build();

    private static final SdkField<String> TEXT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TextType").getter(getter(Block::textTypeAsString)).setter(setter(Builder::textType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TextType").build()).build();

    private static final SdkField<Integer> ROW_INDEX_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RowIndex").getter(getter(Block::rowIndex)).setter(setter(Builder::rowIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RowIndex").build()).build();

    private static final SdkField<Integer> COLUMN_INDEX_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ColumnIndex").getter(getter(Block::columnIndex)).setter(setter(Builder::columnIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnIndex").build()).build();

    private static final SdkField<Integer> ROW_SPAN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RowSpan").getter(getter(Block::rowSpan)).setter(setter(Builder::rowSpan))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RowSpan").build()).build();

    private static final SdkField<Integer> COLUMN_SPAN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ColumnSpan").getter(getter(Block::columnSpan)).setter(setter(Builder::columnSpan))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ColumnSpan").build()).build();

    private static final SdkField<Geometry> GEOMETRY_FIELD = SdkField.<Geometry> builder(MarshallingType.SDK_POJO)
            .memberName("Geometry").getter(getter(Block::geometry)).setter(setter(Builder::geometry))
            .constructor(Geometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Block::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<List<Relationship>> RELATIONSHIPS_FIELD = SdkField
            .<List<Relationship>> builder(MarshallingType.LIST)
            .memberName("Relationships")
            .getter(getter(Block::relationships))
            .setter(setter(Builder::relationships))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Relationships").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Relationship> builder(MarshallingType.SDK_POJO)
                                            .constructor(Relationship::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ENTITY_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EntityTypes")
            .getter(getter(Block::entityTypesAsStrings))
            .setter(setter(Builder::entityTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SELECTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SelectionStatus").getter(getter(Block::selectionStatusAsString))
            .setter(setter(Builder::selectionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectionStatus").build()).build();

    private static final SdkField<Integer> PAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Page")
            .getter(getter(Block::page)).setter(setter(Builder::page))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Page").build()).build();

    private static final SdkField<Query> QUERY_FIELD = SdkField.<Query> builder(MarshallingType.SDK_POJO).memberName("Query")
            .getter(getter(Block::query)).setter(setter(Builder::query)).constructor(Query::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Query").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BLOCK_TYPE_FIELD,
            CONFIDENCE_FIELD, TEXT_FIELD, TEXT_TYPE_FIELD, ROW_INDEX_FIELD, COLUMN_INDEX_FIELD, ROW_SPAN_FIELD,
            COLUMN_SPAN_FIELD, GEOMETRY_FIELD, ID_FIELD, RELATIONSHIPS_FIELD, ENTITY_TYPES_FIELD, SELECTION_STATUS_FIELD,
            PAGE_FIELD, QUERY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String blockType;

    private final Float confidence;

    private final String text;

    private final String textType;

    private final Integer rowIndex;

    private final Integer columnIndex;

    private final Integer rowSpan;

    private final Integer columnSpan;

    private final Geometry geometry;

    private final String id;

    private final List<Relationship> relationships;

    private final List<String> entityTypes;

    private final String selectionStatus;

    private final Integer page;

    private final Query query;

    private Block(BuilderImpl builder) {
        this.blockType = builder.blockType;
        this.confidence = builder.confidence;
        this.text = builder.text;
        this.textType = builder.textType;
        this.rowIndex = builder.rowIndex;
        this.columnIndex = builder.columnIndex;
        this.rowSpan = builder.rowSpan;
        this.columnSpan = builder.columnSpan;
        this.geometry = builder.geometry;
        this.id = builder.id;
        this.relationships = builder.relationships;
        this.entityTypes = builder.entityTypes;
        this.selectionStatus = builder.selectionStatus;
        this.page = builder.page;
        this.query = builder.query;
    }

    /**
     * <p>
     * The type of text item that's recognized. In operations for text detection, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters that
     * aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In text analysis operations, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on a
     * document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
     * <code>Block</code> object or a VALUE <code>Block</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
     * that aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
     * rows or columns, with a cell span of one row and one column each.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table, or
     * embedded as the first row of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of text below a
     * table or embedded as the last row of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in the
     * cell.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
     * <code>Relationships</code> array for this cell contain data from individual cells.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
     * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
     * selection element.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can be returned
     * as part of a Key-Value pair or a detected cell.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that attaches it
     * to its answer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with an alias and
     * ID for ease of locating in a response. Also contains location and confidence score.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blockType} will
     * return {@link BlockType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blockTypeAsString}.
     * </p>
     * 
     * @return The type of text item that's recognized. In operations for text detection, the following types are
     *         returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
     *         page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
     *         that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In text analysis operations, the following types are returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
     *         detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object
     *         is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
     *         characters that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or
     *         more rows or columns, with a cell span of one row and one column each.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table, or
     *         embedded as the first row of a table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of text
     *         below a table or embedded as the last row of a table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text
     *         in the cell.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
     *         <code>Relationships</code> array for this cell contain data from individual cells.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
     *         that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status
     *         of the selection element.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can be
     *         returned as part of a Key-Value pair or a detected cell.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that
     *         attaches it to its answer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with an
     *         alias and ID for ease of locating in a response. Also contains location and confidence score.
     *         </p>
     *         </li>
     * @see BlockType
     */
    public final BlockType blockType() {
        return BlockType.fromValue(blockType);
    }

    /**
     * <p>
     * The type of text item that's recognized. In operations for text detection, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters that
     * aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In text analysis operations, the following types are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on a
     * document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
     * <code>Block</code> object or a VALUE <code>Block</code> object.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script characters
     * that aren't separated by spaces.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
     * rows or columns, with a cell span of one row and one column each.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table, or
     * embedded as the first row of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of text below a
     * table or embedded as the last row of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in the
     * cell.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
     * <code>Relationships</code> array for this cell contain data from individual cells.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
     * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
     * selection element.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can be returned
     * as part of a Key-Value pair or a detected cell.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that attaches it
     * to its answer.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with an alias and
     * ID for ease of locating in a response. Also contains location and confidence score.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blockType} will
     * return {@link BlockType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blockTypeAsString}.
     * </p>
     * 
     * @return The type of text item that's recognized. In operations for text detection, the following types are
     *         returned:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
     *         page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
     *         that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         In text analysis operations, the following types are returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
     *         detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object
     *         is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
     *         characters that aren't separated by spaces.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or
     *         more rows or columns, with a cell span of one row and one column each.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table, or
     *         embedded as the first row of a table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of text
     *         below a table or embedded as the last row of a table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text
     *         in the cell.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
     *         <code>Relationships</code> array for this cell contain data from individual cells.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
     *         that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status
     *         of the selection element.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can be
     *         returned as part of a Key-Value pair or a detected cell.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that
     *         attaches it to its answer.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with an
     *         alias and ID for ease of locating in a response. Also contains location and confidence score.
     *         </p>
     *         </li>
     * @see BlockType
     */
    public final String blockTypeAsString() {
        return blockType;
    }

    /**
     * <p>
     * The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy of the
     * geometry points around the recognized text.
     * </p>
     * 
     * @return The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy of
     *         the geometry points around the recognized text.
     */
    public final Float confidence() {
        return confidence;
    }

    /**
     * <p>
     * The word or line of text that's recognized by Amazon Textract.
     * </p>
     * 
     * @return The word or line of text that's recognized by Amazon Textract.
     */
    public final String text() {
        return text;
    }

    /**
     * <p>
     * The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #textType} will
     * return {@link TextType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #textTypeAsString}.
     * </p>
     * 
     * @return The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
     * @see TextType
     */
    public final TextType textType() {
        return TextType.fromValue(textType);
    }

    /**
     * <p>
     * The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #textType} will
     * return {@link TextType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #textTypeAsString}.
     * </p>
     * 
     * @return The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
     * @see TextType
     */
    public final String textTypeAsString() {
        return textType;
    }

    /**
     * <p>
     * The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't returned by
     * <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't
     *         returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public final Integer rowIndex() {
        return rowIndex;
    }

    /**
     * <p>
     * The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> isn't returned
     * by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> isn't
     *         returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public final Integer columnIndex() {
        return columnIndex;
    }

    /**
     * <p>
     * The number of rows that a table cell spans. <code>RowSpan</code> isn't returned by
     * <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The number of rows that a table cell spans. <code>RowSpan</code> isn't returned by
     *         <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public final Integer rowSpan() {
        return rowSpan;
    }

    /**
     * <p>
     * The number of columns that a table cell spans. <code>ColumnSpan</code> isn't returned by
     * <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     * </p>
     * 
     * @return The number of columns that a table cell spans. <code>ColumnSpan</code> isn't returned by
     *         <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
     */
    public final Integer columnSpan() {
        return columnSpan;
    }

    /**
     * <p>
     * The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that surrounds
     * the text, and a finer-grain polygon for more accurate spatial information.
     * </p>
     * 
     * @return The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that
     *         surrounds the text, and a finer-grain polygon for more accurate spatial information.
     */
    public final Geometry geometry() {
        return geometry;
    }

    /**
     * <p>
     * The identifier for the recognized text. The identifier is only unique for a single operation.
     * </p>
     * 
     * @return The identifier for the recognized text. The identifier is only unique for a single operation.
     */
    public final String id() {
        return id;
    }

    /**
     * For responses, this returns true if the service returned a value for the Relationships property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelationships() {
        return relationships != null && !(relationships instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of relationship objects that describe how blocks are related to each other. For example, a LINE block
     * object contains a CHILD relationship type with the WORD blocks that make up the line of text. There aren't
     * Relationship objects in the list for relationships that don't exist, such as when the current block has no child
     * blocks.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelationships} method.
     * </p>
     * 
     * @return A list of relationship objects that describe how blocks are related to each other. For example, a LINE
     *         block object contains a CHILD relationship type with the WORD blocks that make up the line of text. There
     *         aren't Relationship objects in the list for relationships that don't exist, such as when the current
     *         block has no child blocks.
     */
    public final List<Relationship> relationships() {
        return relationships;
    }

    /**
     * <p>
     * The type of entity.
     * </p>
     * <p>
     * The following entity types can be returned by FORMS analysis:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>KEY</i> - An identifier for a field on the document.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>VALUE</i> - The field text.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following entity types can be returned by TABLES analysis:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section title is a
     * cell that typically spans an entire row above a section.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
     * additional, smaller table that contains summary information for another table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row corresponds to
     * the headers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     * <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityTypes} method.
     * </p>
     * 
     * @return The type of entity. </p>
     *         <p>
     *         The following entity types can be returned by FORMS analysis:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>KEY</i> - An identifier for a field on the document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>VALUE</i> - The field text.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following entity types can be returned by TABLES analysis:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section
     *         title is a cell that typically spans an entire row above a section.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
     *         additional, smaller table that contains summary information for another table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row
     *         corresponds to the headers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     *         <code>GetDocumentTextDetection</code>.
     */
    public final List<EntityType> entityTypes() {
        return EntityTypesCopier.copyStringToEnum(entityTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the EntityTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEntityTypes() {
        return entityTypes != null && !(entityTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of entity.
     * </p>
     * <p>
     * The following entity types can be returned by FORMS analysis:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>KEY</i> - An identifier for a field on the document.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>VALUE</i> - The field text.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following entity types can be returned by TABLES analysis:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section title is a
     * cell that typically spans an entire row above a section.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
     * additional, smaller table that contains summary information for another table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row corresponds to
     * the headers.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     * <code>GetDocumentTextDetection</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityTypes} method.
     * </p>
     * 
     * @return The type of entity. </p>
     *         <p>
     *         The following entity types can be returned by FORMS analysis:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>KEY</i> - An identifier for a field on the document.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>VALUE</i> - The field text.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following entity types can be returned by TABLES analysis:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section
     *         title is a cell that typically spans an entire row above a section.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
     *         additional, smaller table that contains summary information for another table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row
     *         corresponds to the headers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
     *         <code>GetDocumentTextDetection</code>.
     */
    public final List<String> entityTypesAsStrings() {
        return entityTypes;
    }

    /**
     * <p>
     * The selection status of a selection element, such as an option button or check box.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #selectionStatus}
     * will return {@link SelectionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #selectionStatusAsString}.
     * </p>
     * 
     * @return The selection status of a selection element, such as an option button or check box.
     * @see SelectionStatus
     */
    public final SelectionStatus selectionStatus() {
        return SelectionStatus.fromValue(selectionStatus);
    }

    /**
     * <p>
     * The selection status of a selection element, such as an option button or check box.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #selectionStatus}
     * will return {@link SelectionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #selectionStatusAsString}.
     * </p>
     * 
     * @return The selection status of a selection element, such as an option button or check box.
     * @see SelectionStatus
     */
    public final String selectionStatusAsString() {
        return selectionStatus;
    }

    /**
     * <p>
     * The page on which a block was detected. <code>Page</code> is returned by synchronous and asynchronous operations.
     * Page values greater than 1 are only returned for multipage documents that are in PDF or TIFF format. A scanned
     * image (JPEG/PNG) provided to an asynchronous operation, even if it contains multiple document pages, is
     * considered a single-page document. This means that for scanned images the value of <code>Page</code> is always 1.
     * </p>
     * 
     * @return The page on which a block was detected. <code>Page</code> is returned by synchronous and asynchronous
     *         operations. Page values greater than 1 are only returned for multipage documents that are in PDF or TIFF
     *         format. A scanned image (JPEG/PNG) provided to an asynchronous operation, even if it contains multiple
     *         document pages, is considered a single-page document. This means that for scanned images the value of
     *         <code>Page</code> is always 1.
     */
    public final Integer page() {
        return page;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final Query query() {
        return query;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(blockTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(text());
        hashCode = 31 * hashCode + Objects.hashCode(textTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(rowIndex());
        hashCode = 31 * hashCode + Objects.hashCode(columnIndex());
        hashCode = 31 * hashCode + Objects.hashCode(rowSpan());
        hashCode = 31 * hashCode + Objects.hashCode(columnSpan());
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelationships() ? relationships() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEntityTypes() ? entityTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(selectionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(page());
        hashCode = 31 * hashCode + Objects.hashCode(query());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Block)) {
            return false;
        }
        Block other = (Block) obj;
        return Objects.equals(blockTypeAsString(), other.blockTypeAsString()) && Objects.equals(confidence(), other.confidence())
                && Objects.equals(text(), other.text()) && Objects.equals(textTypeAsString(), other.textTypeAsString())
                && Objects.equals(rowIndex(), other.rowIndex()) && Objects.equals(columnIndex(), other.columnIndex())
                && Objects.equals(rowSpan(), other.rowSpan()) && Objects.equals(columnSpan(), other.columnSpan())
                && Objects.equals(geometry(), other.geometry()) && Objects.equals(id(), other.id())
                && hasRelationships() == other.hasRelationships() && Objects.equals(relationships(), other.relationships())
                && hasEntityTypes() == other.hasEntityTypes()
                && Objects.equals(entityTypesAsStrings(), other.entityTypesAsStrings())
                && Objects.equals(selectionStatusAsString(), other.selectionStatusAsString())
                && Objects.equals(page(), other.page()) && Objects.equals(query(), other.query());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Block").add("BlockType", blockTypeAsString()).add("Confidence", confidence())
                .add("Text", text()).add("TextType", textTypeAsString()).add("RowIndex", rowIndex())
                .add("ColumnIndex", columnIndex()).add("RowSpan", rowSpan()).add("ColumnSpan", columnSpan())
                .add("Geometry", geometry()).add("Id", id()).add("Relationships", hasRelationships() ? relationships() : null)
                .add("EntityTypes", hasEntityTypes() ? entityTypesAsStrings() : null)
                .add("SelectionStatus", selectionStatusAsString()).add("Page", page()).add("Query", query()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BlockType":
            return Optional.ofNullable(clazz.cast(blockTypeAsString()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "Text":
            return Optional.ofNullable(clazz.cast(text()));
        case "TextType":
            return Optional.ofNullable(clazz.cast(textTypeAsString()));
        case "RowIndex":
            return Optional.ofNullable(clazz.cast(rowIndex()));
        case "ColumnIndex":
            return Optional.ofNullable(clazz.cast(columnIndex()));
        case "RowSpan":
            return Optional.ofNullable(clazz.cast(rowSpan()));
        case "ColumnSpan":
            return Optional.ofNullable(clazz.cast(columnSpan()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Relationships":
            return Optional.ofNullable(clazz.cast(relationships()));
        case "EntityTypes":
            return Optional.ofNullable(clazz.cast(entityTypesAsStrings()));
        case "SelectionStatus":
            return Optional.ofNullable(clazz.cast(selectionStatusAsString()));
        case "Page":
            return Optional.ofNullable(clazz.cast(page()));
        case "Query":
            return Optional.ofNullable(clazz.cast(query()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Block, T> g) {
        return obj -> g.apply((Block) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Block> {
        /**
         * <p>
         * The type of text item that's recognized. In operations for text detection, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
         * that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In text analysis operations, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on
         * a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
         * <code>Block</code> object or a VALUE <code>Block</code> object.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         * characters that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
         * rows or columns, with a cell span of one row and one column each.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table, or
         * embedded as the first row of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of text below
         * a table or embedded as the last row of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in
         * the cell.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
         * <code>Relationships</code> array for this cell contain data from individual cells.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
         * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
         * selection element.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can be
         * returned as part of a Key-Value pair or a detected cell.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that attaches
         * it to its answer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with an alias
         * and ID for ease of locating in a response. Also contains location and confidence score.
         * </p>
         * </li>
         * </ul>
         * 
         * @param blockType
         *        The type of text item that's recognized. In operations for text detection, the following types are
         *        returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In text analysis operations, the following types are returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
         *        detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET
         *        object is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two
         *        or more rows or columns, with a cell span of one row and one column each.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table,
         *        or embedded as the first row of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of
         *        text below a table or embedded as the last row of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the
         *        text in the cell.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
         *        <code>Relationships</code> array for this cell contain data from individual cells.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
         *        that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the
         *        status of the selection element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can
         *        be returned as part of a Key-Value pair or a detected cell.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that
         *        attaches it to its answer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with
         *        an alias and ID for ease of locating in a response. Also contains location and confidence score.
         *        </p>
         *        </li>
         * @see BlockType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BlockType
         */
        Builder blockType(String blockType);

        /**
         * <p>
         * The type of text item that's recognized. In operations for text detection, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script characters
         * that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * </ul>
         * <p>
         * In text analysis operations, the following types are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's detected on
         * a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET object is a KEY
         * <code>Block</code> object or a VALUE <code>Block</code> object.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         * characters that aren't separated by spaces.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two or more
         * rows or columns, with a cell span of one row and one column each.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table, or
         * embedded as the first row of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of text below
         * a table or embedded as the last row of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the text in
         * the cell.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
         * <code>Relationships</code> array for this cell contain data from individual cells.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box that's
         * detected on a document page. Use the value of <code>SelectionStatus</code> to determine the status of the
         * selection element.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can be
         * returned as part of a Key-Value pair or a detected cell.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that attaches
         * it to its answer.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with an alias
         * and ID for ease of locating in a response. Also contains location and confidence score.
         * </p>
         * </li>
         * </ul>
         * 
         * @param blockType
         *        The type of text item that's recognized. In operations for text detection, the following types are
         *        returned:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of the LINE <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        In text analysis operations, the following types are returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>PAGE</i> - Contains a list of child <code>Block</code> objects that are detected on a document
         *        page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>KEY_VALUE_SET</i> - Stores the KEY and VALUE <code>Block</code> objects for linked text that's
         *        detected on a document page. Use the <code>EntityType</code> field to determine if a KEY_VALUE_SET
         *        object is a KEY <code>Block</code> object or a VALUE <code>Block</code> object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>WORD</i> - A word that's detected on a document page. A word is one or more ISO basic Latin script
         *        characters that aren't separated by spaces.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>LINE</i> - A string of tab-delimited, contiguous words that are detected on a document page.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE</i> - A table that's detected on a document page. A table is grid-based information with two
         *        or more rows or columns, with a cell span of one row and one column each.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_TITLE</i> - The title of a table. A title is typically a line of text above or below a table,
         *        or embedded as the first row of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_FOOTER</i> - The footer associated with a table. A footer is typically a line or lines of
         *        text below a table or embedded as the last row of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>CELL</i> - A cell within a detected table. The cell is the parent of the block that contains the
         *        text in the cell.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>MERGED_CELL</i> - A cell in a table whose content spans more than one row or column. The
         *        <code>Relationships</code> array for this cell contain data from individual cells.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SELECTION_ELEMENT</i> - A selection element such as an option button (radio button) or a check box
         *        that's detected on a document page. Use the value of <code>SelectionStatus</code> to determine the
         *        status of the selection element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SIGNATURE</i> - The location and confidence score of a signature detected on a document page. Can
         *        be returned as part of a Key-Value pair or a detected cell.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>QUERY</i> - A question asked during the call of AnalyzeDocument. Contains an alias and an ID that
         *        attaches it to its answer.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>QUERY_RESULT</i> - A response to a question asked during the call of analyze document. Comes with
         *        an alias and ID for ease of locating in a response. Also contains location and confidence score.
         *        </p>
         *        </li>
         * @see BlockType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BlockType
         */
        Builder blockType(BlockType blockType);

        /**
         * <p>
         * The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy of the
         * geometry points around the recognized text.
         * </p>
         * 
         * @param confidence
         *        The confidence score that Amazon Textract has in the accuracy of the recognized text and the accuracy
         *        of the geometry points around the recognized text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * The word or line of text that's recognized by Amazon Textract.
         * </p>
         * 
         * @param text
         *        The word or line of text that's recognized by Amazon Textract.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder text(String text);

        /**
         * <p>
         * The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
         * </p>
         * 
         * @param textType
         *        The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
         * @see TextType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextType
         */
        Builder textType(String textType);

        /**
         * <p>
         * The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
         * </p>
         * 
         * @param textType
         *        The kind of text that Amazon Textract has detected. Can check for handwritten text and printed text.
         * @see TextType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TextType
         */
        Builder textType(TextType textType);

        /**
         * <p>
         * The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't returned
         * by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param rowIndex
         *        The row in which a table cell is located. The first row position is 1. <code>RowIndex</code> isn't
         *        returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rowIndex(Integer rowIndex);

        /**
         * <p>
         * The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> isn't
         * returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param columnIndex
         *        The column in which a table cell appears. The first column position is 1. <code>ColumnIndex</code> 
         *        isn't returned by <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnIndex(Integer columnIndex);

        /**
         * <p>
         * The number of rows that a table cell spans. <code>RowSpan</code> isn't returned by
         * <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param rowSpan
         *        The number of rows that a table cell spans. <code>RowSpan</code> isn't returned by
         *        <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rowSpan(Integer rowSpan);

        /**
         * <p>
         * The number of columns that a table cell spans. <code>ColumnSpan</code> isn't returned by
         * <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param columnSpan
         *        The number of columns that a table cell spans. <code>ColumnSpan</code> isn't returned by
         *        <code>DetectDocumentText</code> and <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnSpan(Integer columnSpan);

        /**
         * <p>
         * The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that
         * surrounds the text, and a finer-grain polygon for more accurate spatial information.
         * </p>
         * 
         * @param geometry
         *        The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box
         *        that surrounds the text, and a finer-grain polygon for more accurate spatial information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(Geometry geometry);

        /**
         * <p>
         * The location of the recognized text on the image. It includes an axis-aligned, coarse bounding box that
         * surrounds the text, and a finer-grain polygon for more accurate spatial information.
         * </p>
         * This is a convenience method that creates an instance of the {@link Geometry.Builder} avoiding the need to
         * create one manually via {@link Geometry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Geometry.Builder#build()} is called immediately and its result is
         * passed to {@link #geometry(Geometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link Geometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(Geometry)
         */
        default Builder geometry(Consumer<Geometry.Builder> geometry) {
            return geometry(Geometry.builder().applyMutation(geometry).build());
        }

        /**
         * <p>
         * The identifier for the recognized text. The identifier is only unique for a single operation.
         * </p>
         * 
         * @param id
         *        The identifier for the recognized text. The identifier is only unique for a single operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A list of relationship objects that describe how blocks are related to each other. For example, a LINE block
         * object contains a CHILD relationship type with the WORD blocks that make up the line of text. There aren't
         * Relationship objects in the list for relationships that don't exist, such as when the current block has no
         * child blocks.
         * </p>
         * 
         * @param relationships
         *        A list of relationship objects that describe how blocks are related to each other. For example, a LINE
         *        block object contains a CHILD relationship type with the WORD blocks that make up the line of text.
         *        There aren't Relationship objects in the list for relationships that don't exist, such as when the
         *        current block has no child blocks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Collection<Relationship> relationships);

        /**
         * <p>
         * A list of relationship objects that describe how blocks are related to each other. For example, a LINE block
         * object contains a CHILD relationship type with the WORD blocks that make up the line of text. There aren't
         * Relationship objects in the list for relationships that don't exist, such as when the current block has no
         * child blocks.
         * </p>
         * 
         * @param relationships
         *        A list of relationship objects that describe how blocks are related to each other. For example, a LINE
         *        block object contains a CHILD relationship type with the WORD blocks that make up the line of text.
         *        There aren't Relationship objects in the list for relationships that don't exist, such as when the
         *        current block has no child blocks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Relationship... relationships);

        /**
         * <p>
         * A list of relationship objects that describe how blocks are related to each other. For example, a LINE block
         * object contains a CHILD relationship type with the WORD blocks that make up the line of text. There aren't
         * Relationship objects in the list for relationships that don't exist, such as when the current block has no
         * child blocks.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.textract.model.Relationship.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.textract.model.Relationship#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.textract.model.Relationship.Builder#build()} is called immediately and
         * its result is passed to {@link #relationships(List<Relationship>)}.
         * 
         * @param relationships
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.textract.model.Relationship.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relationships(java.util.Collection<Relationship>)
         */
        Builder relationships(Consumer<Relationship.Builder>... relationships);

        /**
         * <p>
         * The type of entity.
         * </p>
         * <p>
         * The following entity types can be returned by FORMS analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following entity types can be returned by TABLES analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section title
         * is a cell that typically spans an entire row above a section.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
         * additional, smaller table that contains summary information for another table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row corresponds
         * to the headers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. </p>
         *        <p>
         *        The following entity types can be returned by FORMS analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following entity types can be returned by TABLES analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section
         *        title is a cell that typically spans an entire row above a section.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or
         *        an additional, smaller table that contains summary information for another table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row
         *        corresponds to the headers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypesWithStrings(Collection<String> entityTypes);

        /**
         * <p>
         * The type of entity.
         * </p>
         * <p>
         * The following entity types can be returned by FORMS analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following entity types can be returned by TABLES analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section title
         * is a cell that typically spans an entire row above a section.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
         * additional, smaller table that contains summary information for another table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row corresponds
         * to the headers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. </p>
         *        <p>
         *        The following entity types can be returned by FORMS analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following entity types can be returned by TABLES analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section
         *        title is a cell that typically spans an entire row above a section.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or
         *        an additional, smaller table that contains summary information for another table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row
         *        corresponds to the headers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypesWithStrings(String... entityTypes);

        /**
         * <p>
         * The type of entity.
         * </p>
         * <p>
         * The following entity types can be returned by FORMS analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following entity types can be returned by TABLES analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section title
         * is a cell that typically spans an entire row above a section.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
         * additional, smaller table that contains summary information for another table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row corresponds
         * to the headers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. </p>
         *        <p>
         *        The following entity types can be returned by FORMS analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following entity types can be returned by TABLES analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section
         *        title is a cell that typically spans an entire row above a section.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or
         *        an additional, smaller table that contains summary information for another table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row
         *        corresponds to the headers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypes(Collection<EntityType> entityTypes);

        /**
         * <p>
         * The type of entity.
         * </p>
         * <p>
         * The following entity types can be returned by FORMS analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>KEY</i> - An identifier for a field on the document.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>VALUE</i> - The field text.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following entity types can be returned by TABLES analysis:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section title
         * is a cell that typically spans an entire row above a section.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or an
         * additional, smaller table that contains summary information for another table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row corresponds
         * to the headers.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         * <code>GetDocumentTextDetection</code>.
         * </p>
         * 
         * @param entityTypes
         *        The type of entity. </p>
         *        <p>
         *        The following entity types can be returned by FORMS analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>KEY</i> - An identifier for a field on the document.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>VALUE</i> - The field text.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following entity types can be returned by TABLES analysis:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>COLUMN_HEADER</i> - Identifies a cell that is a header of a column.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_TITLE</i> - Identifies a cell that is a title within the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SECTION_TITLE</i> - Identifies a cell that is a title of a section within a table. A section
         *        title is a cell that typically spans an entire row above a section.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_FOOTER</i> - Identifies a cell that is a footer of a table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>TABLE_SUMMARY</i> - Identifies a summary cell of a table. A summary cell can be a row of a table or
         *        an additional, smaller table that contains summary information for another table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>STRUCTURED_TABLE </i> - Identifies a table with column headers where the content of each row
         *        corresponds to the headers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>SEMI_STRUCTURED_TABLE</i> - Identifies a non-structured table.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>EntityTypes</code> isn't returned by <code>DetectDocumentText</code> and
         *        <code>GetDocumentTextDetection</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypes(EntityType... entityTypes);

        /**
         * <p>
         * The selection status of a selection element, such as an option button or check box.
         * </p>
         * 
         * @param selectionStatus
         *        The selection status of a selection element, such as an option button or check box.
         * @see SelectionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SelectionStatus
         */
        Builder selectionStatus(String selectionStatus);

        /**
         * <p>
         * The selection status of a selection element, such as an option button or check box.
         * </p>
         * 
         * @param selectionStatus
         *        The selection status of a selection element, such as an option button or check box.
         * @see SelectionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SelectionStatus
         */
        Builder selectionStatus(SelectionStatus selectionStatus);

        /**
         * <p>
         * The page on which a block was detected. <code>Page</code> is returned by synchronous and asynchronous
         * operations. Page values greater than 1 are only returned for multipage documents that are in PDF or TIFF
         * format. A scanned image (JPEG/PNG) provided to an asynchronous operation, even if it contains multiple
         * document pages, is considered a single-page document. This means that for scanned images the value of
         * <code>Page</code> is always 1.
         * </p>
         * 
         * @param page
         *        The page on which a block was detected. <code>Page</code> is returned by synchronous and asynchronous
         *        operations. Page values greater than 1 are only returned for multipage documents that are in PDF or
         *        TIFF format. A scanned image (JPEG/PNG) provided to an asynchronous operation, even if it contains
         *        multiple document pages, is considered a single-page document. This means that for scanned images the
         *        value of <code>Page</code> is always 1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder page(Integer page);

        /**
         * <p/>
         * 
         * @param query
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder query(Query query);

        /**
         * <p/>
         * This is a convenience method that creates an instance of the {@link Query.Builder} avoiding the need to
         * create one manually via {@link Query#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Query.Builder#build()} is called immediately and its result is
         * passed to {@link #query(Query)}.
         * 
         * @param query
         *        a consumer that will call methods on {@link Query.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #query(Query)
         */
        default Builder query(Consumer<Query.Builder> query) {
            return query(Query.builder().applyMutation(query).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String blockType;

        private Float confidence;

        private String text;

        private String textType;

        private Integer rowIndex;

        private Integer columnIndex;

        private Integer rowSpan;

        private Integer columnSpan;

        private Geometry geometry;

        private String id;

        private List<Relationship> relationships = DefaultSdkAutoConstructList.getInstance();

        private List<String> entityTypes = DefaultSdkAutoConstructList.getInstance();

        private String selectionStatus;

        private Integer page;

        private Query query;

        private BuilderImpl() {
        }

        private BuilderImpl(Block model) {
            blockType(model.blockType);
            confidence(model.confidence);
            text(model.text);
            textType(model.textType);
            rowIndex(model.rowIndex);
            columnIndex(model.columnIndex);
            rowSpan(model.rowSpan);
            columnSpan(model.columnSpan);
            geometry(model.geometry);
            id(model.id);
            relationships(model.relationships);
            entityTypesWithStrings(model.entityTypes);
            selectionStatus(model.selectionStatus);
            page(model.page);
            query(model.query);
        }

        public final String getBlockType() {
            return blockType;
        }

        public final void setBlockType(String blockType) {
            this.blockType = blockType;
        }

        @Override
        public final Builder blockType(String blockType) {
            this.blockType = blockType;
            return this;
        }

        @Override
        public final Builder blockType(BlockType blockType) {
            this.blockType(blockType == null ? null : blockType.toString());
            return this;
        }

        public final Float getConfidence() {
            return confidence;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final String getText() {
            return text;
        }

        public final void setText(String text) {
            this.text = text;
        }

        @Override
        public final Builder text(String text) {
            this.text = text;
            return this;
        }

        public final String getTextType() {
            return textType;
        }

        public final void setTextType(String textType) {
            this.textType = textType;
        }

        @Override
        public final Builder textType(String textType) {
            this.textType = textType;
            return this;
        }

        @Override
        public final Builder textType(TextType textType) {
            this.textType(textType == null ? null : textType.toString());
            return this;
        }

        public final Integer getRowIndex() {
            return rowIndex;
        }

        public final void setRowIndex(Integer rowIndex) {
            this.rowIndex = rowIndex;
        }

        @Override
        public final Builder rowIndex(Integer rowIndex) {
            this.rowIndex = rowIndex;
            return this;
        }

        public final Integer getColumnIndex() {
            return columnIndex;
        }

        public final void setColumnIndex(Integer columnIndex) {
            this.columnIndex = columnIndex;
        }

        @Override
        public final Builder columnIndex(Integer columnIndex) {
            this.columnIndex = columnIndex;
            return this;
        }

        public final Integer getRowSpan() {
            return rowSpan;
        }

        public final void setRowSpan(Integer rowSpan) {
            this.rowSpan = rowSpan;
        }

        @Override
        public final Builder rowSpan(Integer rowSpan) {
            this.rowSpan = rowSpan;
            return this;
        }

        public final Integer getColumnSpan() {
            return columnSpan;
        }

        public final void setColumnSpan(Integer columnSpan) {
            this.columnSpan = columnSpan;
        }

        @Override
        public final Builder columnSpan(Integer columnSpan) {
            this.columnSpan = columnSpan;
            return this;
        }

        public final Geometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        public final void setGeometry(Geometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        @Override
        public final Builder geometry(Geometry geometry) {
            this.geometry = geometry;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final List<Relationship.Builder> getRelationships() {
            List<Relationship.Builder> result = RelationshipListCopier.copyToBuilder(this.relationships);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelationships(Collection<Relationship.BuilderImpl> relationships) {
            this.relationships = RelationshipListCopier.copyFromBuilder(relationships);
        }

        @Override
        public final Builder relationships(Collection<Relationship> relationships) {
            this.relationships = RelationshipListCopier.copy(relationships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Relationship... relationships) {
            relationships(Arrays.asList(relationships));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Consumer<Relationship.Builder>... relationships) {
            relationships(Stream.of(relationships).map(c -> Relationship.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getEntityTypes() {
            if (entityTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return entityTypes;
        }

        public final void setEntityTypes(Collection<String> entityTypes) {
            this.entityTypes = EntityTypesCopier.copy(entityTypes);
        }

        @Override
        public final Builder entityTypesWithStrings(Collection<String> entityTypes) {
            this.entityTypes = EntityTypesCopier.copy(entityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTypesWithStrings(String... entityTypes) {
            entityTypesWithStrings(Arrays.asList(entityTypes));
            return this;
        }

        @Override
        public final Builder entityTypes(Collection<EntityType> entityTypes) {
            this.entityTypes = EntityTypesCopier.copyEnumToString(entityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTypes(EntityType... entityTypes) {
            entityTypes(Arrays.asList(entityTypes));
            return this;
        }

        public final String getSelectionStatus() {
            return selectionStatus;
        }

        public final void setSelectionStatus(String selectionStatus) {
            this.selectionStatus = selectionStatus;
        }

        @Override
        public final Builder selectionStatus(String selectionStatus) {
            this.selectionStatus = selectionStatus;
            return this;
        }

        @Override
        public final Builder selectionStatus(SelectionStatus selectionStatus) {
            this.selectionStatus(selectionStatus == null ? null : selectionStatus.toString());
            return this;
        }

        public final Integer getPage() {
            return page;
        }

        public final void setPage(Integer page) {
            this.page = page;
        }

        @Override
        public final Builder page(Integer page) {
            this.page = page;
            return this;
        }

        public final Query.Builder getQuery() {
            return query != null ? query.toBuilder() : null;
        }

        public final void setQuery(Query.BuilderImpl query) {
            this.query = query != null ? query.build() : null;
        }

        @Override
        public final Builder query(Query query) {
            this.query = query;
            return this;
        }

        @Override
        public Block build() {
            return new Block(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
