/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssmsap;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ssmsap.internal.SsmSapServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ssmsap.model.ConflictException;
import software.amazon.awssdk.services.ssmsap.model.DeleteResourcePermissionRequest;
import software.amazon.awssdk.services.ssmsap.model.DeleteResourcePermissionResponse;
import software.amazon.awssdk.services.ssmsap.model.DeregisterApplicationRequest;
import software.amazon.awssdk.services.ssmsap.model.DeregisterApplicationResponse;
import software.amazon.awssdk.services.ssmsap.model.GetApplicationRequest;
import software.amazon.awssdk.services.ssmsap.model.GetApplicationResponse;
import software.amazon.awssdk.services.ssmsap.model.GetComponentRequest;
import software.amazon.awssdk.services.ssmsap.model.GetComponentResponse;
import software.amazon.awssdk.services.ssmsap.model.GetDatabaseRequest;
import software.amazon.awssdk.services.ssmsap.model.GetDatabaseResponse;
import software.amazon.awssdk.services.ssmsap.model.GetOperationRequest;
import software.amazon.awssdk.services.ssmsap.model.GetOperationResponse;
import software.amazon.awssdk.services.ssmsap.model.GetResourcePermissionRequest;
import software.amazon.awssdk.services.ssmsap.model.GetResourcePermissionResponse;
import software.amazon.awssdk.services.ssmsap.model.InternalServerException;
import software.amazon.awssdk.services.ssmsap.model.ListApplicationsRequest;
import software.amazon.awssdk.services.ssmsap.model.ListApplicationsResponse;
import software.amazon.awssdk.services.ssmsap.model.ListComponentsRequest;
import software.amazon.awssdk.services.ssmsap.model.ListComponentsResponse;
import software.amazon.awssdk.services.ssmsap.model.ListDatabasesRequest;
import software.amazon.awssdk.services.ssmsap.model.ListDatabasesResponse;
import software.amazon.awssdk.services.ssmsap.model.ListOperationsRequest;
import software.amazon.awssdk.services.ssmsap.model.ListOperationsResponse;
import software.amazon.awssdk.services.ssmsap.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ssmsap.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ssmsap.model.PutResourcePermissionRequest;
import software.amazon.awssdk.services.ssmsap.model.PutResourcePermissionResponse;
import software.amazon.awssdk.services.ssmsap.model.RegisterApplicationRequest;
import software.amazon.awssdk.services.ssmsap.model.RegisterApplicationResponse;
import software.amazon.awssdk.services.ssmsap.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ssmsap.model.SsmSapException;
import software.amazon.awssdk.services.ssmsap.model.StartApplicationRefreshRequest;
import software.amazon.awssdk.services.ssmsap.model.StartApplicationRefreshResponse;
import software.amazon.awssdk.services.ssmsap.model.TagResourceRequest;
import software.amazon.awssdk.services.ssmsap.model.TagResourceResponse;
import software.amazon.awssdk.services.ssmsap.model.UnauthorizedException;
import software.amazon.awssdk.services.ssmsap.model.UntagResourceRequest;
import software.amazon.awssdk.services.ssmsap.model.UntagResourceResponse;
import software.amazon.awssdk.services.ssmsap.model.UpdateApplicationSettingsRequest;
import software.amazon.awssdk.services.ssmsap.model.UpdateApplicationSettingsResponse;
import software.amazon.awssdk.services.ssmsap.model.ValidationException;
import software.amazon.awssdk.services.ssmsap.transform.DeleteResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.DeregisterApplicationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetComponentRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetDatabaseRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetOperationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.GetResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListDatabasesRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.PutResourcePermissionRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.RegisterApplicationRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.StartApplicationRefreshRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ssmsap.transform.UpdateApplicationSettingsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link SsmSapClient}.
 *
 * @see SsmSapClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultSsmSapClient implements SsmSapClient {
    private static final Logger log = Logger.loggerFor(DefaultSsmSapClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final SsmSapServiceClientConfiguration serviceClientConfiguration;

    protected DefaultSsmSapClient(SsmSapServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Removes permissions associated with the target database.
     * </p>
     *
     * @param deleteResourcePermissionRequest
     * @return Result of the DeleteResourcePermission operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.DeleteResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/DeleteResourcePermission"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourcePermissionResponse deleteResourcePermission(
            DeleteResourcePermissionRequest deleteResourcePermissionRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePermission");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePermissionRequest, DeleteResourcePermissionResponse>()
                            .withOperationName("DeleteResourcePermission").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteResourcePermissionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregister an SAP application with AWS Systems Manager for SAP. This action does not aﬀect the existing setup of
     * your SAP workloads on Amazon EC2.
     * </p>
     *
     * @param deregisterApplicationRequest
     * @return Result of the DeregisterApplication operation returned by the service.
     * @throws UnauthorizedException
     *         The request is not authorized.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.DeregisterApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/DeregisterApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeregisterApplicationResponse deregisterApplication(DeregisterApplicationRequest deregisterApplicationRequest)
            throws UnauthorizedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deregisterApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterApplication");

            return clientHandler.execute(new ClientExecutionParams<DeregisterApplicationRequest, DeregisterApplicationResponse>()
                    .withOperationName("DeregisterApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deregisterApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeregisterApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets an application registered with AWS Systems Manager for SAP. It also returns the components of the
     * application.
     * </p>
     *
     * @param getApplicationRequest
     * @return Result of the GetApplication operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/GetApplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetApplicationResponse getApplication(GetApplicationRequest getApplicationRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApplicationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");

            return clientHandler.execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                    .withOperationName("GetApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the component of an application registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param getComponentRequest
     * @return Result of the GetComponent operation returned by the service.
     * @throws UnauthorizedException
     *         The request is not authorized.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/GetComponent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetComponentResponse getComponent(GetComponentRequest getComponentRequest) throws UnauthorizedException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetComponentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetComponentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComponentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponent");

            return clientHandler.execute(new ClientExecutionParams<GetComponentRequest, GetComponentResponse>()
                    .withOperationName("GetComponent").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getComponentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetComponentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the SAP HANA database of an application registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param getDatabaseRequest
     * @return Result of the GetDatabase operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/GetDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDatabaseResponse getDatabase(GetDatabaseRequest getDatabaseRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDatabaseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDatabaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatabase");

            return clientHandler.execute(new ClientExecutionParams<GetDatabaseRequest, GetDatabaseResponse>()
                    .withOperationName("GetDatabase").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getDatabaseRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDatabaseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the details of an operation by specifying the operation ID.
     * </p>
     *
     * @param getOperationRequest
     * @return Result of the GetOperation operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/GetOperation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetOperationResponse getOperation(GetOperationRequest getOperationRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOperationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOperationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOperation");

            return clientHandler.execute(new ClientExecutionParams<GetOperationRequest, GetOperationResponse>()
                    .withOperationName("GetOperation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getOperationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets permissions associated with the target database.
     * </p>
     *
     * @param getResourcePermissionRequest
     * @return Result of the GetResourcePermission operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.GetResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/GetResourcePermission" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResourcePermissionResponse getResourcePermission(GetResourcePermissionRequest getResourcePermissionRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePermission");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePermissionRequest, GetResourcePermissionResponse>()
                    .withOperationName("GetResourcePermission").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the applications registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                    .withOperationName("ListApplications").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listApplicationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the components registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param listComponentsRequest
     * @return Result of the ListComponents operation returned by the service.
     * @throws UnauthorizedException
     *         The request is not authorized.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/ListComponents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListComponentsResponse listComponents(ListComponentsRequest listComponentsRequest) throws UnauthorizedException,
            ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListComponentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComponentsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponents");

            return clientHandler.execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                    .withOperationName("ListComponents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listComponentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the SAP HANA databases of an application registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param listDatabasesRequest
     * @return Result of the ListDatabases operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/ListDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDatabasesResponse listDatabases(ListDatabasesRequest listDatabasesRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDatabasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatabasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatabases");

            return clientHandler.execute(new ClientExecutionParams<ListDatabasesRequest, ListDatabasesResponse>()
                    .withOperationName("ListDatabases").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDatabasesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatabasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the operations performed by AWS Systems Manager for SAP.
     * </p>
     *
     * @param listOperationsRequest
     * @return Result of the ListOperations operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/ListOperations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListOperationsResponse listOperations(ListOperationsRequest listOperationsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOperationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOperations");

            return clientHandler.execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                    .withOperationName("ListOperations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all tags on an SAP HANA application and/or database registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         A conflict has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds permissions to the target database.
     * </p>
     *
     * @param putResourcePermissionRequest
     * @return Result of the PutResourcePermission operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.PutResourcePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/PutResourcePermission" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutResourcePermissionResponse putResourcePermission(PutResourcePermissionRequest putResourcePermissionRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutResourcePermissionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePermissionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePermission");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePermissionRequest, PutResourcePermissionResponse>()
                    .withOperationName("PutResourcePermission").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putResourcePermissionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePermissionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Register an SAP application with AWS Systems Manager for SAP. You must meet the following requirements before
     * registering.
     * </p>
     * <p>
     * The SAP application you want to register with AWS Systems Manager for SAP is running on Amazon EC2.
     * </p>
     * <p>
     * AWS Systems Manager Agent must be setup on an Amazon EC2 instance along with the required IAM permissions.
     * </p>
     * <p>
     * Amazon EC2 instance(s) must have access to the secrets created in AWS Secrets Manager to manage SAP applications
     * and components.
     * </p>
     *
     * @param registerApplicationRequest
     * @return Result of the RegisterApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         A conflict has occurred.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.RegisterApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/RegisterApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterApplicationResponse registerApplication(RegisterApplicationRequest registerApplicationRequest)
            throws ResourceNotFoundException, ValidationException, ConflictException, InternalServerException,
            AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterApplication");

            return clientHandler.execute(new ClientExecutionParams<RegisterApplicationRequest, RegisterApplicationResponse>()
                    .withOperationName("RegisterApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(registerApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Refreshes a registered application.
     * </p>
     *
     * @param startApplicationRefreshRequest
     * @return Result of the StartApplicationRefresh operation returned by the service.
     * @throws UnauthorizedException
     *         The request is not authorized.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         A conflict has occurred.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.StartApplicationRefresh
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/StartApplicationRefresh"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartApplicationRefreshResponse startApplicationRefresh(StartApplicationRefreshRequest startApplicationRefreshRequest)
            throws UnauthorizedException, ResourceNotFoundException, ValidationException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartApplicationRefreshResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartApplicationRefreshResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startApplicationRefreshRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startApplicationRefreshRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartApplicationRefresh");

            return clientHandler
                    .execute(new ClientExecutionParams<StartApplicationRefreshRequest, StartApplicationRefreshResponse>()
                            .withOperationName("StartApplicationRefresh").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startApplicationRefreshRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartApplicationRefreshRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates tag for a resource by specifying the ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         A conflict has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete the tags for a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         A conflict has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConflictException, AwsServiceException, SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the settings of an application registered with AWS Systems Manager for SAP.
     * </p>
     *
     * @param updateApplicationSettingsRequest
     * @return Result of the UpdateApplicationSettings operation returned by the service.
     * @throws UnauthorizedException
     *         The request is not authorized.
     * @throws ResourceNotFoundException
     *         The resource is not available.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ConflictException
     *         A conflict has occurred.
     * @throws InternalServerException
     *         An internal error has occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws SsmSapException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample SsmSapClient.UpdateApplicationSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ssm-sap-2018-05-10/UpdateApplicationSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateApplicationSettingsResponse updateApplicationSettings(
            UpdateApplicationSettingsRequest updateApplicationSettingsRequest) throws UnauthorizedException,
            ResourceNotFoundException, ValidationException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, SsmSapException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateApplicationSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApplicationSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Ssm Sap");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplicationSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationSettingsRequest, UpdateApplicationSettingsResponse>()
                            .withOperationName("UpdateApplicationSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateApplicationSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateApplicationSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SsmSapServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = SsmSapServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(SsmSapException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final SsmSapServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
