/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Parameters specified in a Systems Manager document that run on the server when the command is run.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentParameter implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentParameter.Builder, DocumentParameter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DocumentParameter::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DocumentParameter::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(DocumentParameter::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> DEFAULT_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultValue").getter(getter(DocumentParameter::defaultValue)).setter(setter(Builder::defaultValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            DESCRIPTION_FIELD, DEFAULT_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String type;

    private final String description;

    private final String defaultValue;

    private DocumentParameter(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.description = builder.description;
        this.defaultValue = builder.defaultValue;
    }

    /**
     * <p>
     * The name of the parameter.
     * </p>
     * 
     * @return The name of the parameter.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of parameter. The type can be either String or StringList.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DocumentParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter. The type can be either String or StringList.
     * @see DocumentParameterType
     */
    public final DocumentParameterType type() {
        return DocumentParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of parameter. The type can be either String or StringList.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DocumentParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of parameter. The type can be either String or StringList.
     * @see DocumentParameterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * A description of what the parameter does, how to use it, the default value, and whether or not the parameter is
     * optional.
     * </p>
     * 
     * @return A description of what the parameter does, how to use it, the default value, and whether or not the
     *         parameter is optional.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * If specified, the default values for the parameters. Parameters without a default value are required. Parameters
     * with a default value are optional.
     * </p>
     * 
     * @return If specified, the default values for the parameters. Parameters without a default value are required.
     *         Parameters with a default value are optional.
     */
    public final String defaultValue() {
        return defaultValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(defaultValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentParameter)) {
            return false;
        }
        DocumentParameter other = (DocumentParameter) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(defaultValue(), other.defaultValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentParameter").add("Name", name()).add("Type", typeAsString())
                .add("Description", description()).add("DefaultValue", defaultValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DefaultValue":
            return Optional.ofNullable(clazz.cast(defaultValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentParameter, T> g) {
        return obj -> g.apply((DocumentParameter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentParameter> {
        /**
         * <p>
         * The name of the parameter.
         * </p>
         * 
         * @param name
         *        The name of the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of parameter. The type can be either String or StringList.
         * </p>
         * 
         * @param type
         *        The type of parameter. The type can be either String or StringList.
         * @see DocumentParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of parameter. The type can be either String or StringList.
         * </p>
         * 
         * @param type
         *        The type of parameter. The type can be either String or StringList.
         * @see DocumentParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentParameterType
         */
        Builder type(DocumentParameterType type);

        /**
         * <p>
         * A description of what the parameter does, how to use it, the default value, and whether or not the parameter
         * is optional.
         * </p>
         * 
         * @param description
         *        A description of what the parameter does, how to use it, the default value, and whether or not the
         *        parameter is optional.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * If specified, the default values for the parameters. Parameters without a default value are required.
         * Parameters with a default value are optional.
         * </p>
         * 
         * @param defaultValue
         *        If specified, the default values for the parameters. Parameters without a default value are required.
         *        Parameters with a default value are optional.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultValue(String defaultValue);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private String description;

        private String defaultValue;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentParameter model) {
            name(model.name);
            type(model.type);
            description(model.description);
            defaultValue(model.defaultValue);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DocumentParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDefaultValue() {
            return defaultValue;
        }

        public final void setDefaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
        }

        @Override
        public final Builder defaultValue(String defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        @Override
        public DocumentParameter build() {
            return new DocumentParameter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
