/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.schemas.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.schemas.SchemasAsyncClient;
import software.amazon.awssdk.services.schemas.model.DescribeCodeBindingRequest;
import software.amazon.awssdk.services.schemas.model.DescribeCodeBindingResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface SchemasAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link SchemasAsyncClient#describeCodeBinding} API until the desired condition {@code CodeBindingExists} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeCodeBindingRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeCodeBindingResponse>> waitUntilCodeBindingExists(
            DescribeCodeBindingRequest describeCodeBindingRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SchemasAsyncClient#describeCodeBinding} API until the desired condition {@code CodeBindingExists} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeCodeBindingRequest#builder()}
     *
     * @param describeCodeBindingRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeCodeBindingResponse>> waitUntilCodeBindingExists(
            Consumer<DescribeCodeBindingRequest.Builder> describeCodeBindingRequest) {
        return waitUntilCodeBindingExists(DescribeCodeBindingRequest.builder().applyMutation(describeCodeBindingRequest).build());
    }

    /**
     * Polls {@link SchemasAsyncClient#describeCodeBinding} API until the desired condition {@code CodeBindingExists} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeCodeBindingRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeCodeBindingResponse>> waitUntilCodeBindingExists(
            DescribeCodeBindingRequest describeCodeBindingRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link SchemasAsyncClient#describeCodeBinding} API until the desired condition {@code CodeBindingExists} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeCodeBindingRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeCodeBindingResponse>> waitUntilCodeBindingExists(
            Consumer<DescribeCodeBindingRequest.Builder> describeCodeBindingRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilCodeBindingExists(DescribeCodeBindingRequest.builder().applyMutation(describeCodeBindingRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link SchemasAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultSchemasAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link SchemasAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link SchemasAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link SchemasAsyncWaiter}
     */
    static SchemasAsyncWaiter create() {
        return DefaultSchemasAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link SchemasAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(SchemasAsyncClient client);

        /**
         * Builds an instance of {@link SchemasAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link SchemasAsyncWaiter}
         */
        SchemasAsyncWaiter build();
    }
}
