/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration for hyperparameter tuning resources for use in training jobs launched by the tuning job. These
 * resources include compute instances and storage volumes. Specify one or more compute instance configurations and
 * allocation strategies to select resources (optional).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HyperParameterTuningInstanceConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<HyperParameterTuningInstanceConfig.Builder, HyperParameterTuningInstanceConfig> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(HyperParameterTuningInstanceConfig::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount").getter(getter(HyperParameterTuningInstanceConfig::instanceCount))
            .setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSizeInGB").getter(getter(HyperParameterTuningInstanceConfig::volumeSizeInGB))
            .setter(setter(Builder::volumeSizeInGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSizeInGB").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            INSTANCE_COUNT_FIELD, VOLUME_SIZE_IN_GB_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Integer instanceCount;

    private final Integer volumeSizeInGB;

    private HyperParameterTuningInstanceConfig(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.instanceCount = builder.instanceCount;
        this.volumeSizeInGB = builder.volumeSizeInGB;
    }

    /**
     * <p>
     * The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose (no GPUs)
     * instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no GPUs) instance types:
     * ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">instance type
     * descriptions</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link TrainingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose
     *         (no GPUs) instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no GPUs)
     *         instance types: ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">instance
     *         type descriptions</a>.
     * @see TrainingInstanceType
     */
    public final TrainingInstanceType instanceType() {
        return TrainingInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose (no GPUs)
     * instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no GPUs) instance types:
     * ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">instance type
     * descriptions</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link TrainingInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose
     *         (no GPUs) instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no GPUs)
     *         instance types: ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">instance
     *         type descriptions</a>.
     * @see TrainingInstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The number of instances of the type specified by <code>InstanceType</code>. Choose an instance count larger than
     * 1 for distributed training algorithms. See <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">Step 2: Launch a SageMaker
     * Distributed Training Job Using the SageMaker Python SDK</a> for more information.
     * </p>
     * 
     * @return The number of instances of the type specified by <code>InstanceType</code>. Choose an instance count
     *         larger than 1 for distributed training algorithms. See <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">Step 2: Launch a
     *         SageMaker Distributed Training Job Using the SageMaker Python SDK</a> for more information.
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * <p>
     * The volume size in GB of the data to be processed for hyperparameter optimization (optional).
     * </p>
     * 
     * @return The volume size in GB of the data to be processed for hyperparameter optimization (optional).
     */
    public final Integer volumeSizeInGB() {
        return volumeSizeInGB;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSizeInGB());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HyperParameterTuningInstanceConfig)) {
            return false;
        }
        HyperParameterTuningInstanceConfig other = (HyperParameterTuningInstanceConfig) obj;
        return Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(instanceCount(), other.instanceCount())
                && Objects.equals(volumeSizeInGB(), other.volumeSizeInGB());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HyperParameterTuningInstanceConfig").add("InstanceType", instanceTypeAsString())
                .add("InstanceCount", instanceCount()).add("VolumeSizeInGB", volumeSizeInGB()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "VolumeSizeInGB":
            return Optional.ofNullable(clazz.cast(volumeSizeInGB()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HyperParameterTuningInstanceConfig, T> g) {
        return obj -> g.apply((HyperParameterTuningInstanceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HyperParameterTuningInstanceConfig> {
        /**
         * <p>
         * The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose (no
         * GPUs) instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no GPUs) instance
         * types: ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">instance type
         * descriptions</a>.
         * </p>
         * 
         * @param instanceType
         *        The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose
         *        (no GPUs) instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no
         *        GPUs) instance types: ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see
         *        <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">
         *        instance type descriptions</a>.
         * @see TrainingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose (no
         * GPUs) instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no GPUs) instance
         * types: ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">instance type
         * descriptions</a>.
         * </p>
         * 
         * @param instanceType
         *        The instance type used for processing of hyperparameter optimization jobs. Choose from general purpose
         *        (no GPUs) instance types: ml.m5.xlarge, ml.m5.2xlarge, and ml.m5.4xlarge or compute optimized (no
         *        GPUs) instance types: ml.c5.xlarge and ml.c5.2xlarge. For more information about instance types, see
         *        <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/notebooks-available-instance-types.html">
         *        instance type descriptions</a>.
         * @see TrainingInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInstanceType
         */
        Builder instanceType(TrainingInstanceType instanceType);

        /**
         * <p>
         * The number of instances of the type specified by <code>InstanceType</code>. Choose an instance count larger
         * than 1 for distributed training algorithms. See <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">Step 2: Launch a SageMaker
         * Distributed Training Job Using the SageMaker Python SDK</a> for more information.
         * </p>
         * 
         * @param instanceCount
         *        The number of instances of the type specified by <code>InstanceType</code>. Choose an instance count
         *        larger than 1 for distributed training algorithms. See <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/data-parallel-use-api.html">Step 2: Launch a
         *        SageMaker Distributed Training Job Using the SageMaker Python SDK</a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * The volume size in GB of the data to be processed for hyperparameter optimization (optional).
         * </p>
         * 
         * @param volumeSizeInGB
         *        The volume size in GB of the data to be processed for hyperparameter optimization (optional).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSizeInGB(Integer volumeSizeInGB);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Integer instanceCount;

        private Integer volumeSizeInGB;

        private BuilderImpl() {
        }

        private BuilderImpl(HyperParameterTuningInstanceConfig model) {
            instanceType(model.instanceType);
            instanceCount(model.instanceCount);
            volumeSizeInGB(model.volumeSizeInGB);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(TrainingInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final Integer getVolumeSizeInGB() {
            return volumeSizeInGB;
        }

        public final void setVolumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
        }

        @Override
        public final Builder volumeSizeInGB(Integer volumeSizeInGB) {
            this.volumeSizeInGB = volumeSizeInGB;
            return this;
        }

        @Override
        public HyperParameterTuningInstanceConfig build() {
            return new HyperParameterTuningInstanceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
