/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a metric to minimize or maximize as the objective of an AutoML job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoMLJobObjective implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoMLJobObjective.Builder, AutoMLJobObjective> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(AutoMLJobObjective::metricNameAsString)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private AutoMLJobObjective(BuilderImpl builder) {
        this.metricName = builder.metricName;
    }

    /**
     * <p>
     * The name of the objective metric used to measure the predictive quality of a machine learning system. During
     * training, the model's parameters are updated iteratively to optimize its performance based on the feedback
     * provided by the objective metric when evaluating the model on the validation dataset.
     * </p>
     * <p>
     * The list of available metrics supported by Autopilot and the default metric applied when you do not specify a
     * metric name explicitly depend on the problem type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For tabular problem types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List of available metrics:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>, <code>RMSE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>, <code>F1</code>,
     * <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>, <code>Recall</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
     * <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>, <code>RecallMacro</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a description of each metric, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
     * >Autopilot metrics for classification and regression</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Default objective metrics:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Regression: <code>MSE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Binary classification: <code>F1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Multiclass classification: <code>Accuracy</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For image or text classification problem types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List of available metrics: <code>Accuracy</code>
     * </p>
     * <p>
     * For a description of each metric, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html">Autopilot
     * metrics for text and image classification</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Default objective metrics: <code>Accuracy</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For time-series forecasting problem types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>, <code>MASE</code>,
     * <code>MAPE</code>, <code>WAPE</code>
     * </p>
     * <p>
     * For a description of each metric, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot metrics for
     * time-series forecasting</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Default objective metrics: <code>AverageWeightedQuantileLoss</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does not require
     * setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without requiring multiple
     * candidates to be trained and evaluated. Instead, using your dataset, Autopilot directly fine-tunes your target
     * model to enhance a default objective metric, the cross-entropy loss. After fine-tuning a language model, you can
     * evaluate the quality of its generated text using different metrics. For a list of the available metrics, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for fine-tuning LLMs
     * in Autopilot</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link AutoMLMetricEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The name of the objective metric used to measure the predictive quality of a machine learning system.
     *         During training, the model's parameters are updated iteratively to optimize its performance based on the
     *         feedback provided by the objective metric when evaluating the model on the validation dataset.</p>
     *         <p>
     *         The list of available metrics supported by Autopilot and the default metric applied when you do not
     *         specify a metric name explicitly depend on the problem type.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For tabular problem types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         List of available metrics:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>,
     *         <code>RMSE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>,
     *         <code>F1</code>, <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>,
     *         <code>Recall</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
     *         <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>,
     *         <code>RecallMacro</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For a description of each metric, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
     *         >Autopilot metrics for classification and regression</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Default objective metrics:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Regression: <code>MSE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Binary classification: <code>F1</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Multiclass classification: <code>Accuracy</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For image or text classification problem types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         List of available metrics: <code>Accuracy</code>
     *         </p>
     *         <p>
     *         For a description of each metric, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html"
     *         >Autopilot metrics for text and image classification</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Default objective metrics: <code>Accuracy</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For time-series forecasting problem types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>,
     *         <code>MASE</code>, <code>MAPE</code>, <code>WAPE</code>
     *         </p>
     *         <p>
     *         For a description of each metric, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot metrics
     *         for time-series forecasting</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Default objective metrics: <code>AverageWeightedQuantileLoss</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does not
     *         require setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without requiring
     *         multiple candidates to be trained and evaluated. Instead, using your dataset, Autopilot directly
     *         fine-tunes your target model to enhance a default objective metric, the cross-entropy loss. After
     *         fine-tuning a language model, you can evaluate the quality of its generated text using different metrics.
     *         For a list of the available metrics, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for
     *         fine-tuning LLMs in Autopilot</a>.
     *         </p>
     *         </li>
     * @see AutoMLMetricEnum
     */
    public final AutoMLMetricEnum metricName() {
        return AutoMLMetricEnum.fromValue(metricName);
    }

    /**
     * <p>
     * The name of the objective metric used to measure the predictive quality of a machine learning system. During
     * training, the model's parameters are updated iteratively to optimize its performance based on the feedback
     * provided by the objective metric when evaluating the model on the validation dataset.
     * </p>
     * <p>
     * The list of available metrics supported by Autopilot and the default metric applied when you do not specify a
     * metric name explicitly depend on the problem type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For tabular problem types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List of available metrics:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>, <code>RMSE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>, <code>F1</code>,
     * <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>, <code>Recall</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
     * <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>, <code>RecallMacro</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a description of each metric, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
     * >Autopilot metrics for classification and regression</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Default objective metrics:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Regression: <code>MSE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Binary classification: <code>F1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Multiclass classification: <code>Accuracy</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For image or text classification problem types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List of available metrics: <code>Accuracy</code>
     * </p>
     * <p>
     * For a description of each metric, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html">Autopilot
     * metrics for text and image classification</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Default objective metrics: <code>Accuracy</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For time-series forecasting problem types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>, <code>MASE</code>,
     * <code>MAPE</code>, <code>WAPE</code>
     * </p>
     * <p>
     * For a description of each metric, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot metrics for
     * time-series forecasting</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Default objective metrics: <code>AverageWeightedQuantileLoss</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does not require
     * setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without requiring multiple
     * candidates to be trained and evaluated. Instead, using your dataset, Autopilot directly fine-tunes your target
     * model to enhance a default objective metric, the cross-entropy loss. After fine-tuning a language model, you can
     * evaluate the quality of its generated text using different metrics. For a list of the available metrics, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for fine-tuning LLMs
     * in Autopilot</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link AutoMLMetricEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The name of the objective metric used to measure the predictive quality of a machine learning system.
     *         During training, the model's parameters are updated iteratively to optimize its performance based on the
     *         feedback provided by the objective metric when evaluating the model on the validation dataset.</p>
     *         <p>
     *         The list of available metrics supported by Autopilot and the default metric applied when you do not
     *         specify a metric name explicitly depend on the problem type.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For tabular problem types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         List of available metrics:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>,
     *         <code>RMSE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>,
     *         <code>F1</code>, <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>,
     *         <code>Recall</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
     *         <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>,
     *         <code>RecallMacro</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For a description of each metric, see <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
     *         >Autopilot metrics for classification and regression</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Default objective metrics:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Regression: <code>MSE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Binary classification: <code>F1</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Multiclass classification: <code>Accuracy</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For image or text classification problem types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         List of available metrics: <code>Accuracy</code>
     *         </p>
     *         <p>
     *         For a description of each metric, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html"
     *         >Autopilot metrics for text and image classification</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Default objective metrics: <code>Accuracy</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For time-series forecasting problem types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>,
     *         <code>MASE</code>, <code>MAPE</code>, <code>WAPE</code>
     *         </p>
     *         <p>
     *         For a description of each metric, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot metrics
     *         for time-series forecasting</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Default objective metrics: <code>AverageWeightedQuantileLoss</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does not
     *         require setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without requiring
     *         multiple candidates to be trained and evaluated. Instead, using your dataset, Autopilot directly
     *         fine-tunes your target model to enhance a default objective metric, the cross-entropy loss. After
     *         fine-tuning a language model, you can evaluate the quality of its generated text using different metrics.
     *         For a list of the available metrics, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for
     *         fine-tuning LLMs in Autopilot</a>.
     *         </p>
     *         </li>
     * @see AutoMLMetricEnum
     */
    public final String metricNameAsString() {
        return metricName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoMLJobObjective)) {
            return false;
        }
        AutoMLJobObjective other = (AutoMLJobObjective) obj;
        return Objects.equals(metricNameAsString(), other.metricNameAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoMLJobObjective").add("MetricName", metricNameAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoMLJobObjective, T> g) {
        return obj -> g.apply((AutoMLJobObjective) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoMLJobObjective> {
        /**
         * <p>
         * The name of the objective metric used to measure the predictive quality of a machine learning system. During
         * training, the model's parameters are updated iteratively to optimize its performance based on the feedback
         * provided by the objective metric when evaluating the model on the validation dataset.
         * </p>
         * <p>
         * The list of available metrics supported by Autopilot and the default metric applied when you do not specify a
         * metric name explicitly depend on the problem type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For tabular problem types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * List of available metrics:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>,
         * <code>RMSE</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>,
         * <code>F1</code>, <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>,
         * <code>Recall</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
         * <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>, <code>RecallMacro</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a description of each metric, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
         * >Autopilot metrics for classification and regression</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Default objective metrics:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Regression: <code>MSE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Binary classification: <code>F1</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Multiclass classification: <code>Accuracy</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For image or text classification problem types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * List of available metrics: <code>Accuracy</code>
         * </p>
         * <p>
         * For a description of each metric, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html"
         * >Autopilot metrics for text and image classification</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Default objective metrics: <code>Accuracy</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For time-series forecasting problem types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>, <code>MASE</code>,
         * <code>MAPE</code>, <code>WAPE</code>
         * </p>
         * <p>
         * For a description of each metric, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot metrics for
         * time-series forecasting</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Default objective metrics: <code>AverageWeightedQuantileLoss</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does not
         * require setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without requiring
         * multiple candidates to be trained and evaluated. Instead, using your dataset, Autopilot directly fine-tunes
         * your target model to enhance a default objective metric, the cross-entropy loss. After fine-tuning a language
         * model, you can evaluate the quality of its generated text using different metrics. For a list of the
         * available metrics, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for fine-tuning
         * LLMs in Autopilot</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The name of the objective metric used to measure the predictive quality of a machine learning system.
         *        During training, the model's parameters are updated iteratively to optimize its performance based on
         *        the feedback provided by the objective metric when evaluating the model on the validation dataset.</p>
         *        <p>
         *        The list of available metrics supported by Autopilot and the default metric applied when you do not
         *        specify a metric name explicitly depend on the problem type.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For tabular problem types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        List of available metrics:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>,
         *        <code>RMSE</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>,
         *        <code>F1</code>, <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>,
         *        <code>Recall</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
         *        <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>,
         *        <code>RecallMacro</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a description of each metric, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
         *        >Autopilot metrics for classification and regression</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Default objective metrics:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Regression: <code>MSE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Binary classification: <code>F1</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Multiclass classification: <code>Accuracy</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For image or text classification problem types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        List of available metrics: <code>Accuracy</code>
         *        </p>
         *        <p>
         *        For a description of each metric, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html"
         *        >Autopilot metrics for text and image classification</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Default objective metrics: <code>Accuracy</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For time-series forecasting problem types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>,
         *        <code>MASE</code>, <code>MAPE</code>, <code>WAPE</code>
         *        </p>
         *        <p>
         *        For a description of each metric, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot
         *        metrics for time-series forecasting</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Default objective metrics: <code>AverageWeightedQuantileLoss</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does
         *        not require setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without
         *        requiring multiple candidates to be trained and evaluated. Instead, using your dataset, Autopilot
         *        directly fine-tunes your target model to enhance a default objective metric, the cross-entropy loss.
         *        After fine-tuning a language model, you can evaluate the quality of its generated text using different
         *        metrics. For a list of the available metrics, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for
         *        fine-tuning LLMs in Autopilot</a>.
         *        </p>
         *        </li>
         * @see AutoMLMetricEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLMetricEnum
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The name of the objective metric used to measure the predictive quality of a machine learning system. During
         * training, the model's parameters are updated iteratively to optimize its performance based on the feedback
         * provided by the objective metric when evaluating the model on the validation dataset.
         * </p>
         * <p>
         * The list of available metrics supported by Autopilot and the default metric applied when you do not specify a
         * metric name explicitly depend on the problem type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For tabular problem types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * List of available metrics:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>,
         * <code>RMSE</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>,
         * <code>F1</code>, <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>,
         * <code>Recall</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
         * <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>, <code>RecallMacro</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For a description of each metric, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
         * >Autopilot metrics for classification and regression</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Default objective metrics:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Regression: <code>MSE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Binary classification: <code>F1</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Multiclass classification: <code>Accuracy</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For image or text classification problem types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * List of available metrics: <code>Accuracy</code>
         * </p>
         * <p>
         * For a description of each metric, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html"
         * >Autopilot metrics for text and image classification</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Default objective metrics: <code>Accuracy</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For time-series forecasting problem types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>, <code>MASE</code>,
         * <code>MAPE</code>, <code>WAPE</code>
         * </p>
         * <p>
         * For a description of each metric, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot metrics for
         * time-series forecasting</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Default objective metrics: <code>AverageWeightedQuantileLoss</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does not
         * require setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without requiring
         * multiple candidates to be trained and evaluated. Instead, using your dataset, Autopilot directly fine-tunes
         * your target model to enhance a default objective metric, the cross-entropy loss. After fine-tuning a language
         * model, you can evaluate the quality of its generated text using different metrics. For a list of the
         * available metrics, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for fine-tuning
         * LLMs in Autopilot</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The name of the objective metric used to measure the predictive quality of a machine learning system.
         *        During training, the model's parameters are updated iteratively to optimize its performance based on
         *        the feedback provided by the objective metric when evaluating the model on the validation dataset.</p>
         *        <p>
         *        The list of available metrics supported by Autopilot and the default metric applied when you do not
         *        specify a metric name explicitly depend on the problem type.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For tabular problem types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        List of available metrics:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Regression: <code>InferenceLatency</code>, <code>MAE</code>, <code>MSE</code>, <code>R2</code>,
         *        <code>RMSE</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Binary classification: <code>Accuracy</code>, <code>AUC</code>, <code>BalancedAccuracy</code>,
         *        <code>F1</code>, <code>InferenceLatency</code>, <code>LogLoss</code>, <code>Precision</code>,
         *        <code>Recall</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Multiclass classification: <code>Accuracy</code>, <code>BalancedAccuracy</code>, <code>F1macro</code>,
         *        <code>InferenceLatency</code>, <code>LogLoss</code>, <code>PrecisionMacro</code>,
         *        <code>RecallMacro</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For a description of each metric, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-metrics-validation.html#autopilot-metrics"
         *        >Autopilot metrics for classification and regression</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Default objective metrics:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Regression: <code>MSE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Binary classification: <code>F1</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Multiclass classification: <code>Accuracy</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For image or text classification problem types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        List of available metrics: <code>Accuracy</code>
         *        </p>
         *        <p>
         *        For a description of each metric, see <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/dg/text-classification-data-format-and-metric.html"
         *        >Autopilot metrics for text and image classification</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Default objective metrics: <code>Accuracy</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For time-series forecasting problem types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        List of available metrics: <code>RMSE</code>, <code>wQL</code>, <code>Average wQL</code>,
         *        <code>MASE</code>, <code>MAPE</code>, <code>WAPE</code>
         *        </p>
         *        <p>
         *        For a description of each metric, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/timeseries-objective-metric.html">Autopilot
         *        metrics for time-series forecasting</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Default objective metrics: <code>AverageWeightedQuantileLoss</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For text generation problem types (LLMs fine-tuning): Fine-tuning language models in Autopilot does
         *        not require setting the <code>AutoMLJobObjective</code> field. Autopilot fine-tunes LLMs without
         *        requiring multiple candidates to be trained and evaluated. Instead, using your dataset, Autopilot
         *        directly fine-tunes your target model to enhance a default objective metric, the cross-entropy loss.
         *        After fine-tuning a language model, you can evaluate the quality of its generated text using different
         *        metrics. For a list of the available metrics, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/llms-finetuning-models.html">Metrics for
         *        fine-tuning LLMs in Autopilot</a>.
         *        </p>
         *        </li>
         * @see AutoMLMetricEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMLMetricEnum
         */
        Builder metricName(AutoMLMetricEnum metricName);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoMLJobObjective model) {
            metricName(model.metricName);
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public final Builder metricName(AutoMLMetricEnum metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        @Override
        public AutoMLJobObjective build() {
            return new AutoMLJobObjective(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
