/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rum;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.rum.model.AccessDeniedException;
import software.amazon.awssdk.services.rum.model.BatchCreateRumMetricDefinitionsRequest;
import software.amazon.awssdk.services.rum.model.BatchCreateRumMetricDefinitionsResponse;
import software.amazon.awssdk.services.rum.model.BatchDeleteRumMetricDefinitionsRequest;
import software.amazon.awssdk.services.rum.model.BatchDeleteRumMetricDefinitionsResponse;
import software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsRequest;
import software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsResponse;
import software.amazon.awssdk.services.rum.model.ConflictException;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.CreateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.DeleteAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.DeleteRumMetricsDestinationRequest;
import software.amazon.awssdk.services.rum.model.DeleteRumMetricsDestinationResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse;
import software.amazon.awssdk.services.rum.model.GetAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.GetAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.InternalServerException;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest;
import software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse;
import software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsRequest;
import software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsResponse;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.rum.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.rum.model.PutRumEventsRequest;
import software.amazon.awssdk.services.rum.model.PutRumEventsResponse;
import software.amazon.awssdk.services.rum.model.PutRumMetricsDestinationRequest;
import software.amazon.awssdk.services.rum.model.PutRumMetricsDestinationResponse;
import software.amazon.awssdk.services.rum.model.ResourceNotFoundException;
import software.amazon.awssdk.services.rum.model.RumException;
import software.amazon.awssdk.services.rum.model.RumRequest;
import software.amazon.awssdk.services.rum.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.rum.model.TagResourceRequest;
import software.amazon.awssdk.services.rum.model.TagResourceResponse;
import software.amazon.awssdk.services.rum.model.ThrottlingException;
import software.amazon.awssdk.services.rum.model.UntagResourceRequest;
import software.amazon.awssdk.services.rum.model.UntagResourceResponse;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorRequest;
import software.amazon.awssdk.services.rum.model.UpdateAppMonitorResponse;
import software.amazon.awssdk.services.rum.model.UpdateRumMetricDefinitionRequest;
import software.amazon.awssdk.services.rum.model.UpdateRumMetricDefinitionResponse;
import software.amazon.awssdk.services.rum.model.ValidationException;
import software.amazon.awssdk.services.rum.paginators.BatchGetRumMetricDefinitionsPublisher;
import software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher;
import software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher;
import software.amazon.awssdk.services.rum.paginators.ListRumMetricsDestinationsPublisher;
import software.amazon.awssdk.services.rum.transform.BatchCreateRumMetricDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.BatchDeleteRumMetricDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.BatchGetRumMetricDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.CreateAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.DeleteAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.DeleteRumMetricsDestinationRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetAppMonitorDataRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.GetAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListAppMonitorsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListRumMetricsDestinationsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.PutRumEventsRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.PutRumMetricsDestinationRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UpdateAppMonitorRequestMarshaller;
import software.amazon.awssdk.services.rum.transform.UpdateRumMetricDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link RumAsyncClient}.
 *
 * @see RumAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRumAsyncClient implements RumAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultRumAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final RumServiceClientConfiguration serviceClientConfiguration;

    protected DefaultRumAsyncClient(RumServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Specifies the extended metrics and custom metrics that you want a CloudWatch RUM app monitor to send to a
     * destination. Valid destinations include CloudWatch and Evidently.
     * </p>
     * <p>
     * By default, RUM app monitors send some metrics to CloudWatch. These default metrics are listed in <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-metrics.html">CloudWatch
     * metrics that you can collect with CloudWatch RUM</a>.
     * </p>
     * <p>
     * In addition to these default metrics, you can choose to send extended metrics or custom metrics or both.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Extended metrics enable you to send metrics with additional dimensions not included in the default metrics. You
     * can also send extended metrics to Evidently as well as CloudWatch. The valid dimension names for the additional
     * dimensions for extended metrics are <code>BrowserName</code>, <code>CountryCode</code>, <code>DeviceType</code>,
     * <code>FileType</code>, <code>OSName</code>, and <code>PageId</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-vended-metrics.html">
     * Extended metrics that you can send to CloudWatch and CloudWatch Evidently</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom metrics are metrics that you define. You can send custom metrics to CloudWatch or to CloudWatch Evidently
     * or to both. With custom metrics, you can use any metric name and namespace, and to derive the metrics you can use
     * any custom events, built-in events, custom attributes, or default attributes.
     * </p>
     * <p>
     * You can't send custom metrics to the <code>AWS/RUM</code> namespace. You must send custom metrics to a custom
     * namespace that you define. The namespace that you use can't start with <code>AWS/</code>. CloudWatch RUM prepends
     * <code>RUM/CustomMetrics/</code> to the custom namespace that you define, so the final namespace for your metrics
     * in CloudWatch is <code>RUM/CustomMetrics/<i>your-custom-namespace</i> </code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The maximum number of metric definitions that you can specify in one <code>BatchCreateRumMetricDefinitions</code>
     * operation is 200.
     * </p>
     * <p>
     * The maximum number of metric definitions that one destination can contain is 2000.
     * </p>
     * <p>
     * Extended metrics sent to CloudWatch and RUM custom metrics are charged as CloudWatch custom metrics. Each
     * combination of additional dimension name and dimension value counts as a custom metric. For more information, see
     * <a href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * You must have already created a destination for the metrics before you send them. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_PutRumMetricsDestination.html">
     * PutRumMetricsDestination</a>.
     * </p>
     * <p>
     * If some metric definitions specified in a <code>BatchCreateRumMetricDefinitions</code> operations are not valid,
     * those metric definitions fail and return errors, but all valid metric definitions in the same operation still
     * succeed.
     * </p>
     *
     * @param batchCreateRumMetricDefinitionsRequest
     * @return A Java Future containing the result of the BatchCreateRumMetricDefinitions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.BatchCreateRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchCreateRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCreateRumMetricDefinitionsResponse> batchCreateRumMetricDefinitions(
            BatchCreateRumMetricDefinitionsRequest batchCreateRumMetricDefinitionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchCreateRumMetricDefinitionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateRumMetricDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCreateRumMetricDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCreateRumMetricDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchCreateRumMetricDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCreateRumMetricDefinitionsRequest, BatchCreateRumMetricDefinitionsResponse>()
                            .withOperationName("BatchCreateRumMetricDefinitions")
                            .withMarshaller(new BatchCreateRumMetricDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchCreateRumMetricDefinitionsRequest));
            CompletableFuture<BatchCreateRumMetricDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified metrics from being sent to an extended metrics destination.
     * </p>
     * <p>
     * If some metric definition IDs specified in a <code>BatchDeleteRumMetricDefinitions</code> operations are not
     * valid, those metric definitions fail and return errors, but all valid metric definition IDs in the same operation
     * are still deleted.
     * </p>
     * <p>
     * The maximum number of metric definitions that you can specify in one <code>BatchDeleteRumMetricDefinitions</code>
     * operation is 200.
     * </p>
     *
     * @param batchDeleteRumMetricDefinitionsRequest
     * @return A Java Future containing the result of the BatchDeleteRumMetricDefinitions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.BatchDeleteRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchDeleteRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteRumMetricDefinitionsResponse> batchDeleteRumMetricDefinitions(
            BatchDeleteRumMetricDefinitionsRequest batchDeleteRumMetricDefinitionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchDeleteRumMetricDefinitionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteRumMetricDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteRumMetricDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteRumMetricDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDeleteRumMetricDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteRumMetricDefinitionsRequest, BatchDeleteRumMetricDefinitionsResponse>()
                            .withOperationName("BatchDeleteRumMetricDefinitions")
                            .withMarshaller(new BatchDeleteRumMetricDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchDeleteRumMetricDefinitionsRequest));
            CompletableFuture<BatchDeleteRumMetricDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of metrics and dimensions that a RUM app monitor is sending to a single destination.
     * </p>
     *
     * @param batchGetRumMetricDefinitionsRequest
     * @return A Java Future containing the result of the BatchGetRumMetricDefinitions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.BatchGetRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchGetRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetRumMetricDefinitionsResponse> batchGetRumMetricDefinitions(
            BatchGetRumMetricDefinitionsRequest batchGetRumMetricDefinitionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetRumMetricDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRumMetricDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetRumMetricDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetRumMetricDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetRumMetricDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetRumMetricDefinitionsRequest, BatchGetRumMetricDefinitionsResponse>()
                            .withOperationName("BatchGetRumMetricDefinitions")
                            .withMarshaller(new BatchGetRumMetricDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetRumMetricDefinitionsRequest));
            CompletableFuture<BatchGetRumMetricDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of metrics and dimensions that a RUM app monitor is sending to a single destination.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #batchGetRumMetricDefinitions(software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.BatchGetRumMetricDefinitionsPublisher publisher = client.batchGetRumMetricDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.BatchGetRumMetricDefinitionsPublisher publisher = client.batchGetRumMetricDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #batchGetRumMetricDefinitions(software.amazon.awssdk.services.rum.model.BatchGetRumMetricDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param batchGetRumMetricDefinitionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.BatchGetRumMetricDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/BatchGetRumMetricDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    public BatchGetRumMetricDefinitionsPublisher batchGetRumMetricDefinitionsPaginator(
            BatchGetRumMetricDefinitionsRequest batchGetRumMetricDefinitionsRequest) {
        return new BatchGetRumMetricDefinitionsPublisher(this, applyPaginatorUserAgent(batchGetRumMetricDefinitionsRequest));
    }

    /**
     * <p>
     * Creates a Amazon CloudWatch RUM app monitor, which collects telemetry data from your application and sends that
     * data to RUM. The data includes performance and reliability information such as page load time, client-side
     * errors, and user behavior.
     * </p>
     * <p>
     * You use this operation only to create a new app monitor. To update an existing app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_UpdateAppMonitor.html"
     * >UpdateAppMonitor</a> instead.
     * </p>
     * <p>
     * After you create an app monitor, sign in to the CloudWatch RUM console to get the JavaScript code snippet to add
     * to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param createAppMonitorRequest
     * @return A Java Future containing the result of the CreateAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.CreateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/CreateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppMonitorResponse> createAppMonitor(CreateAppMonitorRequest createAppMonitorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppMonitorRequest, CreateAppMonitorResponse>()
                            .withOperationName("CreateAppMonitor")
                            .withMarshaller(new CreateAppMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAppMonitorRequest));
            CompletableFuture<CreateAppMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing app monitor. This immediately stops the collection of data.
     * </p>
     *
     * @param deleteAppMonitorRequest
     * @return A Java Future containing the result of the DeleteAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.DeleteAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppMonitorResponse> deleteAppMonitor(DeleteAppMonitorRequest deleteAppMonitorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppMonitorRequest, DeleteAppMonitorResponse>()
                            .withOperationName("DeleteAppMonitor")
                            .withMarshaller(new DeleteAppMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppMonitorRequest));
            CompletableFuture<DeleteAppMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a destination for CloudWatch RUM extended metrics, so that the specified app monitor stops sending
     * extended metrics to that destination.
     * </p>
     *
     * @param deleteRumMetricsDestinationRequest
     * @return A Java Future containing the result of the DeleteRumMetricsDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.DeleteRumMetricsDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/DeleteRumMetricsDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRumMetricsDestinationResponse> deleteRumMetricsDestination(
            DeleteRumMetricsDestinationRequest deleteRumMetricsDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRumMetricsDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRumMetricsDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRumMetricsDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRumMetricsDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRumMetricsDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRumMetricsDestinationRequest, DeleteRumMetricsDestinationResponse>()
                            .withOperationName("DeleteRumMetricsDestination")
                            .withMarshaller(new DeleteRumMetricsDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRumMetricsDestinationRequest));
            CompletableFuture<DeleteRumMetricsDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the complete configuration information for one app monitor.
     * </p>
     *
     * @param getAppMonitorRequest
     * @return A Java Future containing the result of the GetAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppMonitorResponse> getAppMonitor(GetAppMonitorRequest getAppMonitorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppMonitorRequest, GetAppMonitorResponse>()
                            .withOperationName("GetAppMonitor")
                            .withMarshaller(new GetAppMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAppMonitorRequest));
            CompletableFuture<GetAppMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return A Java Future containing the result of the GetAppMonitorData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppMonitorDataResponse> getAppMonitorData(GetAppMonitorDataRequest getAppMonitorDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppMonitorDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppMonitorData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppMonitorDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAppMonitorDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppMonitorDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppMonitorDataRequest, GetAppMonitorDataResponse>()
                            .withOperationName("GetAppMonitorData")
                            .withMarshaller(new GetAppMonitorDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAppMonitorDataRequest));
            CompletableFuture<GetAppMonitorDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the raw performance events that RUM has collected from your web application, so that you can do your
     * own processing or analysis of this data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher publisher = client.getAppMonitorDataPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.GetAppMonitorDataPublisher publisher = client.getAppMonitorDataPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.GetAppMonitorDataResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getAppMonitorData(software.amazon.awssdk.services.rum.model.GetAppMonitorDataRequest)} operation.</b>
     * </p>
     *
     * @param getAppMonitorDataRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.GetAppMonitorData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/GetAppMonitorData" target="_top">AWS API
     *      Documentation</a>
     */
    public GetAppMonitorDataPublisher getAppMonitorDataPaginator(GetAppMonitorDataRequest getAppMonitorDataRequest) {
        return new GetAppMonitorDataPublisher(this, applyPaginatorUserAgent(getAppMonitorDataRequest));
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return A Java Future containing the result of the ListAppMonitors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppMonitorsResponse> listAppMonitors(ListAppMonitorsRequest listAppMonitorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppMonitorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppMonitors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppMonitorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppMonitorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppMonitorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppMonitorsRequest, ListAppMonitorsResponse>()
                            .withOperationName("ListAppMonitors")
                            .withMarshaller(new ListAppMonitorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppMonitorsRequest));
            CompletableFuture<ListAppMonitorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the Amazon CloudWatch RUM app monitors in the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher publisher = client.listAppMonitorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListAppMonitorsPublisher publisher = client.listAppMonitorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.ListAppMonitorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppMonitors(software.amazon.awssdk.services.rum.model.ListAppMonitorsRequest)} operation.</b>
     * </p>
     *
     * @param listAppMonitorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListAppMonitors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListAppMonitors" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAppMonitorsPublisher listAppMonitorsPaginator(ListAppMonitorsRequest listAppMonitorsRequest) {
        return new ListAppMonitorsPublisher(this, applyPaginatorUserAgent(listAppMonitorsRequest));
    }

    /**
     * <p>
     * Returns a list of destinations that you have created to receive RUM extended metrics, for the specified app
     * monitor.
     * </p>
     * <p>
     * For more information about extended metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_AddRumMetrcs.html">AddRumMetrics</a>.
     * </p>
     *
     * @param listRumMetricsDestinationsRequest
     * @return A Java Future containing the result of the ListRumMetricsDestinations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListRumMetricsDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListRumMetricsDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRumMetricsDestinationsResponse> listRumMetricsDestinations(
            ListRumMetricsDestinationsRequest listRumMetricsDestinationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRumMetricsDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRumMetricsDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRumMetricsDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRumMetricsDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRumMetricsDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRumMetricsDestinationsRequest, ListRumMetricsDestinationsResponse>()
                            .withOperationName("ListRumMetricsDestinations")
                            .withMarshaller(new ListRumMetricsDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRumMetricsDestinationsRequest));
            CompletableFuture<ListRumMetricsDestinationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of destinations that you have created to receive RUM extended metrics, for the specified app
     * monitor.
     * </p>
     * <p>
     * For more information about extended metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_AddRumMetrcs.html">AddRumMetrics</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRumMetricsDestinations(software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListRumMetricsDestinationsPublisher publisher = client.listRumMetricsDestinationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.rum.paginators.ListRumMetricsDestinationsPublisher publisher = client.listRumMetricsDestinationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRumMetricsDestinations(software.amazon.awssdk.services.rum.model.ListRumMetricsDestinationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRumMetricsDestinationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListRumMetricsDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListRumMetricsDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListRumMetricsDestinationsPublisher listRumMetricsDestinationsPaginator(
            ListRumMetricsDestinationsRequest listRumMetricsDestinationsRequest) {
        return new ListRumMetricsDestinationsPublisher(this, applyPaginatorUserAgent(listRumMetricsDestinationsRequest));
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch RUM resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends telemetry events about your application performance and user behavior to CloudWatch RUM. The code snippet
     * that RUM generates for you to add to your application includes <code>PutRumEvents</code> operations to send this
     * data to RUM.
     * </p>
     * <p>
     * Each <code>PutRumEvents</code> operation can send a batch of events from one user session.
     * </p>
     *
     * @param putRumEventsRequest
     * @return A Java Future containing the result of the PutRumEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.PutRumEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRumEventsResponse> putRumEvents(PutRumEventsRequest putRumEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRumEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRumEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRumEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutRumEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "dataplane.";
            String resolvedHostExpression = "dataplane.";

            CompletableFuture<PutRumEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRumEventsRequest, PutRumEventsResponse>()
                            .withOperationName("PutRumEvents").withMarshaller(new PutRumEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(putRumEventsRequest));
            CompletableFuture<PutRumEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a destination to receive extended metrics from CloudWatch RUM. You can send extended metrics
     * to CloudWatch or to a CloudWatch Evidently experiment.
     * </p>
     * <p>
     * For more information about extended metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchCreateRumMetricDefinitions.html"
     * >BatchCreateRumMetricDefinitions</a>.
     * </p>
     *
     * @param putRumMetricsDestinationRequest
     * @return A Java Future containing the result of the PutRumMetricsDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.PutRumMetricsDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/PutRumMetricsDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRumMetricsDestinationResponse> putRumMetricsDestination(
            PutRumMetricsDestinationRequest putRumMetricsDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRumMetricsDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRumMetricsDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRumMetricsDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRumMetricsDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRumMetricsDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRumMetricsDestinationRequest, PutRumMetricsDestinationResponse>()
                            .withOperationName("PutRumMetricsDestination")
                            .withMarshaller(new PutRumMetricsDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putRumMetricsDestinationRequest));
            CompletableFuture<PutRumMetricsDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch RUM resource. Currently, the only
     * resources that can be tagged app monitors.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key for the resource, this tag is appended to the list of tags associated with the alarm. If you specify a tag
     * key that is already associated with the resource, the new tag value that you specify replaces the previous value
     * for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon
     * Web Services resources</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing app monitor. When you use this operation, only the parts of the app
     * monitor configuration that you specify in this operation are changed. For any parameters that you omit, the
     * existing values are kept.
     * </p>
     * <p>
     * You can't use this operation to change the tags of an existing app monitor. To change the tags of an existing app
     * monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_TagResource.html">TagResource</a>.
     * </p>
     * <p>
     * To create a new app monitor, use <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_CreateAppMonitor.html"
     * >CreateAppMonitor</a>.
     * </p>
     * <p>
     * After you update an app monitor, sign in to the CloudWatch RUM console to get the updated JavaScript code snippet
     * to add to your web application. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-RUM-find-code-snippet.html">How
     * do I find a code snippet that I've already generated?</a>
     * </p>
     *
     * @param updateAppMonitorRequest
     * @return A Java Future containing the result of the UpdateAppMonitor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UpdateAppMonitor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateAppMonitor" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppMonitorResponse> updateAppMonitor(UpdateAppMonitorRequest updateAppMonitorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppMonitorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppMonitor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppMonitorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppMonitorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppMonitorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppMonitorRequest, UpdateAppMonitorResponse>()
                            .withOperationName("UpdateAppMonitor")
                            .withMarshaller(new UpdateAppMonitorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAppMonitorRequest));
            CompletableFuture<UpdateAppMonitorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies one existing metric definition for CloudWatch RUM extended metrics. For more information about extended
     * metrics, see <a
     * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_BatchCreateRumMetricsDefinitions.html"
     * >BatchCreateRumMetricsDefinitions</a>.
     * </p>
     *
     * @param updateRumMetricDefinitionRequest
     * @return A Java Future containing the result of the UpdateRumMetricDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>InternalServerException Internal service exception.</li>
     *         <li>ValidationException One of the arguments for the request is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>RumException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample RumAsyncClient.UpdateRumMetricDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/rum-2018-05-10/UpdateRumMetricDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRumMetricDefinitionResponse> updateRumMetricDefinition(
            UpdateRumMetricDefinitionRequest updateRumMetricDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRumMetricDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "RUM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRumMetricDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRumMetricDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRumMetricDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRumMetricDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRumMetricDefinitionRequest, UpdateRumMetricDefinitionResponse>()
                            .withOperationName("UpdateRumMetricDefinition")
                            .withMarshaller(new UpdateRumMetricDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRumMetricDefinitionRequest));
            CompletableFuture<UpdateRumMetricDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final RumServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(RumException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends RumRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
