/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoverycontrolconfig.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.route53recoverycontrolconfig.Route53RecoveryControlConfigClient;
import software.amazon.awssdk.services.route53recoverycontrolconfig.internal.UserAgentUtils;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest;
import software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.route53recoverycontrolconfig.Route53RecoveryControlConfigClient#listAssociatedRoute53HealthChecksPaginator(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksResponse}
 * that can be used to iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable responses = client.listAssociatedRoute53HealthChecksPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable responses = client
 *             .listAssociatedRoute53HealthChecksPaginator(request);
 *     for (software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.route53recoverycontrolconfig.paginators.ListAssociatedRoute53HealthChecksIterable responses = client.listAssociatedRoute53HealthChecksPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listAssociatedRoute53HealthChecks(software.amazon.awssdk.services.route53recoverycontrolconfig.model.ListAssociatedRoute53HealthChecksRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListAssociatedRoute53HealthChecksIterable implements SdkIterable<ListAssociatedRoute53HealthChecksResponse> {
    private final Route53RecoveryControlConfigClient client;

    private final ListAssociatedRoute53HealthChecksRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListAssociatedRoute53HealthChecksIterable(Route53RecoveryControlConfigClient client,
            ListAssociatedRoute53HealthChecksRequest firstRequest) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.nextPageFetcher = new ListAssociatedRoute53HealthChecksResponseFetcher();
    }

    @Override
    public Iterator<ListAssociatedRoute53HealthChecksResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated
     * {@link ListAssociatedRoute53HealthChecksResponse#healthCheckIds()} member. The returned iterable is used to
     * iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<String> healthCheckIds() {
        Function<ListAssociatedRoute53HealthChecksResponse, Iterator<String>> getIterator = response -> {
            if (response != null && response.healthCheckIds() != null) {
                return response.healthCheckIds().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.<ListAssociatedRoute53HealthChecksResponse, String> builder().pagesIterable(this)
                .itemIteratorFunction(getIterator).build();
    }

    private class ListAssociatedRoute53HealthChecksResponseFetcher implements
            SyncPageFetcher<ListAssociatedRoute53HealthChecksResponse> {
        @Override
        public boolean hasNextPage(ListAssociatedRoute53HealthChecksResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListAssociatedRoute53HealthChecksResponse nextPage(ListAssociatedRoute53HealthChecksResponse previousPage) {
            if (previousPage == null) {
                return client.listAssociatedRoute53HealthChecks(firstRequest);
            }
            return client.listAssociatedRoute53HealthChecks(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
